import { DeepPartial } from "@zwave-js/shared";
import type { Format, TransformableInfo } from "logform";
import winston, { Logger } from "winston";
import type Transport from "winston-transport";
export declare const timestampFormat = "HH:mm:ss.SSS";
export declare type DataDirection = "inbound" | "outbound" | "none";
export declare function getDirectionPrefix(direction: DataDirection): "« " | "» " | "  ";
/** The space the directional arrows, grouping brackets and padding occupies */
export declare const CONTROL_CHAR_WIDTH = 2;
export declare const directionPrefixPadding: string;
/**
 * The width of a log line in (visible) characters, excluding the timestamp and
 * label, but including the direction prefix
 */
export declare const LOG_WIDTH = 80;
/** The width of the columns containing the timestamp and channel */
export declare const LOG_PREFIX_WIDTH = 20;
export interface ZWaveLogInfo extends Omit<TransformableInfo, "message"> {
    direction: string;
    /** Primary tags are printed before the message and must fit into the first line.
     * They don't have to be enclosed in square brackets */
    primaryTags?: string;
    /** Secondary tags are right-aligned in the first line and printed in a dim color */
    secondaryTags?: string;
    secondaryTagPadding?: number;
    multiline?: boolean;
    timestamp?: string;
    label?: string;
    message: string | string[];
}
export declare type MessageRecord = Record<string, string | number | boolean | null | undefined>;
export interface MessageOrCCLogEntry {
    tags: string[];
    message?: MessageRecord;
}
/** Returns the tag used to log node related messages */
export declare function getNodeTag(nodeId: number): string;
export declare type ZWaveLogger = Omit<Logger, "log"> & {
    log: (info: ZWaveLogInfo) => void;
};
export declare class ZWaveLoggerBase {
    constructor(loggers: ZWaveLogContainer, logLabel: string);
    logger: ZWaveLogger;
    container: ZWaveLogContainer;
}
export interface LogConfig {
    enabled: boolean;
    level: number;
    transports: Transport[];
    logToFile: boolean;
    filename: string;
    forceConsole: boolean;
}
export declare class ZWaveLogContainer extends winston.Container {
    private fileTransport;
    private consoleTransport;
    private loglevelVisibleCache;
    private logConfig;
    constructor(config?: DeepPartial<LogConfig>);
    getLogger(label: string): ZWaveLogger;
    updateConfiguration(config: DeepPartial<LogConfig>): void;
    getConfiguredTransports(): Transport[];
    /** The common logger format for all channels */
    createLoggerFormat(channel: string): Format;
    /** Prints a formatted and colorized log message */
    logMessagePrinter: Format;
    /** Formats the log message and calculates the necessary paddings */
    logMessageFormatter: Format;
    /** Tests whether a log using the given loglevel will be logged */
    isLoglevelVisible(loglevel: string): boolean;
    destroy(): void;
    private createLogTransports;
    private createConsoleTransport;
    private isConsoleTransportSilent;
    private isFileTransportSilent;
    private createFileTransport;
}
/**
 * Checks the LOG_NODES env variable whether logs should be written for a given node id
 */
export declare function shouldLogNode(nodeId: number): boolean;
/**
 * Tests if a given message fits into a single log line
 * @param info The message that should be logged
 * @param messageLength The length that should be assumed for the actual message without pre and postfixes.
 * Can be set to 0 to exclude the message from the calculation
 */
export declare function messageFitsIntoOneLine(info: ZWaveLogInfo, messageLength: number): boolean;
export declare function messageToLines(message: string | string[]): string[];
/** Splits a message record into multiple lines and auto-aligns key-value pairs */
export declare function messageRecordToLines(message: MessageRecord): string[];
/** Wraps an array of strings in square brackets and joins them with spaces */
export declare function tagify(tags: string[]): string;
/** Unsilences the console transport of a logger and returns the original value */
export declare function unsilence(logger: winston.Logger): boolean;
/** Restores the console transport of a logger to its original silence state */
export declare function restoreSilence(logger: winston.Logger, original: boolean): void;
//# sourceMappingURL=shared.d.ts.map