"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAPI = exports.API = exports.getCCValueMetadata = exports.ccValueMetadata = exports.ccKeyValuePair = exports.ccValue = exports.getCCResponsePredicate = exports.getExpectedCCResponse = exports.expectedCCResponse = exports.CCCommand = exports.getImplementedVersionStatic = exports.getImplementedVersion = exports.implementedVersion = exports.getCCConstructor = exports.getCommandClassStatic = exports.getCommandClass = exports.commandClass = exports.CommandClass = exports.gotDeserializationOptions = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const typeguards_1 = require("alcalzone-shared/typeguards");
const Types_1 = require("../node/Types");
const API_1 = require("./API");
const EncapsulatingCommandClass_1 = require("./EncapsulatingCommandClass");
function gotDeserializationOptions(options) {
    return "data" in options && Buffer.isBuffer(options.data);
}
exports.gotDeserializationOptions = gotDeserializationOptions;
function gotCCCommandOptions(options) {
    return typeof options.nodeId === "number" || typeguards_1.isArray(options.nodeId);
}
// @publicAPI
class CommandClass {
    // empty constructor to parse messages
    constructor(driver, options) {
        var _a, _b, _c, _d;
        /**
         * @internal
         * Whether the command should be sent encrypted
         * This only has an effect if the target node supports Security.
         */
        this.secure = false;
        /** Which variables should be persisted when requested */
        this._registeredCCValues = new Map();
        this.driver = driver;
        // Extract the cc from declared metadata if not provided by the CC constructor
        this.ccId =
            ("ccId" in options && options.ccId) || getCommandClass(this);
        // Default to the root endpoint - Inherited classes may override this behavior
        this.endpointIndex = (_a = ("endpoint" in options ? options.endpoint : undefined)) !== null && _a !== void 0 ? _a : 0;
        // Default to non-supervised commands
        this.supervised = (_b = ("supervised" in options ? options.supervised : undefined)) !== null && _b !== void 0 ? _b : false;
        // We cannot use @ccValue for non-derived classes, so register interviewComplete as an internal value here
        this.registerValue("interviewComplete", true);
        if (gotDeserializationOptions(options)) {
            // For deserialized commands, try to invoke the correct subclass constructor
            const ccCommand = CommandClass.getCCCommand(options.data);
            if (ccCommand != undefined) {
                const CommandConstructor = getCCCommandConstructor(this.ccId, ccCommand);
                if (CommandConstructor &&
                    (new.target) !== CommandConstructor) {
                    return new CommandConstructor(driver, options);
                }
            }
            // If the constructor is correct or none was found, fall back to normal deserialization
            if (options.fromEncapsulation) {
                // Propagate the node ID and endpoint index from the encapsulating CC
                this.nodeId = options.encapCC.nodeId;
                if (!this.endpointIndex && options.encapCC.endpointIndex) {
                    this.endpointIndex = options.encapCC.endpointIndex;
                }
                // And remember which CC encapsulates this CC
                this.encapsulatingCC = options.encapCC;
            }
            else {
                this.nodeId = options.nodeId;
            }
            ({
                ccId: this.ccId,
                ccCommand: this.ccCommand,
                payload: this.payload,
            } = this.deserialize(options.data));
        }
        else if (gotCCCommandOptions(options)) {
            const { nodeId, ccCommand = getCCCommand(this), payload = Buffer.allocUnsafe(0), } = options;
            this.nodeId = nodeId;
            this.ccCommand = ccCommand;
            this.payload = payload;
        }
        if (this.isSinglecast() && this.nodeId !== core_1.NODE_ID_BROADCAST) {
            // For singlecast CCs, set the CC version as high as possible
            this.version = this.driver.getSafeCCVersionForNode(this.ccId, this.nodeId, this.endpointIndex);
            // If we received a CC from a node, it must support at least version 1
            // Make sure that the interview is complete or we cannot be sure that the assumption is correct
            const node = this.getNodeUnsafe();
            if ((node === null || node === void 0 ? void 0 : node.interviewStage) === Types_1.InterviewStage.Complete &&
                this.version === 0 &&
                gotDeserializationOptions(options)) {
                this.version = 1;
            }
            // If the node or endpoint is included securely, send secure commands if possible
            const endpoint = (_c = this.getNodeUnsafe()) === null || _c === void 0 ? void 0 : _c.getEndpoint(this.endpointIndex);
            this.secure =
                (node === null || node === void 0 ? void 0 : node.isSecure) !== false &&
                    !!((_d = (endpoint !== null && endpoint !== void 0 ? endpoint : node)) === null || _d === void 0 ? void 0 : _d.isCCSecure(this.ccId)) &&
                    !!this.driver.securityManager;
        }
        else {
            // For multicast and broadcast CCs, we just use the highest implemented version to serialize
            // Older nodes will ignore the additional fields
            this.version = getImplementedVersion(this.ccId);
            // Security does not support multicast
        }
    }
    /** Returns true if this CC is an extended CC (0xF100..0xFFFF) */
    isExtended() {
        return this.ccId >= 0xf100;
    }
    /** Whether the interview for this CC was previously completed */
    get interviewComplete() {
        return !!this.getValueDB().getValue({
            commandClass: this.ccId,
            endpoint: this.endpointIndex,
            property: "interviewComplete",
        });
    }
    set interviewComplete(value) {
        this.getValueDB().setValue({
            commandClass: this.ccId,
            endpoint: this.endpointIndex,
            property: "interviewComplete",
        }, value);
    }
    /** Can be used by endpoints to test if the root device was already interviewed */
    get rootDeviceInterviewComplete() {
        return !!this.getValueDB().getValue({
            commandClass: this.ccId,
            endpoint: 0,
            property: "interviewComplete",
        });
    }
    /**
     * Deserializes a CC from a buffer that contains a serialized CC
     */
    deserialize(data) {
        const ccId = CommandClass.getCommandClass(data);
        const ccIdLength = this.isExtended() ? 2 : 1;
        if (data.length > ccIdLength) {
            // This is not a NoOp CC (contains command and payload)
            const ccCommand = data[ccIdLength];
            const payload = data.slice(ccIdLength + 1);
            return {
                ccId,
                ccCommand,
                payload,
            };
        }
        else {
            // NoOp CC (no command, no payload)
            const payload = Buffer.allocUnsafe(0);
            return { ccId, payload };
        }
    }
    /**
     * Serializes this CommandClass to be embedded in a message payload or another CC
     */
    serialize() {
        // NoOp CCs have no command and no payload
        if (this.ccId === core_1.CommandClasses["No Operation"])
            return Buffer.from([this.ccId]);
        else if (this.ccCommand == undefined) {
            throw new core_1.ZWaveError("Cannot serialize a Command Class without a command", core_1.ZWaveErrorCodes.CC_Invalid);
        }
        const payloadLength = this.payload.length;
        const ccIdLength = this.isExtended() ? 2 : 1;
        const data = Buffer.allocUnsafe(ccIdLength + 1 + payloadLength);
        data.writeUIntBE(this.ccId, 0, ccIdLength);
        data[ccIdLength] = this.ccCommand;
        if (payloadLength > 0 /* implies payload != undefined */) {
            this.payload.copy(data, 1 + ccIdLength);
        }
        return data;
    }
    /** Extracts the CC id from a buffer that contains a serialized CC */
    static getCommandClass(data) {
        return core_1.parseCCId(data).ccId;
    }
    /** Extracts the CC command from a buffer that contains a serialized CC  */
    static getCCCommand(data) {
        if (data[0] === 0)
            return undefined; // NoOp
        const isExtendedCC = data[0] >= 0xf1;
        return isExtendedCC ? data[2] : data[1];
    }
    /**
     * Retrieves the correct constructor for the CommandClass in the given Buffer.
     * It is assumed that the buffer only contains the serialized CC. This throws if the CC is not implemented.
     */
    static getConstructor(ccData) {
        // Encapsulated CCs don't have the two header bytes
        const cc = CommandClass.getCommandClass(ccData);
        const ret = getCCConstructor(cc);
        if (!ret) {
            const ccName = core_1.getCCName(cc);
            throw new core_1.ZWaveError(`The command class ${ccName} is not implemented`, core_1.ZWaveErrorCodes.CC_NotImplemented);
        }
        return ret;
    }
    /**
     * Creates an instance of the CC that is serialized in the given buffer
     */
    static from(driver, options) {
        // Fall back to unspecified command class in case we receive one that is not implemented
        const Constructor = CommandClass.getConstructor(options.data);
        const ret = new Constructor(driver, options);
        return ret;
    }
    /** Generates a representation of this CC for the log */
    toLogEntry() {
        let tag = this.constructor.name;
        const message = {};
        if (this.constructor === CommandClass) {
            tag = `${shared_1.getEnumMemberName(core_1.CommandClasses, this.ccId)} CC (not implemented)`;
            if (this.ccCommand != undefined) {
                message.command = shared_1.num2hex(this.ccCommand);
            }
        }
        if (this.payload.length > 0) {
            message.payload = shared_1.buffer2hex(this.payload);
        }
        return {
            tags: [tag],
            message,
        };
    }
    /** Generates the JSON representation of this CC */
    toJSON() {
        return this.toJSONInternal();
    }
    toJSONInternal() {
        const ret = {
            nodeId: this.nodeId,
            ccId: core_1.CommandClasses[this.ccId] || shared_1.num2hex(this.ccId),
        };
        if (this.ccCommand != undefined) {
            ret.ccCommand = shared_1.num2hex(this.ccCommand);
        }
        if (this.payload.length > 0) {
            ret.payload = "0x" + this.payload.toString("hex");
        }
        return ret;
    }
    toJSONInherited(props) {
        const { payload, ...ret } = this.toJSONInternal();
        return core_1.stripUndefined({ ...ret, ...props });
    }
    // This needs to be overwritten per command class. In the default implementation, don't do anything
    /**
     * Performs the interview procedure for this CC according to SDS14223
     * @param complete Whether a complete interview should be performed or only the necessary steps on startup
     */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async interview(complete = true) {
        // Empty on purpose
    }
    /** Determines which CC interviews must be performed before this CC can be interviewed */
    determineRequiredCCInterviews() {
        // By default, all CCs require the VersionCC interview
        // There are two exceptions to this rule:
        // * ManufacturerSpecific must be interviewed first
        // * VersionCC itself must be done after that
        // These exceptions are defined in the overrides of this method of each corresponding CC
        return [core_1.CommandClasses.Version];
    }
    /**
     * Whether the endpoint interview may be skipped by a CC. Can be overwritten by a subclass.
     */
    skipEndpointInterview() {
        // By default no interview may be skipped
        return false;
    }
    /**
     * Maps a BasicCC value to a more specific CC implementation. Returns true if the value was mapped, false otherwise.
     * @param value The value of the received BasicCC
     */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    setMappedBasicValue(value) {
        // By default, don't map
        return false;
    }
    isSinglecast() {
        return typeof this.nodeId === "number";
    }
    isMulticast() {
        return typeguards_1.isArray(this.nodeId);
    }
    /**
     * Returns the node this CC is linked to. Throws if the controller is not yet ready.
     */
    getNode() {
        if (this.isSinglecast()) {
            return this.driver.controller.nodes.get(this.nodeId);
        }
    }
    /**
     * @internal
     * Returns the node this CC is linked to (or undefined if the node doesn't exist)
     */
    getNodeUnsafe() {
        try {
            return this.getNode();
        }
        catch (e) {
            // This was expected
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Driver_NotReady) {
                return undefined;
            }
            // Something else happened
            throw e;
        }
    }
    getEndpoint() {
        var _a;
        return (_a = this.getNode()) === null || _a === void 0 ? void 0 : _a.getEndpoint(this.endpointIndex);
    }
    /** Returns the value DB for this CC's node */
    getValueDB() {
        const node = this.getNode();
        if (node == undefined) {
            throw new core_1.ZWaveError("The node for this CC does not exist or the driver is not ready yet", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return node.valueDB;
    }
    /**
     * Creates a value that will be stored in the valueDB alongside with the ones marked with `@ccValue()`
     * @param property The property the value belongs to
     * @param internal Whether the value should be exposed to library users
     */
    registerValue(property, internal = false) {
        this._registeredCCValues.set(property, internal);
    }
    /** Returns a list of all value names that are defined for this CommandClass */
    getDefinedValueIDs() {
        // In order to compare value ids, we need them to be strings
        const ret = new Map();
        const addValueId = (property, propertyKey) => {
            const valueId = {
                commandClass: this.ccId,
                endpoint: this.endpointIndex,
                property,
                propertyKey,
            };
            const dbKey = core_1.valueIdToString(valueId);
            if (!ret.has(dbKey))
                ret.set(dbKey, valueId);
        };
        // Return all manually registered CC values that are not internal
        const registeredCCValueNames = [...this._registeredCCValues]
            .filter(([, isInternal]) => !isInternal)
            .map(([key]) => key);
        registeredCCValueNames.forEach((property) => addValueId(property));
        // Return all defined non-internal CC values that are available in the current version of this CC
        const valueDefinitions = getCCValueDefinitions(this);
        const definedCCValueNames = [...valueDefinitions]
            .filter(([, options]) => options.internal !== true &&
            (options.minVersion == undefined ||
                options.minVersion <= this.version))
            .map(([key]) => key);
        definedCCValueNames.forEach((property) => addValueId(property));
        const kvpDefinitions = getCCKeyValuePairDefinitions(this);
        // Also return all existing value ids that are not internal (values AND metadata without values!)
        const existingValueIds = [
            ...this.getValueDB().getValues(this.ccId),
            ...this.getValueDB().getAllMetadata(this.ccId),
        ]
            .filter((valueId) => valueId.endpoint === this.endpointIndex)
            // allow the value id if it is NOT registered or it is registered as non-internal
            .filter((valueId) => !this._registeredCCValues.has(valueId.property) ||
            this._registeredCCValues.get(valueId.property) === false)
            // allow the value id if it is NOT defined or it is defined as non-internal
            .filter((valueId) => !valueDefinitions.has(valueId.property) ||
            valueDefinitions.get(valueId.property) === false)
            .filter((valueId) => !kvpDefinitions.has(valueId.property) ||
            kvpDefinitions.get(valueId.property) === false);
        existingValueIds.forEach(({ property, propertyKey }) => addValueId(property, propertyKey));
        return [...ret.values()];
    }
    /** Determines if the given value is an internal value */
    isInternalValue(property) {
        // A value is internal if any of the possible definitions say so (true)
        if (this._registeredCCValues.get(property) === true)
            return true;
        const ccValueDefinition = getCCValueDefinitions(this).get(property);
        if ((ccValueDefinition === null || ccValueDefinition === void 0 ? void 0 : ccValueDefinition.internal) === true)
            return true;
        const ccKeyValuePairDefinition = getCCKeyValuePairDefinitions(this).get(property);
        if ((ccKeyValuePairDefinition === null || ccKeyValuePairDefinition === void 0 ? void 0 : ccKeyValuePairDefinition.internal) === true)
            return true;
        return false;
    }
    /** Determines if the given value should always be persisted */
    shouldValueAlwaysBeCreated(property) {
        const ccValueDefinition = getCCValueDefinitions(this).get(property);
        if ((ccValueDefinition === null || ccValueDefinition === void 0 ? void 0 : ccValueDefinition.forceCreation) === true)
            return true;
        const ccKeyValuePairDefinition = getCCKeyValuePairDefinitions(this).get(property);
        if ((ccKeyValuePairDefinition === null || ccKeyValuePairDefinition === void 0 ? void 0 : ccKeyValuePairDefinition.forceCreation) === true)
            return true;
        return false;
    }
    /** Determines if the given value should be persisted or represents an event */
    isStatefulValue(property) {
        const ccValueDefinition = getCCValueDefinitions(this).get(property);
        if ((ccValueDefinition === null || ccValueDefinition === void 0 ? void 0 : ccValueDefinition.stateful) === false)
            return false;
        const ccKeyValuePairDefinition = getCCKeyValuePairDefinitions(this).get(property);
        if ((ccKeyValuePairDefinition === null || ccKeyValuePairDefinition === void 0 ? void 0 : ccKeyValuePairDefinition.stateful) === false)
            return false;
        return true;
    }
    /** Persists all values on the given node into the value. Returns true if the process succeeded, false otherwise */
    persistValues(valueNames) {
        // In order to avoid cluttering applications with heaps of unsupported properties,
        // we filter out those that are only available in future versions of this CC
        // or have no version constraint
        const keyValuePairDefinitions = getCCKeyValuePairDefinitions(this);
        const keyValuePairs = [...keyValuePairDefinitions].filter(([, options]) => options.minVersion == undefined ||
            options.minVersion <= this.version);
        const ccValueDefinitions = [...getCCValueDefinitions(this)].filter(([, options]) => options.minVersion == undefined ||
            options.minVersion <= this.version);
        // If not specified otherwise, persist all registered values in the value db
        // But filter out those that don't match the minimum version
        if (!valueNames) {
            valueNames = [
                ...this._registeredCCValues.keys(),
                ...ccValueDefinitions.map(([key]) => key),
                ...keyValuePairs.map(([key]) => key),
            ];
        }
        let db;
        try {
            db = this.getValueDB();
        }
        catch (_a) {
            return false;
        }
        const cc = getCommandClass(this);
        for (const variable of valueNames) {
            // interviewComplete automatically updates the value DB, so no need to persist again
            if (variable === "interviewComplete")
                continue;
            // Only persist non-undefined values
            const sourceValue = this[variable];
            if (sourceValue == undefined &&
                !this.shouldValueAlwaysBeCreated(variable)) {
                continue;
            }
            if (keyValuePairDefinitions.has(variable)) {
                // This value is one or more key value pair(s) to be stored in a map
                if (sourceValue instanceof Map) {
                    // Just copy the entries
                    for (const [propertyKey, value] of sourceValue.entries()) {
                        db.setValue({
                            commandClass: cc,
                            endpoint: this.endpointIndex,
                            property: variable,
                            propertyKey,
                        }, value);
                    }
                }
                else if (typeguards_1.isArray(sourceValue)) {
                    const [propertyKey, value] = sourceValue;
                    db.setValue({
                        commandClass: cc,
                        endpoint: this.endpointIndex,
                        property: variable,
                        propertyKey,
                    }, value);
                }
                else {
                    throw new core_1.ZWaveError(`ccKeyValuePairs can only be Maps or [key, value]-tuples`, core_1.ZWaveErrorCodes.Argument_Invalid);
                }
            }
            else {
                // This value belongs to a simple property
                const valueId = {
                    commandClass: cc,
                    endpoint: this.endpointIndex,
                    property: variable,
                };
                // Avoid overwriting existing values with undefined if forceCreation is true
                if (sourceValue != undefined || !db.hasValue(valueId)) {
                    // Tell the value DB if this is a stateful value
                    const stateful = this.isStatefulValue(variable);
                    db.setValue(valueId, sourceValue, { stateful });
                }
            }
        }
        return true;
    }
    /** Serializes all values to be stored in the cache */
    serializeValuesForCache() {
        const ccValues = this.getValueDB().getValues(getCommandClass(this));
        return (ccValues
            // only serialize non-undefined values
            .filter(({ value }) => value != undefined)
            .map(({ value, commandClass, ...props }) => {
            return {
                ...props,
                value: core_1.serializeCacheValue(value),
            };
        }));
    }
    /** Serializes metadata to be stored in the cache */
    serializeMetadataForCache() {
        const allMetadata = this.getValueDB().getAllMetadata(getCommandClass(this));
        return (allMetadata
            // Strip out the command class
            .map(({ commandClass, ...props }) => props));
    }
    /** Deserializes values from the cache */
    deserializeValuesFromCache(values) {
        const cc = getCommandClass(this);
        for (const val of values) {
            this.getValueDB().setValue({
                commandClass: cc,
                endpoint: val.endpoint,
                property: val.property,
                propertyKey: val.propertyKey,
            }, core_1.deserializeCacheValue(val.value), {
                // Don't emit the added/updated events, as this will spam applications with untranslated events
                noEvent: true,
                // Don't throw when there is an invalid Value ID in the cache
                noThrow: true,
            });
        }
    }
    /** Deserializes value metadata from the cache */
    deserializeMetadataFromCache(allMetadata) {
        const cc = getCommandClass(this);
        for (const meta of allMetadata) {
            this.getValueDB().setMetadata({
                commandClass: cc,
                endpoint: meta.endpoint,
                property: meta.property,
                propertyKey: meta.propertyKey,
            }, meta.metadata, {
                // Don't emit the added/updated events, as this will spam applications with untranslated events
                noEvent: true,
                // Don't throw when there is an invalid Value ID in the cache
                noThrow: true,
            });
        }
    }
    /**
     * When a CC supports to be split into multiple partial CCs, this can be used to identify the
     * session the partial CCs belong to.
     * If a CC expects `mergePartialCCs` to be always called, you should return an empty object here.
     */
    getPartialCCSessionId() {
        return undefined; // Only select CCs support to be split
    }
    /** When a CC supports to be split into multiple partial CCs, this indicates that the last report hasn't been received yet */
    expectMoreMessages() {
        return false; // By default, all CCs are monolithic
    }
    /** Include previously received partial responses into a final CC */
    /* istanbul ignore next */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    mergePartialCCs(partials) {
        // This is highly CC dependent
        // Overwrite this in derived classes, by default do nothing
    }
    /** Tests whether this CC expects at least one command in return */
    expectsCCResponse() {
        let expected = getExpectedCCResponse(this);
        // Evaluate dynamic CC responses
        if (typeof expected === "function" &&
            !shared_1.staticExtends(expected, CommandClass)) {
            expected = expected(this);
        }
        if (expected === undefined)
            return false;
        if (typeguards_1.isArray(expected)) {
            return expected.every((cc) => shared_1.staticExtends(cc, CommandClass));
        }
        else {
            return shared_1.staticExtends(expected, CommandClass);
        }
    }
    isExpectedCCResponse(received) {
        var _a;
        if (received.nodeId !== this.nodeId)
            return false;
        let expected = getExpectedCCResponse(this);
        // Evaluate dynamic CC responses
        if (typeof expected === "function" &&
            !shared_1.staticExtends(expected, CommandClass)) {
            expected = expected(this);
        }
        if (expected == undefined) {
            // Fallback, should not happen if the expected response is defined correctly
            return false;
        }
        else if (typeguards_1.isArray(expected) &&
            expected.every((cc) => shared_1.staticExtends(cc, CommandClass))) {
            // The CC always expects a response from the given list, check if the received
            // message is in that list
            if (expected.every((base) => !(received instanceof base))) {
                return false;
            }
        }
        else if (shared_1.staticExtends(expected, CommandClass)) {
            // The CC always expects the same single response, check if this is the one
            if (!(received instanceof expected))
                return false;
        }
        // If the CC wants to test the response, let it
        const predicate = getCCResponsePredicate(this);
        const ret = (_a = predicate === null || predicate === void 0 ? void 0 : predicate(this, received)) !== null && _a !== void 0 ? _a : true;
        if (ret === "checkEncapsulated") {
            if (EncapsulatingCommandClass_1.isEncapsulatingCommandClass(this) &&
                EncapsulatingCommandClass_1.isEncapsulatingCommandClass(received)) {
                return this.encapsulated.isExpectedCCResponse(received.encapsulated);
            }
            else {
                // Fallback, should not happen if the expected response is defined correctly
                return false;
            }
        }
        return ret;
    }
    /**
     * Translates a property identifier into a speaking name for use in an external API
     * @param property The property identifier that should be translated
     * @param _propertyKey The (optional) property key the translated name may depend on
     */
    translateProperty(property, _propertyKey) {
        // Overwrite this in derived classes, by default just return the property key
        return property.toString();
    }
    /**
     * Translates a property key into a speaking name for use in an external API
     * @param property The property the key in question belongs to
     * @param propertyKey The property key for which the speaking name should be retrieved
     */
    translatePropertyKey(property, propertyKey) {
        // Overwrite this in derived classes, by default just return the property key
        return propertyKey.toString();
    }
    /** Whether this CC needs to exchange one or more messages before it can be sent */
    requiresPreTransmitHandshake() {
        return false; // By default it doesn't
    }
    /**
     * Perform a handshake before the actual message will be transmitted.
     */
    preTransmitHandshake() {
        return Promise.resolve();
        // By default do nothing
        // If handshake messages should be sent, they need the highest priority
    }
    /** Returns the number of bytes that are added to the payload by this CC */
    computeEncapsulationOverhead() {
        // Default is ccId (+ ccCommand):
        return (this.isExtended() ? 2 : 1) + 1;
    }
    /** Computes the maximum net payload size that can be transmitted inside this CC */
    getMaxPayloadLength(baseLength) {
        let ret = baseLength;
        let cur = this;
        while (cur) {
            ret -= cur.computeEncapsulationOverhead();
            cur = EncapsulatingCommandClass_1.isEncapsulatingCommandClass(cur)
                ? cur.encapsulated
                : undefined;
        }
        return ret;
    }
    /** Checks whether this CC is encapsulated with one that has the given CC id and (optionally) CC Command */
    isEncapsulatedWith(ccId, ccCommand) {
        let cc = this;
        // Check whether there was a S0 encapsulation
        while (cc.encapsulatingCC) {
            cc = cc.encapsulatingCC;
            if (cc.ccId === ccId &&
                (ccCommand === undefined || cc.ccCommand === ccCommand)) {
                return true;
            }
        }
        return false;
    }
}
exports.CommandClass = CommandClass;
// =======================
// use decorators to link command class values to actual command classes
const METADATA_commandClass = Symbol("commandClass");
const METADATA_commandClassMap = Symbol("commandClassMap");
const METADATA_ccResponse = Symbol("ccResponse");
const METADATA_ccCommand = Symbol("ccCommand");
const METADATA_ccCommandMap = Symbol("ccCommandMap");
const METADATA_ccValues = Symbol("ccValues");
const METADATA_ccKeyValuePairs = Symbol("ccKeyValuePairs");
const METADATA_ccValueMeta = Symbol("ccValueMeta");
const METADATA_version = Symbol("version");
const METADATA_API = Symbol("API");
const METADATA_APIMap = Symbol("APIMap");
function getCCCommandMapKey(ccId, ccCommand) {
    return JSON.stringify({ ccId, ccCommand });
}
/**
 * Defines the command class associated with a Z-Wave message
 */
function commandClass(cc) {
    return (messageClass) => {
        Reflect.defineMetadata(METADATA_commandClass, cc, messageClass);
        // also store a map in the Message metadata for lookup.
        const map = Reflect.getMetadata(METADATA_commandClassMap, CommandClass) ||
            new Map();
        map.set(cc, messageClass);
        Reflect.defineMetadata(METADATA_commandClassMap, map, CommandClass);
    };
}
exports.commandClass = commandClass;
/**
 * Retrieves the command class defined for a Z-Wave message class
 */
function getCommandClass(cc) {
    // get the class constructor
    const constr = cc.constructor;
    // retrieve the current metadata
    const ret = cc instanceof CommandClass
        ? Reflect.getMetadata(METADATA_commandClass, constr)
        : cc instanceof API_1.CCAPI
            ? Reflect.getMetadata(METADATA_API, constr)
            : undefined;
    if (ret == undefined) {
        throw new core_1.ZWaveError(`No command class defined for ${constr.name}!`, core_1.ZWaveErrorCodes.CC_Invalid);
    }
    return ret;
}
exports.getCommandClass = getCommandClass;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
function getCommandClassStatic(classConstructor) {
    // retrieve the current metadata
    const ret = Reflect.getMetadata(METADATA_commandClass, classConstructor);
    if (ret == undefined) {
        throw new core_1.ZWaveError(`No command class defined for ${classConstructor.name}!`, core_1.ZWaveErrorCodes.CC_Invalid);
    }
    return ret;
}
exports.getCommandClassStatic = getCommandClassStatic;
/**
 * Looks up the command class constructor for a given command class type and function type
 */
function getCCConstructor(cc) {
    // Retrieve the constructor map from the CommandClass class
    const map = Reflect.getMetadata(METADATA_commandClassMap, CommandClass);
    if (map != undefined)
        return map.get(cc);
}
exports.getCCConstructor = getCCConstructor;
/**
 * Defines the implemented version of a Z-Wave command class
 */
function implementedVersion(version) {
    return (ccClass) => {
        Reflect.defineMetadata(METADATA_version, version, ccClass);
    };
}
exports.implementedVersion = implementedVersion;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
function getImplementedVersion(cc) {
    // get the class constructor
    let constr;
    if (typeof cc === "number") {
        constr = getCCConstructor(cc);
    }
    else {
        constr = cc.constructor;
    }
    // retrieve the current metadata
    let ret;
    if (constr != undefined)
        ret = Reflect.getMetadata(METADATA_version, constr);
    if (ret == undefined)
        ret = 0;
    return ret;
}
exports.getImplementedVersion = getImplementedVersion;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
function getImplementedVersionStatic(classConstructor) {
    // retrieve the current metadata
    const ret = Reflect.getMetadata(METADATA_version, classConstructor) || 0;
    return ret;
}
exports.getImplementedVersionStatic = getImplementedVersionStatic;
/**
 * Defines the CC command a subclass of a CC implements
 */
function CCCommand(command) {
    return (ccClass) => {
        Reflect.defineMetadata(METADATA_ccCommand, command, ccClass);
        // also store a map in the Message metadata for lookup.
        const ccId = getCommandClassStatic(ccClass);
        const map = Reflect.getMetadata(METADATA_ccCommandMap, CommandClass) ||
            new Map();
        map.set(getCCCommandMapKey(ccId, command), ccClass);
        Reflect.defineMetadata(METADATA_ccCommandMap, map, CommandClass);
    };
}
exports.CCCommand = CCCommand;
/**
 * Retrieves the CC command a subclass of a CC implements
 */
function getCCCommand(cc) {
    // get the class constructor
    const constr = cc.constructor;
    // retrieve the current metadata
    const ret = Reflect.getMetadata(METADATA_ccCommand, constr);
    return ret;
}
/**
 * Looks up the command class constructor for a given command class type and function type
 */
// wotan-disable-next-line no-misused-generics
function getCCCommandConstructor(ccId, ccCommand) {
    // Retrieve the constructor map from the CommandClass class
    const map = Reflect.getMetadata(METADATA_ccCommandMap, CommandClass);
    if (map != undefined)
        return map.get(getCCCommandMapKey(ccId, ccCommand));
}
/**
 * Defines the expected response associated with a Z-Wave message
 */
function expectedCCResponse(cc, predicate) {
    return (ccClass) => {
        Reflect.defineMetadata(METADATA_ccResponse, { cc, predicate }, ccClass);
    };
}
exports.expectedCCResponse = expectedCCResponse;
/**
 * Retrieves the expected response (static or dynamic) defined for a Z-Wave message class
 */
function getExpectedCCResponse(ccClass) {
    // get the class constructor
    const constr = ccClass.constructor;
    // retrieve the current metadata
    const ret = Reflect.getMetadata(METADATA_ccResponse, constr);
    return ret === null || ret === void 0 ? void 0 : ret.cc;
}
exports.getExpectedCCResponse = getExpectedCCResponse;
/**
 * Retrieves the CC response predicate defined for a Z-Wave message class
 */
function getCCResponsePredicate(ccClass) {
    // get the class constructor
    const constr = ccClass.constructor;
    // retrieve the current metadata
    const ret = Reflect.getMetadata(METADATA_ccResponse, constr);
    return ret === null || ret === void 0 ? void 0 : ret.predicate;
}
exports.getCCResponsePredicate = getCCResponsePredicate;
/**
 * Marks the decorated property as a value of the Command Class. This allows saving it on the node with persistValues()
 * @param internal Whether the value should be exposed to library users
 */
function ccValue(options) {
    return (target, property) => {
        var _a;
        if (!target || !(target instanceof CommandClass))
            return;
        // Set default arguments
        if (!options)
            options = {};
        if (options.internal == undefined)
            options.internal = false;
        if (options.minVersion == undefined)
            options.minVersion = 1;
        if (options.forceCreation == undefined)
            options.forceCreation = false;
        // get the class constructor
        const constr = target.constructor;
        const cc = getCommandClassStatic(constr);
        // retrieve the current metadata
        const metadata = (_a = Reflect.getMetadata(METADATA_ccValues, CommandClass)) !== null && _a !== void 0 ? _a : {};
        if (!(cc in metadata))
            metadata[cc] = new Map();
        // And add the variable
        const variables = metadata[cc];
        variables.set(property, options);
        // store back to the object
        Reflect.defineMetadata(METADATA_ccValues, metadata, CommandClass);
    };
}
exports.ccValue = ccValue;
/**
 * Returns all CC values and their definitions that have been defined with @ccValue()
 */
function getCCValueDefinitions(commandClass) {
    var _a;
    // get the class constructor
    const constr = commandClass.constructor;
    const cc = getCommandClassStatic(constr);
    // retrieve the current metadata
    const metadata = (_a = Reflect.getMetadata(METADATA_ccValues, CommandClass)) !== null && _a !== void 0 ? _a : {};
    if (!(cc in metadata))
        return new Map();
    return metadata[cc];
}
/**
 * Marks the decorated property as the key of a Command Class's key value pair,
 * which can later be saved with persistValues()
 * @param internal Whether the key value pair should be exposed to library users
 */
function ccKeyValuePair(options) {
    return (target, property) => {
        if (!target || !(target instanceof CommandClass))
            return;
        // Set default arguments
        if (!options)
            options = {};
        if (options.internal == undefined)
            options.internal = false;
        if (options.minVersion == undefined)
            options.minVersion = 1;
        if (options.forceCreation == undefined)
            options.forceCreation = false;
        // get the class constructor
        const constr = target.constructor;
        const cc = getCommandClassStatic(constr);
        // retrieve the current metadata
        const metadata = Reflect.getMetadata(METADATA_ccKeyValuePairs, CommandClass) || {};
        if (!(cc in metadata))
            metadata[cc] = new Map();
        // And add the variable
        const variables = metadata[cc];
        variables.set(property, options);
        // store back to the object
        Reflect.defineMetadata(METADATA_ccKeyValuePairs, metadata, CommandClass);
    };
}
exports.ccKeyValuePair = ccKeyValuePair;
/**
 * Returns all CC key value pairs and their definitions that have been defined with @ccKeyValuePair()
 */
function getCCKeyValuePairDefinitions(commandClass) {
    // get the class constructor
    const constr = commandClass.constructor;
    const cc = getCommandClassStatic(constr);
    // retrieve the current metadata
    const metadata = Reflect.getMetadata(METADATA_ccKeyValuePairs, CommandClass) || {};
    if (!(cc in metadata))
        return new Map();
    return metadata[cc];
}
/**
 * Defines additional metadata for the given CC value
 */
function ccValueMetadata(meta) {
    return (target, property) => {
        if (!target || !(target instanceof CommandClass))
            return;
        // get the class constructor
        const constr = target.constructor;
        const cc = getCommandClassStatic(constr);
        // retrieve the current metadata
        const metadata = Reflect.getMetadata(METADATA_ccValueMeta, CommandClass) || {};
        if (!(cc in metadata))
            metadata[cc] = new Map();
        // And add the variable
        const variables = metadata[cc];
        variables.set(property, meta);
        // store back to the object
        Reflect.defineMetadata(METADATA_ccValueMeta, metadata, CommandClass);
    };
}
exports.ccValueMetadata = ccValueMetadata;
/**
 * Retrieves defined metadata for the given CC value. If none is found, the default settings are returned.
 */
function getCCValueMetadata(cc, property) {
    // retrieve the current metadata
    const metadata = Reflect.getMetadata(METADATA_ccValueMeta, CommandClass) || {};
    if (!(cc in metadata))
        return core_1.ValueMetadata.Any;
    const map = metadata[cc];
    if (map.has(property))
        return map.get(property);
    return core_1.ValueMetadata.Any;
}
exports.getCCValueMetadata = getCCValueMetadata;
/**
 * Defines the simplified API associated with a Z-Wave command class
 */
function API(cc) {
    return (apiClass) => {
        // and store the metadata
        Reflect.defineMetadata(METADATA_API, cc, apiClass);
        // also store a map in the CCAPI metadata for lookup.
        const map = (Reflect.getMetadata(METADATA_APIMap, API_1.CCAPI) ||
            new Map());
        map.set(cc, apiClass);
        Reflect.defineMetadata(METADATA_APIMap, map, API_1.CCAPI);
    };
}
exports.API = API;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
function getAPI(cc) {
    // Retrieve the constructor map from the CCAPI class
    const map = Reflect.getMetadata(METADATA_APIMap, API_1.CCAPI);
    const ret = map === null || map === void 0 ? void 0 : map.get(cc);
    return ret;
}
exports.getAPI = getAPI;

//# sourceMappingURL=CommandClass.js.map
