"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSendThreadMachine = void 0;
const core_1 = require("@zwave-js/core");
const sorted_list_1 = require("alcalzone-shared/sorted-list");
const xstate_1 = require("xstate");
const actions_1 = require("xstate/lib/actions");
const ICommandClassContainer_1 = require("../commandclass/ICommandClassContainer");
const NoOperationCC_1 = require("../commandclass/NoOperationCC");
const ApplicationCommandRequest_1 = require("../controller/ApplicationCommandRequest");
const BridgeApplicationCommandRequest_1 = require("../controller/BridgeApplicationCommandRequest");
const SendDataMessages_1 = require("../controller/SendDataMessages");
const Constants_1 = require("../message/Constants");
const CommandQueueMachine_1 = require("./CommandQueueMachine");
const StateMachineShared_1 = require("./StateMachineShared");
// These actions must be assign actions or they will be executed out of order
const setCurrentTransaction = xstate_1.assign((ctx) => {
    const queue = ctx.queue;
    const next = ctx.queue.shift();
    return {
        ...ctx,
        queue,
        currentTransaction: next,
    };
});
const deleteCurrentTransaction = xstate_1.assign((ctx) => ({
    ...ctx,
    currentTransaction: undefined,
}));
const deleteHandshakeTransaction = xstate_1.assign((ctx) => ({
    ...ctx,
    handshakeTransaction: undefined,
}));
const resetSendDataAttempts = xstate_1.assign({
    sendDataAttempts: (_) => 0,
});
const incrementSendDataAttempts = xstate_1.assign({
    sendDataAttempts: (ctx) => ctx.sendDataAttempts + 1,
});
const forwardToCommandQueue = xstate_1.forwardTo((ctx) => ctx.commandQueue);
const currentTransactionIsSendData = (ctx) => {
    var _a;
    const msg = (_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message;
    return (msg instanceof SendDataMessages_1.SendDataRequest ||
        msg instanceof SendDataMessages_1.SendDataMulticastRequest);
};
const forwardNodeUpdate = actions_1.pure((ctx, evt) => {
    return actions_1.raise({
        type: "nodeUpdate",
        result: evt.message,
    });
});
const forwardHandshakeResponse = actions_1.pure((ctx, evt) => {
    return actions_1.raise({
        type: "handshakeResponse",
        result: evt.message,
    });
});
const forwardActiveCommandSuccess = actions_1.pure((ctx, evt) => {
    return actions_1.raise({ ...evt, type: "active_command_success" });
});
const forwardActiveCommandFailure = actions_1.pure((ctx, evt) => {
    return actions_1.raise({ ...evt, type: "active_command_failure" });
});
const forwardActiveCommandError = actions_1.pure((ctx, evt) => {
    return actions_1.raise({ ...evt, type: "active_command_error" });
});
const sendCurrentTransactionToCommandQueue = actions_1.send((ctx) => ({
    type: "add",
    transaction: ctx.currentTransaction,
}), { to: (ctx) => ctx.commandQueue });
const resetCommandQueue = actions_1.send("reset", {
    to: (ctx) => ctx.commandQueue,
});
const sortQueue = xstate_1.assign({
    queue: (ctx) => {
        const queue = ctx.queue;
        const items = [...queue];
        queue.clear();
        // Since the send queue is a sorted list, sorting is done on insert/add
        queue.add(...items);
        return queue;
    },
});
const every = (...guards) => ({
    type: "every",
    guards,
});
const guards = {
    maySendFirstMessage: (ctx) => {
        const nextTransaction = ctx.queue.peekStart();
        // We can't send anything if the queue is empty
        if (!nextTransaction)
            return false;
        const message = nextTransaction.message;
        const targetNode = message.getNodeUnsafe();
        // The send queue is sorted automatically. If the first message is for a sleeping node, all messages in the queue are.
        // There are two exceptions:
        // 1. Pings may be used to determine whether a node is really asleep.
        // 2. Responses to handshake requests must always be sent, because some sleeping nodes may try to send us encrypted messages.
        //    If we don't send them, they block the send queue
        return (!targetNode ||
            targetNode.isAwake() ||
            NoOperationCC_1.messageIsPing(message) ||
            nextTransaction.priority === Constants_1.MessagePriority.Handshake);
    },
    requiresNoHandshake: (ctx) => {
        var _a;
        const msg = (_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message;
        if (!(msg instanceof SendDataMessages_1.SendDataRequest)) {
            return true;
        }
        return !msg.command.requiresPreTransmitHandshake();
    },
    isForActiveTransaction: (ctx, evt, meta) => {
        return (((meta.state.matches("sending.handshake") ||
            !!ctx.handshakeTransaction) &&
            evt.transaction === ctx.handshakeTransaction) ||
            ((meta.state.matches("sending.execute") ||
                meta.state.matches("sending.waitForUpdate") ||
                !!ctx.currentTransaction) &&
                evt.transaction === ctx.currentTransaction));
    },
    expectsNodeUpdate: (ctx) => {
        var _a;
        return ((_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message) instanceof SendDataMessages_1.SendDataRequest &&
            ctx.currentTransaction
                .message.command.expectsCCResponse();
    },
    isExpectedUpdate: (ctx, evt, meta) => {
        if (!meta.state.matches("sending.waitForUpdate"))
            return false;
        const sentMsg = ctx.currentTransaction.message;
        const receivedMsg = evt.message;
        return ((receivedMsg instanceof ApplicationCommandRequest_1.ApplicationCommandRequest ||
            receivedMsg instanceof BridgeApplicationCommandRequest_1.BridgeApplicationCommandRequest) &&
            sentMsg.command.isExpectedCCResponse(receivedMsg.command));
    },
    currentTransactionIsSendData,
    mayRetry: (ctx, evt) => {
        const msg = ctx.currentTransaction.message;
        if (msg instanceof SendDataMessages_1.SendDataMulticastRequest) {
            // Don't try to resend multicast messages if they were already transmitted.
            // One or more nodes might have already reacted
            if (evt.reason === "callback NOK") {
                return false;
            }
        }
        return (msg
            .maxSendAttempts > ctx.sendDataAttempts);
    },
    /** Whether the message is an outgoing pre-transmit handshake */
    isPreTransmitHandshakeForCurrentTransaction: (ctx, evt, meta) => {
        if (!meta.state.matches("sending.handshake"))
            return false;
        // Ensure that the current transaction is SendData
        if (!currentTransactionIsSendData(ctx))
            return false;
        const transaction = evt.transaction;
        if (transaction.priority !== Constants_1.MessagePriority.PreTransmitHandshake)
            return false;
        if (!(transaction.message instanceof SendDataMessages_1.SendDataRequest))
            return false;
        // require the handshake to be for the same node
        return (transaction.message.command.nodeId ===
            ctx.currentTransaction.message.command.nodeId);
    },
    isExpectedHandshakeResponse: (ctx, evt, meta) => {
        if (!ctx.handshakeTransaction)
            return false;
        if (!meta.state.matches("sending.handshake.waitForHandshakeResponse"))
            return false;
        const sentMsg = ctx.handshakeTransaction.message;
        const receivedMsg = evt.message;
        if (!ICommandClassContainer_1.isCommandClassContainer(receivedMsg))
            return false;
        return ((receivedMsg instanceof ApplicationCommandRequest_1.ApplicationCommandRequest ||
            receivedMsg instanceof BridgeApplicationCommandRequest_1.BridgeApplicationCommandRequest) &&
            sentMsg.command.isExpectedCCResponse(receivedMsg.command));
    },
    /** Whether the message is an outgoing handshake response to the current node*/
    isHandshakeForCurrentTransaction: (ctx, evt) => {
        // First ensure that the current transaction is SendData
        if (!currentTransactionIsSendData(ctx))
            return false;
        // Then ensure that the event transaction is also SendData
        const transaction = evt.transaction;
        if (transaction.priority !== Constants_1.MessagePriority.Handshake)
            return false;
        if (!(transaction.message instanceof SendDataMessages_1.SendDataRequest))
            return false;
        // require the handshake to be for the same node
        return (transaction.message.command.nodeId ===
            ctx.currentTransaction.message.command.nodeId);
    },
    shouldNotKeepCurrentTransaction: (ctx, evt) => {
        const reducer = evt.reducer;
        return reducer(ctx.currentTransaction, "current").type !== "keep";
    },
    currentTransactionIsPingForNode: (ctx, evt) => {
        var _a;
        const msg = (_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message;
        return (!!msg &&
            NoOperationCC_1.messageIsPing(msg) &&
            msg.getNodeId() === evt.nodeId);
    },
};
function createMessageDroppedUnexpectedError(original) {
    const ret = new core_1.ZWaveError(`Message dropped because of an unexpected error: ${original.message}`, core_1.ZWaveErrorCodes.Controller_MessageDropped);
    if (original.stack)
        ret.stack = original.stack;
    return ret;
}
function createSendThreadMachine(implementations, params) {
    const resolveCurrentTransaction = xstate_1.assign((ctx, evt) => {
        implementations.resolveTransaction(ctx.currentTransaction, evt.result);
        return ctx;
    });
    const resolveCurrentTransactionWithoutMessage = xstate_1.assign((ctx) => {
        implementations.resolveTransaction(ctx.currentTransaction, undefined);
        return ctx;
    });
    const rejectCurrentTransaction = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(ctx.currentTransaction, StateMachineShared_1.sendDataErrorToZWaveError(evt.reason, ctx.currentTransaction, evt.result));
        return ctx;
    });
    const rejectCurrentTransactionWithError = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(ctx.currentTransaction, createMessageDroppedUnexpectedError(evt.error));
        return ctx;
    });
    const rejectCurrentTransactionWithNodeTimeout = xstate_1.assign((ctx) => {
        implementations.rejectTransaction(ctx.currentTransaction, StateMachineShared_1.sendDataErrorToZWaveError("node timeout", ctx.currentTransaction, undefined));
        return ctx;
    });
    const resolveHandshakeTransaction = xstate_1.assign((ctx, evt) => {
        implementations.resolveTransaction(ctx.handshakeTransaction, evt.result);
        return ctx;
    });
    const rejectHandshakeTransaction = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(ctx.handshakeTransaction, StateMachineShared_1.sendDataErrorToZWaveError(evt.reason, ctx.handshakeTransaction, evt.result));
        return ctx;
    });
    const rejectHandshakeTransactionWithError = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(ctx.handshakeTransaction, createMessageDroppedUnexpectedError(evt.error));
        return ctx;
    });
    const rejectHandshakeTransactionWithNodeTimeout = xstate_1.assign((ctx) => {
        const hsTransaction = ctx.handshakeTransaction;
        if (hsTransaction) {
            implementations.rejectTransaction(hsTransaction, StateMachineShared_1.sendDataErrorToZWaveError("node timeout", hsTransaction, undefined));
        }
        return ctx;
    });
    const resolveEventTransaction = xstate_1.assign((ctx, evt) => {
        implementations.resolveTransaction(evt.transaction, evt.result);
        return ctx;
    });
    const rejectEventTransaction = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(evt.transaction, StateMachineShared_1.sendDataErrorToZWaveError(evt.reason, evt.transaction.message, evt.result));
        return ctx;
    });
    const rejectEventTransactionWithError = xstate_1.assign((ctx, evt) => {
        implementations.rejectTransaction(evt.transaction, createMessageDroppedUnexpectedError(evt.error));
        return ctx;
    });
    const notifyUnsolicited = (_, evt) => {
        implementations.notifyUnsolicited(evt.message);
    };
    const reduce = xstate_1.assign({
        queue: (ctx, evt) => {
            const { queue, currentTransaction } = ctx;
            const drop = [];
            const requeue = [];
            const reduceTransaction = (transaction, source) => {
                const reducerResult = evt.reducer(transaction, source);
                switch (reducerResult.type) {
                    case "drop":
                        drop.push(transaction);
                        break;
                    case "requeue":
                        if (reducerResult.priority != undefined) {
                            transaction.priority = reducerResult.priority;
                        }
                        requeue.push(transaction);
                        break;
                    case "resolve":
                        implementations.resolveTransaction(transaction, reducerResult.message);
                        drop.push(transaction);
                        break;
                    case "reject":
                        implementations.rejectTransaction(transaction, new core_1.ZWaveError(reducerResult.message, reducerResult.code, undefined, transaction.stack));
                        drop.push(transaction);
                        break;
                }
            };
            for (const transaction of queue) {
                reduceTransaction(transaction, "queue");
            }
            if (currentTransaction) {
                reduceTransaction(currentTransaction, "current");
            }
            // Now we know what to do with the transactions
            queue.remove(...drop, ...requeue);
            queue.add(...requeue);
            return queue;
        },
    });
    const ret = xstate_1.Machine({
        id: "SendThread",
        initial: "init",
        context: {
            commandQueue: undefined,
            queue: new sorted_list_1.SortedList(),
            sendDataAttempts: 0,
        },
        on: {
            // Forward low-level events and unidentified messages to the command queue
            ACK: { actions: forwardToCommandQueue },
            CAN: { actions: forwardToCommandQueue },
            NAK: { actions: forwardToCommandQueue },
            // messages may come back as "unsolicited", these might be expected updates
            // we need to run them through the serial API machine to avoid mismatches
            message: { actions: forwardToCommandQueue },
            // resolve/reject any un-interesting transactions if they are done
            command_success: [
                // If this notification belongs to an active command, forward it
                {
                    cond: "isForActiveTransaction",
                    actions: forwardActiveCommandSuccess,
                },
                // otherwise just resolve it
                {
                    actions: resolveEventTransaction,
                },
            ],
            command_failure: [
                // If this notification belongs to an active command, forward it
                {
                    cond: "isForActiveTransaction",
                    actions: forwardActiveCommandFailure,
                },
                // otherwise just reject it
                {
                    actions: rejectEventTransaction,
                },
            ],
            command_error: [
                // If this notification belongs to an active command, forward it
                {
                    cond: "isForActiveTransaction",
                    actions: forwardActiveCommandError,
                },
                // otherwise just reject it
                {
                    actions: rejectEventTransactionWithError,
                },
            ],
            // handle newly added messages
            add: [
                // Trigger outgoing handshakes immediately without queueing
                {
                    cond: "isPreTransmitHandshakeForCurrentTransaction",
                    actions: [
                        forwardToCommandQueue,
                        // and inform the state machine when it is the one we've waited for
                        xstate_1.assign({
                            handshakeTransaction: (_, evt) => evt.transaction,
                        }),
                    ],
                },
                // Forward all handshake messages that could have to do with the current transaction
                {
                    cond: "isHandshakeForCurrentTransaction",
                    actions: forwardToCommandQueue,
                },
                {
                    actions: [
                        xstate_1.assign({
                            queue: (ctx, evt) => {
                                ctx.queue.add(evt.transaction);
                                return ctx.queue;
                            },
                        }),
                        actions_1.raise("trigger"),
                    ],
                },
            ],
            unsolicited: [
                // If a message is returned by the serial API, they might be an expected node update
                {
                    cond: "isExpectedHandshakeResponse",
                    actions: forwardHandshakeResponse,
                },
                {
                    cond: "isExpectedUpdate",
                    actions: forwardNodeUpdate,
                },
                // Return unsolicited messages to the driver
                { actions: notifyUnsolicited },
            ],
            // Accept external commands to sort the queue
            sortQueue: {
                actions: [sortQueue, actions_1.raise("trigger")],
            },
        },
        states: {
            init: {
                entry: xstate_1.assign({
                    commandQueue: () => xstate_1.spawn(CommandQueueMachine_1.createCommandQueueMachine(implementations, params), {
                        name: "commandQueue",
                    }),
                }),
                // Spawn the command queue when starting the send thread
                always: "idle",
            },
            idle: {
                id: "idle",
                entry: [deleteCurrentTransaction, resetSendDataAttempts],
                always: [
                    { cond: "maySendFirstMessage", target: "sending" },
                ],
                on: {
                    trigger: [
                        {
                            cond: "maySendFirstMessage",
                            target: "sending",
                        },
                    ],
                    reduce: {
                        // Reducing may reorder the queue, so raise a trigger afterwards
                        actions: [reduce, actions_1.raise("trigger")],
                    },
                },
            },
            sending: {
                id: "sending",
                // Use the first transaction in the queue as the current one
                entry: setCurrentTransaction,
                initial: "beforeSend",
                on: {
                    NIF: {
                        // Pings are not retransmitted and won't receive a response if the node wake up after the ping was sent
                        // Therefore resolve pending pings so the communication may proceed immediately
                        cond: "currentTransactionIsPingForNode",
                        actions: [
                            resolveCurrentTransactionWithoutMessage,
                        ],
                        target: "sending.done",
                        internal: true,
                    },
                    reduce: [
                        // If the current transaction should not be kept, tell the send queue to abort it and go back to idle
                        {
                            cond: "shouldNotKeepCurrentTransaction",
                            actions: [resetCommandQueue, reduce],
                            target: "sending.done",
                            internal: true,
                        },
                        { actions: reduce },
                    ],
                },
                states: {
                    beforeSend: {
                        entry: [
                            actions_1.pure((ctx) => currentTransactionIsSendData(ctx)
                                ? incrementSendDataAttempts
                                : undefined),
                            deleteHandshakeTransaction,
                        ],
                        always: [
                            // Skip this step if no handshake is required
                            {
                                cond: "requiresNoHandshake",
                                target: "execute",
                            },
                            // else begin the handshake process
                            {
                                target: "handshake",
                            },
                        ],
                    },
                    handshake: {
                        // Just send the handshake as a side effect
                        invoke: {
                            id: "preTransmitHandshake",
                            src: "preTransmitHandshake",
                            onDone: "#sending.execute",
                        },
                        initial: "waitForCommandResult",
                        on: {
                            handshakeResponse: {
                                actions: resolveHandshakeTransaction,
                            },
                        },
                        states: {
                            // After kicking off the command, wait until it is completed
                            waitForCommandResult: {
                                on: {
                                    // On success, start waiting for the handshake response
                                    active_command_success: "waitForHandshakeResponse",
                                    active_command_failure: [
                                        // On failure, retry SendData commands if possible
                                        {
                                            cond: "mayRetry",
                                            actions: rejectHandshakeTransaction,
                                            target: "#sending.retryWait",
                                        },
                                        // Otherwise reject the transaction
                                        {
                                            actions: [
                                                rejectHandshakeTransaction,
                                                rejectCurrentTransaction,
                                            ],
                                            target: "#sending.done",
                                        },
                                    ],
                                    active_command_error: [
                                        // On failure, retry SendData commands if possible
                                        {
                                            cond: "mayRetry",
                                            actions: rejectHandshakeTransactionWithError,
                                            target: "#sending.retryWait",
                                        },
                                        // Otherwise reject the transaction
                                        {
                                            actions: [
                                                rejectHandshakeTransactionWithError,
                                                rejectCurrentTransactionWithError,
                                            ],
                                            target: "#sending.done",
                                        },
                                    ],
                                },
                            },
                            waitForHandshakeResponse: {
                                after: {
                                    // If an update times out, retry if possible - otherwise reject the entire transaction
                                    REPORT_TIMEOUT: [
                                        // only retry on timeout when configured
                                        ...(params.attempts
                                            .retryAfterTransmitReport
                                            ? [
                                                {
                                                    cond: "mayRetry",
                                                    target: "#sending.retryWait",
                                                    actions: rejectHandshakeTransactionWithNodeTimeout,
                                                },
                                            ]
                                            : []),
                                        {
                                            actions: [
                                                rejectHandshakeTransactionWithNodeTimeout,
                                                rejectCurrentTransactionWithNodeTimeout,
                                            ],
                                            target: "#sending.done",
                                        },
                                    ],
                                },
                            },
                        },
                    },
                    execute: {
                        entry: [
                            deleteHandshakeTransaction,
                            sendCurrentTransactionToCommandQueue,
                        ],
                        on: {
                            active_command_success: [
                                // On success, start waiting for an update
                                {
                                    cond: "expectsNodeUpdate",
                                    target: "waitForUpdate",
                                },
                                // or resolve the current transaction if none is required
                                {
                                    actions: resolveCurrentTransaction,
                                    target: "done",
                                },
                            ],
                            active_command_failure: [
                                // On failure, retry SendData commands if possible
                                {
                                    cond: every("currentTransactionIsSendData", "mayRetry"),
                                    target: "retryWait",
                                },
                                // Otherwise reject the transaction
                                {
                                    actions: rejectCurrentTransaction,
                                    target: "done",
                                },
                            ],
                            active_command_error: [
                                // On failure, retry SendData commands if possible
                                {
                                    cond: every("currentTransactionIsSendData", "mayRetry"),
                                    target: "retryWait",
                                },
                                // Otherwise reject the transaction
                                {
                                    actions: rejectCurrentTransactionWithError,
                                    target: "done",
                                },
                            ],
                        },
                    },
                    waitForUpdate: {
                        on: {
                            nodeUpdate: {
                                actions: resolveCurrentTransaction,
                                target: "done",
                            },
                        },
                        after: {
                            // If an update times out, retry if possible - otherwise reject the transaction
                            REPORT_TIMEOUT: [
                                // only retry on timeout when configured
                                ...(params.attempts.retryAfterTransmitReport
                                    ? [
                                        {
                                            cond: "mayRetry",
                                            target: "retryWait",
                                        },
                                    ]
                                    : []),
                                {
                                    actions: rejectCurrentTransactionWithNodeTimeout,
                                    target: "done",
                                },
                            ],
                        },
                    },
                    retryWait: {
                        invoke: {
                            id: "notify",
                            src: "notifyRetry",
                        },
                        after: {
                            500: "beforeSend",
                        },
                    },
                    done: {
                        // Clean up the context after sending
                        always: {
                            target: "#idle",
                            actions: [
                                deleteCurrentTransaction,
                                deleteHandshakeTransaction,
                                resetSendDataAttempts,
                            ],
                        },
                    },
                },
            },
        },
    }, {
        services: {
            preTransmitHandshake: async (ctx) => {
                // Execute the pre transmit handshake and swallow all errors
                try {
                    await ctx.currentTransaction
                        .message.command.preTransmitHandshake();
                }
                catch (e) { }
            },
            notifyRetry: (ctx) => {
                var _a;
                (_a = implementations.notifyRetry) === null || _a === void 0 ? void 0 : _a.call(implementations, "SendData", undefined, ctx.currentTransaction.message, ctx.sendDataAttempts, ctx.currentTransaction.message
                    .maxSendAttempts, 500);
                return Promise.resolve();
            },
        },
        guards: {
            ...guards,
            every: (ctx, event, { cond }) => {
                const keys = cond.guards;
                return keys.every((guardKey) => guards[guardKey](ctx, event, undefined));
            },
        },
        delays: {
            REPORT_TIMEOUT: params.timeouts.report,
        },
    });
    return ret;
}
exports.createSendThreadMachine = createSendThreadMachine;

//# sourceMappingURL=SendThreadMachine.js.map
