"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IndicatorProperty = exports.loadIndicatorsInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const indicatorsConfigPath = path_1.default.join(utils_1.configDir, "indicators.json");
/** @internal */
async function loadIndicatorsInternal() {
    if (!(await fs_extra_1.pathExists(indicatorsConfigPath))) {
        throw new core_1.ZWaveError("The config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(indicatorsConfigPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("indicators", "the database is not an object");
        }
        if (!("indicators" in definition)) {
            utils_1.throwInvalidConfig("indicators", `the required key "indicators" is missing`);
        }
        if (!("properties" in definition)) {
            utils_1.throwInvalidConfig("indicators", `the required key "properties" is missing`);
        }
        const indicators = new Map();
        for (const [id, label] of objects_1.entries(definition.indicators)) {
            if (!utils_1.hexKeyRegexNDigits.test(id)) {
                utils_1.throwInvalidConfig("indicators", `found non-hex key "${id}" in "indicators"`);
            }
            const idNum = parseInt(id.slice(2), 16);
            indicators.set(idNum, label);
        }
        const properties = new Map();
        for (const [id, propDefinition] of objects_1.entries(definition.properties)) {
            if (!utils_1.hexKeyRegexNDigits.test(id)) {
                utils_1.throwInvalidConfig("indicators", `found non-hex key "${id}" in "properties"`);
            }
            const idNum = parseInt(id.slice(2), 16);
            properties.set(idNum, new IndicatorProperty(idNum, propDefinition));
        }
        return { indicators, properties };
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("indicators");
        }
    }
}
exports.loadIndicatorsInternal = loadIndicatorsInternal;
class IndicatorProperty {
    constructor(id, definition) {
        this.id = id;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("indicators", `The label for property ${shared_1.num2hex(id)} is not a string!`);
        }
        this.label = definition.label;
        if (definition.description != undefined &&
            typeof definition.description !== "string") {
            utils_1.throwInvalidConfig("indicators", `The description for property ${shared_1.num2hex(id)} is not a string!`);
        }
        this.description = definition.description;
        if (definition.min != undefined && typeof definition.min !== "number") {
            utils_1.throwInvalidConfig("indicators", `The minimum value for property ${shared_1.num2hex(id)} is not a number!`);
        }
        this.min = definition.min;
        if (definition.max != undefined && typeof definition.max !== "number") {
            utils_1.throwInvalidConfig("indicators", `The maximum value for property ${shared_1.num2hex(id)} is not a number!`);
        }
        this.max = definition.max;
        if (definition.readonly != undefined &&
            typeof definition.readonly !== "boolean") {
            utils_1.throwInvalidConfig("indicators", `readonly for property ${shared_1.num2hex(id)} is not a boolean!`);
        }
        this.readonly = definition.readonly;
        if (definition.type != undefined &&
            typeof definition.type !== "string") {
            utils_1.throwInvalidConfig("indicators", `type for property ${shared_1.num2hex(id)} is not a string!`);
        }
        // TODO: Validate that the value is ok
        this.type = definition.type;
    }
}
exports.IndicatorProperty = IndicatorProperty;
//# sourceMappingURL=Indicators.js.map