"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Scale = exports.getDefaultScale = exports.loadNamedScalesInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "scales.json");
/** @internal */
async function loadNamedScalesInternal() {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The named scales config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("named scales", `the dictionary is not an object`);
        }
        const namedScales = new Map();
        for (const [name, scales] of objects_1.entries(definition)) {
            if (!/[\w\d]+/.test(name)) {
                utils_1.throwInvalidConfig("named scales", `Name ${name} contains other characters than letters and numbers`);
            }
            const named = new Map();
            for (const [key, scaleDefinition] of objects_1.entries(scales)) {
                if (!utils_1.hexKeyRegexNDigits.test(key)) {
                    utils_1.throwInvalidConfig("named scales", `found non-hex key "${key}" in the definition for "${name}"`);
                }
                const keyNum = parseInt(key.slice(2), 16);
                named.set(keyNum, new Scale(keyNum, scaleDefinition));
            }
            namedScales.set(name, named);
        }
        return namedScales;
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("named scales");
        }
    }
}
exports.loadNamedScalesInternal = loadNamedScalesInternal;
function getDefaultScale(scale) {
    return new Scale(scale, {
        unit: undefined,
        label: "Unknown",
    });
}
exports.getDefaultScale = getDefaultScale;
class Scale {
    constructor(key, definition) {
        this.key = key;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("named scales", `The label for scale ${shared_1.num2hex(key)} is not a string!`);
        }
        this.label = definition.label;
        if (definition.unit != undefined &&
            typeof definition.unit !== "string") {
            utils_1.throwInvalidConfig("named scales", `The unit for scale ${shared_1.num2hex(key)} is not a string!`);
        }
        this.unit = definition.unit;
        if (definition.description != undefined &&
            typeof definition.description !== "string") {
            utils_1.throwInvalidConfig("named scales", `The description for scale ${shared_1.num2hex(key)} is not a string!`);
        }
        this.description = definition.description;
    }
}
exports.Scale = Scale;
//# sourceMappingURL=Scales.js.map