/// <reference types="node" />
import type { ValueID } from "@zwave-js/core";
import { CommandClasses, Maybe, NODE_ID_BROADCAST } from "@zwave-js/core";
import { ObjectKeyMap } from "@zwave-js/shared";
import type { Driver, SendCommandOptions } from "../driver/Driver";
import type { Endpoint } from "../node/Endpoint";
import { VirtualEndpoint } from "../node/VirtualEndpoint";
export declare type ValueIDProperties = Pick<ValueID, "property" | "propertyKey">;
/** Used to identify the method on the CC API class that handles setting values on nodes directly */
export declare const SET_VALUE: unique symbol;
export declare type SetValueImplementation = (property: ValueIDProperties, value: unknown) => Promise<void>;
/** Used to identify the method on the CC API class that handles polling values from nodes */
export declare const POLL_VALUE: unique symbol;
export declare type PollValueImplementation<T extends unknown = unknown> = (property: ValueIDProperties) => Promise<T | undefined>;
export declare function throwUnsupportedProperty(cc: CommandClasses, property: string | number): never;
export declare function throwUnsupportedPropertyKey(cc: CommandClasses, property: string | number, propertyKey: string | number): never;
export declare function throwMissingPropertyKey(cc: CommandClasses, property: string | number): never;
export declare function throwWrongValueType(cc: CommandClasses, property: string | number, expectedType: string, receivedType: string): never;
/**
 * The base class for all CC APIs exposed via `Node.commandClasses.<CCName>`
 * @publicAPI
 */
export declare class CCAPI {
    protected readonly driver: Driver;
    protected readonly endpoint: Endpoint | VirtualEndpoint;
    constructor(driver: Driver, endpoint: Endpoint | VirtualEndpoint);
    protected [SET_VALUE]: SetValueImplementation | undefined;
    /**
     * Can be used on supported CC APIs to set a CC value by property name (and optionally the property key)
     */
    get setValue(): SetValueImplementation | undefined;
    protected [POLL_VALUE]: PollValueImplementation | undefined;
    /**
     * Can be used on supported CC APIs to poll a CC value by property name (and optionally the property key)
     */
    get pollValue(): PollValueImplementation | undefined;
    protected scheduledPolls: ObjectKeyMap<ValueIDProperties, NodeJS.Timeout>;
    /**
     * Schedules a value to be polled after a given time. Schedules are deduplicated on a per-property basis.
     * @returns `true` if the poll was scheduled, `false` otherwise
     */
    protected schedulePoll(property: ValueIDProperties, timeoutMs?: number): boolean;
    /**
     * Retrieves the version of the given CommandClass this endpoint implements
     */
    get version(): number;
    /** Determines if this simplified API instance may be used. */
    isSupported(): boolean;
    /**
     * Determine whether the linked node supports a specific command of this command class.
     * "unknown" means that the information has not been received yet
     */
    supportsCommand(command: number): Maybe<boolean>;
    protected assertSupportsCommand(commandEnum: unknown, command: number): void;
    protected assertPhysicalEndpoint(endpoint: Endpoint | VirtualEndpoint): asserts endpoint is Endpoint;
    /** Returns the command options to use for sendCommand calls */
    protected get commandOptions(): SendCommandOptions;
    /** Creates an instance of this API, scoped to use the given options */
    withOptions(options: SendCommandOptions): this;
    protected isSinglecast(): this is this & {
        endpoint: Endpoint;
    };
    protected isMulticast(): this is this & {
        endpoint: VirtualEndpoint & {
            nodeId: number[];
        };
    };
    protected isBroadcast(): this is this & {
        endpoint: VirtualEndpoint & {
            nodeId: typeof NODE_ID_BROADCAST;
        };
    };
}
/** A CC API that is only available for physical endpoints */
export declare class PhysicalCCAPI extends CCAPI {
    constructor(driver: Driver, endpoint: Endpoint | VirtualEndpoint);
    protected readonly endpoint: Endpoint;
}
export interface CCAPIs {
    [Symbol.iterator](): Iterator<CCAPI>;
    "Alarm Sensor": import("./AlarmSensorCC").AlarmSensorCCAPI;
    Association: import("./AssociationCC").AssociationCCAPI;
    "Association Group Information": import("./AssociationGroupInfoCC").AssociationGroupInfoCCAPI;
    Basic: import("./BasicCC").BasicCCAPI;
    Battery: import("./BatteryCC").BatteryCCAPI;
    "Binary Sensor": import("./BinarySensorCC").BinarySensorCCAPI;
    "Binary Switch": import("./BinarySwitchCC").BinarySwitchCCAPI;
    "CRC-16 Encapsulation": import("./CRC16CC").CRC16CCAPI;
    "Central Scene": import("./CentralSceneCC").CentralSceneCCAPI;
    "Climate Control Schedule": import("./ClimateControlScheduleCC").ClimateControlScheduleCCAPI;
    Clock: import("./ClockCC").ClockCCAPI;
    "Color Switch": import("./ColorSwitchCC").ColorSwitchCCAPI;
    Configuration: import("./ConfigurationCC").ConfigurationCCAPI;
    "Door Lock": import("./DoorLockCC").DoorLockCCAPI;
    "Firmware Update Meta Data": import("./FirmwareUpdateMetaDataCC").FirmwareUpdateMetaDataCCAPI;
    Indicator: import("./IndicatorCC").IndicatorCCAPI;
    Language: import("./LanguageCC").LanguageCCAPI;
    Lock: import("./LockCC").LockCCAPI;
    "Manufacturer Proprietary": import("./ManufacturerProprietaryCC").ManufacturerProprietaryCCAPI;
    "Manufacturer Specific": import("./ManufacturerSpecificCC").ManufacturerSpecificCCAPI;
    Meter: import("./MeterCC").MeterCCAPI;
    "Multi Channel Association": import("./MultiChannelAssociationCC").MultiChannelAssociationCCAPI;
    "Multi Channel": import("./MultiChannelCC").MultiChannelCCAPI;
    "Multi Command": import("./MultiCommandCC").MultiCommandCCAPI;
    "Multilevel Sensor": import("./MultilevelSensorCC").MultilevelSensorCCAPI;
    "Multilevel Switch": import("./MultilevelSwitchCC").MultilevelSwitchCCAPI;
    "No Operation": import("./NoOperationCC").NoOperationCCAPI;
    "Node Naming and Location": import("./NodeNamingCC").NodeNamingAndLocationCCAPI;
    Notification: import("./NotificationCC").NotificationCCAPI;
    Protection: import("./ProtectionCC").ProtectionCCAPI;
    "Scene Activation": import("./SceneActivationCC").SceneActivationCCAPI;
    Security: import("./SecurityCC").SecurityCCAPI;
    "Sound Switch": import("./SoundSwitchCC").SoundSwitchCCAPI;
    Supervision: import("./SupervisionCC").SupervisionCCAPI;
    "Thermostat Mode": import("./ThermostatModeCC").ThermostatModeCCAPI;
    "Thermostat Operating State": import("./ThermostatOperatingStateCC").ThermostatOperatingStateCCAPI;
    "Thermostat Setback": import("./ThermostatSetbackCC").ThermostatSetbackCCAPI;
    "Thermostat Setpoint": import("./ThermostatSetpointCC").ThermostatSetpointCCAPI;
    Time: import("./TimeCC").TimeCCAPI;
    "Time Parameters": import("./TimeParametersCC").TimeParametersCCAPI;
    "User Code": import("./UserCodeCC").UserCodeCCAPI;
    Version: import("./VersionCC").VersionCCAPI;
    "Wake Up": import("./WakeUpCC").WakeUpCCAPI;
    "Z-Wave Plus Info": import("./ZWavePlusCC").ZWavePlusCCAPI;
}
