/// <reference types="node" />
import type { Maybe, ValueID } from "@zwave-js/core";
import { CommandClasses, MessageOrCCLogEntry } from "@zwave-js/core";
import type { Driver } from "../driver/Driver";
import type { ZWaveNode } from "../node/Node";
import { PhysicalCCAPI } from "./API";
import { CCCommandOptions, CommandClass, CommandClassDeserializationOptions, CommandClassOptions } from "./CommandClass";
import type { Association } from "./MultiChannelAssociationCC";
/** Returns the ValueID used to store the maximum number of nodes of an association group */
export declare function getMaxNodesValueId(groupId: number): ValueID;
/** Returns the ValueID used to store the node IDs of an association group */
export declare function getNodeIdsValueId(groupId: number): ValueID;
/** Returns the ValueID used to store the group count of an association group */
export declare function getGroupCountValueId(): ValueID;
/** Returns the ValueID used to store whether a node has a lifeline association */
export declare function getHasLifelineValueId(): ValueID;
export declare function getLifelineGroupIds(node: ZWaveNode): number[];
export declare enum AssociationCommand {
    Set = 1,
    Get = 2,
    Report = 3,
    Remove = 4,
    SupportedGroupingsGet = 5,
    SupportedGroupingsReport = 6
}
export declare class AssociationCCAPI extends PhysicalCCAPI {
    supportsCommand(cmd: AssociationCommand): Maybe<boolean>;
    /**
     * Returns the number of association groups a node supports.
     * Association groups are consecutive, starting at 1.
     */
    getGroupCount(): Promise<number | undefined>;
    /**
     * Returns information about an association group.
     */
    getGroup(groupId: number): Promise<{
        maxNodes: number;
        nodeIds: readonly number[];
    } | undefined>;
    /**
     * Adds new nodes to an association group
     */
    addNodeIds(groupId: number, ...nodeIds: number[]): Promise<void>;
    /**
     * Removes nodes from an association group
     */
    removeNodeIds(options: AssociationCCRemoveOptions): Promise<void>;
    /**
     * Removes nodes from all association groups
     */
    removeNodeIdsFromAllGroups(nodeIds: number[]): Promise<void>;
}
export declare class AssociationCC extends CommandClass {
    ccCommand: AssociationCommand;
    constructor(driver: Driver, options: CommandClassOptions);
    determineRequiredCCInterviews(): readonly CommandClasses[];
    skipEndpointInterview(): boolean;
    /**
     * Returns the number of association groups reported by the node.
     * This only works AFTER the interview process
     */
    getGroupCountCached(): number;
    /**
     * Returns the number of nodes an association group supports.
     * This only works AFTER the interview process
     */
    getMaxNodesCached(groupId: number): number;
    /**
     * Returns all the destinations of all association groups reported by the node.
     * This only works AFTER the interview process
     */
    getAllDestinationsCached(): ReadonlyMap<number, readonly Association[]>;
    interview(complete?: boolean): Promise<void>;
}
interface AssociationCCSetOptions extends CCCommandOptions {
    groupId: number;
    nodeIds: number[];
}
export declare class AssociationCCSet extends AssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | AssociationCCSetOptions);
    groupId: number;
    nodeIds: number[];
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
interface AssociationCCRemoveOptions {
    /** The group from which to remove the nodes. If none is specified, the nodes will be removed from all nodes. */
    groupId?: number;
    /** The nodes to remove. If none are specified, ALL nodes will be removed. */
    nodeIds?: number[];
}
export declare class AssociationCCRemove extends AssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | (AssociationCCRemoveOptions & CCCommandOptions));
    groupId?: number;
    nodeIds?: number[];
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class AssociationCCReport extends AssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _groupId;
    get groupId(): number;
    private _maxNodes;
    get maxNodes(): number;
    private _nodeIds;
    get nodeIds(): readonly number[];
    private _reportsToFollow;
    get reportsToFollow(): number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    mergePartialCCs(partials: AssociationCCReport[]): void;
    toLogEntry(): MessageOrCCLogEntry;
}
interface AssociationCCGetOptions extends CCCommandOptions {
    groupId: number;
}
export declare class AssociationCCGet extends AssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | AssociationCCGetOptions);
    groupId: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class AssociationCCSupportedGroupingsReport extends AssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _groupCount;
    get groupCount(): number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class AssociationCCSupportedGroupingsGet extends AssociationCC {
}
export {};
