/// <reference types="node" />
import type { ValueID } from "@zwave-js/core";
import { CommandClasses, Maybe, MessageOrCCLogEntry } from "@zwave-js/core";
import type { Driver } from "../driver/Driver";
import { PhysicalCCAPI } from "./API";
import { CCCommandOptions, CommandClass, CommandClassDeserializationOptions, CommandClassOptions } from "./CommandClass";
/**
 * @publicAPI
 */
export interface Association {
    nodeId: number;
    endpoint?: number;
}
/** Returns the ValueID used to store the maximum number of nodes of an association group */
export declare function getMaxNodesValueId(groupId: number): ValueID;
/** Returns the ValueID used to store the node IDs of a multi channel association group */
export declare function getNodeIdsValueId(groupId: number): ValueID;
/** Returns the ValueID used to store the endpoint addresses of a multi channel association group */
export declare function getEndpointsValueId(groupId: number): ValueID;
/** Returns the ValueID used to store the number of multi channel association group */
export declare function getGroupCountValueId(): ValueID;
export interface EndpointAddress {
    nodeId: number;
    endpoint: number | number[];
}
export declare enum MultiChannelAssociationCommand {
    Set = 1,
    Get = 2,
    Report = 3,
    Remove = 4,
    SupportedGroupingsGet = 5,
    SupportedGroupingsReport = 6
}
export declare class MultiChannelAssociationCCAPI extends PhysicalCCAPI {
    supportsCommand(cmd: MultiChannelAssociationCommand): Maybe<boolean>;
    /**
     * Returns the number of association groups a node supports.
     * Association groups are consecutive, starting at 1.
     */
    getGroupCount(): Promise<number | undefined>;
    /**
     * Returns information about an association group.
     */
    getGroup(groupId: number): Promise<Pick<MultiChannelAssociationCCReport, "maxNodes" | "nodeIds" | "endpoints"> | undefined>;
    /**
     * Adds new nodes or endpoints to an association group
     */
    addDestinations(options: MultiChannelAssociationCCSetOptions): Promise<void>;
    /**
     * Removes nodes or endpoints from an association group
     */
    removeDestinations(options: MultiChannelAssociationCCRemoveOptions): Promise<void>;
}
export declare class MultiChannelAssociationCC extends CommandClass {
    ccCommand: MultiChannelAssociationCommand;
    constructor(driver: Driver, options: CommandClassOptions);
    determineRequiredCCInterviews(): readonly CommandClasses[];
    skipEndpointInterview(): boolean;
    /**
     * Returns the number of association groups reported by the node.
     * This only works AFTER the interview process
     */
    getGroupCountCached(): number;
    /**
     * Returns the number of nodes an association group supports.
     * This only works AFTER the interview process
     */
    getMaxNodesCached(groupId: number): number;
    /**
     * Returns all the destinations of all association groups reported by the node.
     * This only works AFTER the interview process
     */
    getAllDestinationsCached(): ReadonlyMap<number, readonly Association[]>;
    interview(complete?: boolean): Promise<void>;
}
declare type MultiChannelAssociationCCSetOptions = {
    groupId: number;
} & ({
    nodeIds: number[];
} | {
    endpoints: EndpointAddress[];
} | {
    nodeIds: number[];
    endpoints: EndpointAddress[];
});
export declare class MultiChannelAssociationCCSet extends MultiChannelAssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | (MultiChannelAssociationCCSetOptions & CCCommandOptions));
    groupId: number;
    nodeIds: number[];
    endpoints: EndpointAddress[];
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
interface MultiChannelAssociationCCRemoveOptions {
    /** The group from which to remove the nodes. If none is specified, the nodes will be removed from all groups. */
    groupId?: number;
    /** The nodes to remove. If no nodeIds and no endpoint addresses are specified, ALL nodes will be removed. */
    nodeIds?: number[];
    /** The single endpoints to remove. If no nodeIds and no endpoint addresses are specified, ALL will be removed. */
    endpoints?: EndpointAddress[];
}
export declare class MultiChannelAssociationCCRemove extends MultiChannelAssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | (MultiChannelAssociationCCRemoveOptions & CCCommandOptions));
    groupId?: number;
    nodeIds?: number[];
    endpoints?: EndpointAddress[];
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class MultiChannelAssociationCCReport extends MultiChannelAssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _groupId;
    get groupId(): number;
    private _maxNodes;
    get maxNodes(): number;
    private _nodeIds;
    get nodeIds(): readonly number[];
    private _endpoints;
    get endpoints(): readonly EndpointAddress[];
    private _reportsToFollow;
    get reportsToFollow(): number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    mergePartialCCs(partials: MultiChannelAssociationCCReport[]): void;
    toLogEntry(): MessageOrCCLogEntry;
}
interface MultiChannelAssociationCCGetOptions extends CCCommandOptions {
    groupId: number;
}
export declare class MultiChannelAssociationCCGet extends MultiChannelAssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | MultiChannelAssociationCCGetOptions);
    groupId: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class MultiChannelAssociationCCSupportedGroupingsReport extends MultiChannelAssociationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _groupCount;
    get groupCount(): number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class MultiChannelAssociationCCSupportedGroupingsGet extends MultiChannelAssociationCC {
}
export {};
