/// <reference types="node" />
import { CommandClasses, Maybe, MessageOrCCLogEntry, SecurityManager } from "@zwave-js/core";
import type { ZWaveController } from "../controller/Controller";
import type { Driver } from "../driver/Driver";
import { PhysicalCCAPI } from "./API";
import { CCCommandOptions, CommandClass, CommandClassDeserializationOptions } from "./CommandClass";
export declare enum SecurityCommand {
    CommandsSupportedGet = 2,
    CommandsSupportedReport = 3,
    SchemeGet = 4,
    SchemeReport = 5,
    SchemeInherit = 8,
    NetworkKeySet = 6,
    NetworkKeyVerify = 7,
    NonceGet = 64,
    NonceReport = 128,
    CommandEncapsulation = 129,
    CommandEncapsulationNonceGet = 193
}
export declare class SecurityCCAPI extends PhysicalCCAPI {
    supportsCommand(_cmd: SecurityCommand): Maybe<boolean>;
    sendEncapsulated(encapsulated: CommandClass, requestNextNonce?: boolean): Promise<void>;
    /**
     * Requests a new nonce for Security CC encapsulation
     */
    getNonce(options?: {
        /** Whether the command should be sent as a standalone transaction. Default: false */
        standalone?: boolean;
        /** Whether the received nonce should be stored as "free". Default: false */
        storeAsFreeNonce?: boolean;
    }): Promise<Buffer | undefined>;
    /**
     * Responds to a NonceGet request. The message is sent without any retransmission etc.
     * The return value indicates whether a nonce was successfully sent
     */
    sendNonce(): Promise<boolean>;
    getSecurityScheme(): Promise<[0]>;
    inheritSecurityScheme(): Promise<void>;
    setNetworkKey(networkKey: Buffer): Promise<void>;
    getSupportedCommands(): Promise<Pick<SecurityCCCommandsSupportedReport, "supportedCCs" | "controlledCCs"> | undefined>;
}
export declare class SecurityCC extends CommandClass {
    ccCommand: SecurityCommand;
    nodeId: number;
    driver: Driver & {
        securityManager: SecurityManager;
        controller: ZWaveController & {
            ownNodeId: number;
        };
    };
    interview(complete?: boolean): Promise<void>;
    /** Tests if a should be sent secure and thus requires encapsulation */
    static requiresEncapsulation(cc: CommandClass): boolean;
    /** Encapsulates a command that should be sent encrypted */
    static encapsulate(driver: Driver, cc: CommandClass): SecurityCCCommandEncapsulation;
}
interface SecurityCCNonceReportOptions extends CCCommandOptions {
    nonce: Buffer;
}
export declare class SecurityCCNonceReport extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | SecurityCCNonceReportOptions);
    nonce: Buffer;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCNonceGet extends SecurityCC {
}
interface SecurityCCCommandEncapsulationOptions extends CCCommandOptions {
    encapsulated: CommandClass;
    alternativeNetworkKey?: Buffer;
}
export declare class SecurityCCCommandEncapsulation extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | SecurityCCCommandEncapsulationOptions);
    private sequenced;
    private secondFrame;
    private sequenceCounter;
    private decryptedCCBytes;
    encapsulated: CommandClass;
    private authKey;
    private encryptionKey;
    nonceId: number | undefined;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    mergePartialCCs(partials: SecurityCCCommandEncapsulation[]): void;
    requiresPreTransmitHandshake(): boolean;
    preTransmitHandshake(): Promise<void>;
    serialize(): Buffer;
    protected computeEncapsulationOverhead(): number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCCommandEncapsulationNonceGet extends SecurityCCCommandEncapsulation {
}
export declare class SecurityCCSchemeReport extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
}
export declare class SecurityCCSchemeGet extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | CCCommandOptions);
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCSchemeInherit extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | CCCommandOptions);
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCNetworkKeyVerify extends SecurityCC {
}
interface SecurityCCNetworkKeySetOptions extends CCCommandOptions {
    networkKey: Buffer;
}
export declare class SecurityCCNetworkKeySet extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | SecurityCCNetworkKeySetOptions);
    networkKey: Buffer;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCCommandsSupportedReport extends SecurityCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    readonly reportsToFollow: number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    private _supportedCCs;
    get supportedCCs(): CommandClasses[];
    private _controlledCCs;
    get controlledCCs(): CommandClasses[];
    mergePartialCCs(partials: SecurityCCCommandsSupportedReport[]): void;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SecurityCCCommandsSupportedGet extends SecurityCC {
}
export {};
