/// <reference types="node" />
import { EventEmitter } from "events";
import type { AssociationGroup } from "../commandclass/AssociationGroupInfoCC";
import type { Association } from "../commandclass/MultiChannelAssociationCC";
import { FunctionType } from "../message/Constants";
import { ZWaveNode } from "../node/Node";
import { VirtualNode } from "../node/VirtualNode";
import { ZWaveLibraryTypes } from "./ZWaveLibraryTypes";
export declare type HealNodeStatus = "pending" | "done" | "failed" | "skipped";
export declare type ThrowingMap<K, V> = Map<K, V> & {
    getOrThrow(key: K): V;
};
export declare type ReadonlyThrowingMap<K, V> = ReadonlyMap<K, V> & {
    getOrThrow(key: K): V;
};
interface ControllerEventCallbacks {
    "inclusion failed": () => void;
    "exclusion failed": () => void;
    "inclusion started": (secure: boolean) => void;
    "exclusion started": () => void;
    "inclusion stopped": () => void;
    "exclusion stopped": () => void;
    "node added": (node: ZWaveNode) => void;
    "node removed": (node: ZWaveNode) => void;
    "heal network progress": (progress: ReadonlyMap<number, HealNodeStatus>) => void;
    "heal network done": (result: ReadonlyMap<number, HealNodeStatus>) => void;
}
export declare type ControllerEvents = Extract<keyof ControllerEventCallbacks, string>;
export interface ZWaveController {
    on<TEvent extends ControllerEvents>(event: TEvent, callback: ControllerEventCallbacks[TEvent]): this;
    once<TEvent extends ControllerEvents>(event: TEvent, callback: ControllerEventCallbacks[TEvent]): this;
    removeListener<TEvent extends ControllerEvents>(event: TEvent, callback: ControllerEventCallbacks[TEvent]): this;
    off<TEvent extends ControllerEvents>(event: TEvent, callback: ControllerEventCallbacks[TEvent]): this;
    removeAllListeners(event?: ControllerEvents): this;
    emit<TEvent extends ControllerEvents>(event: TEvent, ...args: Parameters<ControllerEventCallbacks[TEvent]>): boolean;
}
export declare class ZWaveController extends EventEmitter {
    private readonly driver;
    private _libraryVersion;
    get libraryVersion(): string | undefined;
    private _type;
    get type(): ZWaveLibraryTypes | undefined;
    private _homeId;
    /** A 32bit number identifying the current network */
    get homeId(): number | undefined;
    private _ownNodeId;
    /** The ID of the controller in the current network */
    get ownNodeId(): number | undefined;
    private _isSecondary;
    get isSecondary(): boolean | undefined;
    private _isUsingHomeIdFromOtherNetwork;
    get isUsingHomeIdFromOtherNetwork(): boolean | undefined;
    private _isSISPresent;
    get isSISPresent(): boolean | undefined;
    private _wasRealPrimary;
    get wasRealPrimary(): boolean | undefined;
    private _isStaticUpdateController;
    get isStaticUpdateController(): boolean | undefined;
    private _isSlave;
    get isSlave(): boolean | undefined;
    private _serialApiVersion;
    get serialApiVersion(): string | undefined;
    private _manufacturerId;
    get manufacturerId(): number | undefined;
    private _productType;
    get productType(): number | undefined;
    private _productId;
    get productId(): number | undefined;
    private _supportedFunctionTypes;
    get supportedFunctionTypes(): readonly FunctionType[] | undefined;
    /** Checks if a given Z-Wave function type is supported by this controller */
    isFunctionSupported(functionType: FunctionType): boolean;
    private _sucNodeId;
    get sucNodeId(): number | undefined;
    private _supportsTimers;
    get supportsTimers(): boolean | undefined;
    private _nodes;
    /** A dictionary of the nodes connected to this controller */
    get nodes(): ReadonlyThrowingMap<number, ZWaveNode>;
    /** Returns a reference to the (virtual) broadcast node, which allows sending commands to all nodes */
    getBroadcastNode(): VirtualNode;
    /** Creates a virtual node that can be used to send multicast commands to several nodes */
    getMulticastGroup(nodeIDs: number[]): VirtualNode;
    private createValueDBForNode;
    private _exclusionActive;
    private _inclusionActive;
    private _includeNonSecure;
    private _includeController;
    private _nodePendingInclusion;
    private _nodePendingExclusion;
    private _nodePendingReplace;
    private _beginInclusionPromise;
    private _stopInclusionPromise;
    private _replaceFailedPromise;
    /**
     * Starts the inclusion process of new nodes.
     * Resolves to true when the process was started, and false if the inclusion was already active.
     *
     * @param includeNonSecure Whether the node should be included non-securely, even if it supports Security. By default, all nodes will be included securely if possible
     */
    beginInclusion(includeNonSecure?: boolean): Promise<boolean>;
    /** Is used internally to stop an active inclusion process without creating deadlocks */
    private stopInclusionInternal;
    /**
     * Stops an active inclusion process. Resolves to true when the controller leaves inclusion mode,
     * and false if the inclusion was not active.
     */
    stopInclusion(): Promise<boolean>;
    /**
     * Starts the exclusion process of new nodes.
     * Resolves to true when the process was started,
     * and false if an inclusion or exclusion process was already active
     */
    beginExclusion(): Promise<boolean>;
    /** Is used internally to stop an active inclusion process without creating deadlocks */
    private stopExclusionInternal;
    private secureBootstrapS0;
    /** Ensures that the node knows where to reach the controller */
    private bootstrapLifelineAndWakeup;
    /**
     * Stops an active exclusion process. Resolves to true when the controller leaves exclusion mode,
     * and false if the inclusion was not active.
     */
    stopExclusion(): Promise<boolean>;
    /**
     * Is called when an AddNode request is received from the controller.
     * Handles and controls the inclusion process.
     */
    private handleAddNodeRequest;
    /**
     * Is called when an ReplaceFailed request is received from the controller.
     * Handles and controls the replace process.
     */
    private handleReplaceNodeRequest;
    /**
     * Is called when a RemoveNode request is received from the controller.
     * Handles and controls the exclusion process.
     */
    private handleRemoveNodeRequest;
    private _healNetworkActive;
    private _healNetworkProgress;
    /**
     * Requests all alive slave nodes to update their neighbor lists
     */
    beginHealingNetwork(): boolean;
    /**
     * Stops an network healing process. Resolves false if the process was not active, true otherwise.
     */
    stopHealingNetwork(): boolean;
    /**
     * Performs the healing process for a node
     */
    healNode(nodeId: number): Promise<boolean>;
    assignReturnRoute(nodeId: number, destinationNodeId: number): Promise<void>;
    /**
     * Returns a dictionary of all association groups of this node and their information.
     * This only works AFTER the interview process
     */
    getAssociationGroups(nodeId: number): ReadonlyMap<number, AssociationGroup>;
    /** Returns all Associations (Multi Channel or normal) that are configured on a node */
    getAssociations(nodeId: number): ReadonlyMap<number, readonly Association[]>;
    /** Checks if a given association is allowed */
    isAssociationAllowed(nodeId: number, group: number, association: Association): boolean;
    /**
     * Adds associations to a node
     */
    addAssociations(nodeId: number, group: number, associations: Association[]): Promise<void>;
    /**
     * Removes the specific associations from a node
     */
    removeAssociations(nodeId: number, group: number, associations: Association[]): Promise<void>;
    /**
     * Removes a node from all other nodes' associations
     * WARNING: It is not recommended to await this method
     */
    removeNodeFromAllAssocations(nodeId: number): Promise<void>;
    /**
     * Tests if a node is marked as failed in the controller's memory
     * @param nodeId The id of the node in question
     */
    isFailedNode(nodeId: number): Promise<boolean>;
    /**
     * Removes a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to remove
     */
    removeFailedNode(nodeId: number): Promise<void>;
    /**
     * Replace a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to replace
     * @param includeNonSecure Whether the new node should be included non-securely, even if it supports Security. By default, all nodes will be included securely if possible
     *
     */
    replaceFailedNode(nodeId: number, includeNonSecure?: boolean): Promise<boolean>;
}
export {};
