"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transaction = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const comparable_1 = require("alcalzone-shared/comparable");
const Constants_1 = require("../message/Constants");
/**
 * Transactions are used to track and correllate messages with their responses.
 */
class Transaction {
    constructor(driver, message, promise, priority) {
        this.driver = driver;
        this.message = message;
        this.promise = promise;
        this.priority = priority;
        /** The timestamp at which the transaction was created */
        this.creationTimestamp = core_1.highResTimestamp();
        /** Whether the node status should be updated when this transaction times out */
        this.changeNodeStatusOnTimeout = true;
        // We need create the stack on a temporary object or the Error
        // class will try to print the message
        const tmp = { message: "" };
        Error.captureStackTrace(tmp, Transaction);
        this.stack = tmp.stack;
        if (this.stack.startsWith("Error")) {
            this.stack = "Transaction" + this.stack.substr(5);
        }
    }
    /** Compares two transactions in order to plan their transmission sequence */
    compareTo(other) {
        function compareWakeUpPriority(_this, _other) {
            const thisNode = _this.message.getNodeUnsafe();
            const otherNode = _other.message.getNodeUnsafe();
            // We don't require existence of the node object
            // If any transaction is not for a node, it targets the controller
            // which is always awake
            const thisIsAsleep = (thisNode === null || thisNode === void 0 ? void 0 : thisNode.isAwake()) === false;
            const otherIsAsleep = (otherNode === null || otherNode === void 0 ? void 0 : otherNode.isAwake()) === false;
            // If both nodes are asleep, the conventional order applies
            // Asleep nodes always have the lowest priority
            if (thisIsAsleep && !otherIsAsleep)
                return 1;
            if (otherIsAsleep && !thisIsAsleep)
                return -1;
        }
        // delay messages for sleeping nodes
        if (this.priority === Constants_1.MessagePriority.WakeUp) {
            const result = compareWakeUpPriority(this, other);
            if (result != undefined)
                return result;
        }
        else if (other.priority === Constants_1.MessagePriority.WakeUp) {
            const result = compareWakeUpPriority(other, this);
            if (result != undefined)
                return -result;
        }
        function compareNodeQueryPriority(_this, _other) {
            const thisNode = _this.message.getNodeUnsafe();
            const otherNode = _other.message.getNodeUnsafe();
            if (thisNode && otherNode) {
                // Both nodes exist
                const thisListening = thisNode.isListening || thisNode.isFrequentListening;
                const otherListening = otherNode.isListening || otherNode.isFrequentListening;
                // prioritize (-1) the one node that is listening when the other is not
                if (thisListening && !otherListening)
                    return -1;
                if (!thisListening && otherListening)
                    return 1;
            }
        }
        // delay NodeQuery messages for non-listening nodes
        if (this.priority === Constants_1.MessagePriority.NodeQuery) {
            const result = compareNodeQueryPriority(this, other);
            if (result != undefined)
                return result;
        }
        else if (other.priority === Constants_1.MessagePriority.NodeQuery) {
            const result = compareNodeQueryPriority(other, this);
            if (result != undefined)
                return -result;
        }
        // by default, sort by priority
        if (this.priority < other.priority)
            return -1;
        else if (this.priority > other.priority)
            return 1;
        // for equal priority, sort by the timestamp
        return comparable_1.compareNumberOrString(other.creationTimestamp, this.creationTimestamp);
    }
    toJSON() {
        return shared_1.pick(this, [
            "creationTimestamp",
            "changeNodeStatusOnTimeout",
            "tag",
            "message",
            "priority",
            "stack",
        ]);
    }
}
exports.Transaction = Transaction;

//# sourceMappingURL=Transaction.js.map
