"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.interpretEx = exports.createWrapperMachine = exports.respondUnsolicited = exports.isSerialCommandError = exports.sendDataErrorToZWaveError = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const xstate_1 = require("xstate");
const actions_1 = require("xstate/lib/actions");
const SendDataMessages_1 = require("../controller/SendDataMessages");
function sendDataErrorToZWaveError(error, transaction, receivedMessage) {
    switch (error) {
        case "send failure":
        case "CAN":
        case "NAK":
            return new core_1.ZWaveError(`Failed to send the message after 3 attempts`, core_1.ZWaveErrorCodes.Controller_MessageDropped, undefined, transaction.stack);
        case "ACK timeout":
            return new core_1.ZWaveError(`Timeout while waiting for an ACK from the controller`, core_1.ZWaveErrorCodes.Controller_Timeout, undefined, transaction.stack);
        case "response timeout":
            return new core_1.ZWaveError(`Timeout while waiting for a response from the controller`, core_1.ZWaveErrorCodes.Controller_Timeout, undefined, transaction.stack);
        case "callback timeout":
            return new core_1.ZWaveError(`Timeout while waiting for a callback from the controller`, core_1.ZWaveErrorCodes.Controller_Timeout, undefined, transaction.stack);
        case "response NOK":
            if (transaction.message instanceof SendDataMessages_1.SendDataRequest ||
                transaction.message instanceof SendDataMessages_1.SendDataMulticastRequest) {
                return new core_1.ZWaveError(`Failed to send the command after ${transaction.message.maxSendAttempts} attempts. Transmission queue full`, core_1.ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transaction.stack);
            }
            else {
                return new core_1.ZWaveError(`The controller response indicated failure`, core_1.ZWaveErrorCodes.Controller_ResponseNOK, receivedMessage, transaction.stack);
            }
        case "callback NOK":
            if (transaction.message instanceof SendDataMessages_1.SendDataRequest) {
                const status = receivedMessage
                    .transmitStatus;
                return new core_1.ZWaveError(`Failed to send the command after ${transaction.message.maxSendAttempts} attempts (Status ${shared_1.getEnumMemberName(SendDataMessages_1.TransmitStatus, status)})`, status === SendDataMessages_1.TransmitStatus.NoAck
                    ? core_1.ZWaveErrorCodes.Controller_CallbackNOK
                    : core_1.ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transaction.stack);
            }
            else if (transaction.message instanceof SendDataMessages_1.SendDataMulticastRequest) {
                const status = receivedMessage
                    .transmitStatus;
                return new core_1.ZWaveError(`One or more nodes did not respond to the multicast request (Status ${shared_1.getEnumMemberName(SendDataMessages_1.TransmitStatus, status)})`, status === SendDataMessages_1.TransmitStatus.NoAck
                    ? core_1.ZWaveErrorCodes.Controller_CallbackNOK
                    : core_1.ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transaction.stack);
            }
            else {
                return new core_1.ZWaveError(`The controller callback indicated failure`, core_1.ZWaveErrorCodes.Controller_CallbackNOK, receivedMessage, transaction.stack);
            }
        case "node timeout":
            return new core_1.ZWaveError(`Timed out while waiting for a response from the node`, core_1.ZWaveErrorCodes.Controller_NodeTimeout, undefined, transaction.stack);
    }
}
exports.sendDataErrorToZWaveError = sendDataErrorToZWaveError;
/** Tests whether the given error is one that was caused by the serial API execution */
function isSerialCommandError(error) {
    if (!(error instanceof core_1.ZWaveError))
        return false;
    switch (error.code) {
        case core_1.ZWaveErrorCodes.Controller_Timeout:
        case core_1.ZWaveErrorCodes.Controller_ResponseNOK:
        case core_1.ZWaveErrorCodes.Controller_CallbackNOK:
        case core_1.ZWaveErrorCodes.Controller_MessageDropped:
            return true;
    }
    return false;
}
exports.isSerialCommandError = isSerialCommandError;
exports.respondUnsolicited = actions_1.respond((_, evt) => ({
    type: "unsolicited",
    message: evt.message,
}));
/** Creates an auto-forwarding wrapper state machine that can be used to test machines that use sendParent */
function createWrapperMachine(testMachine) {
    return xstate_1.Machine({
        context: {
            child: undefined,
        },
        initial: "main",
        states: {
            main: {
                entry: xstate_1.assign({
                    child: () => xstate_1.spawn(testMachine, {
                        name: "child",
                        autoForward: true,
                    }),
                }),
            },
        },
    });
}
exports.createWrapperMachine = createWrapperMachine;
/** Extends the default xstate interpreter with a restart function that re-attaches all event handlers */
function interpretEx(machine, options) {
    const interpreter = new xstate_1.Interpreter(machine, options);
    return new Proxy(interpreter, {
        get(target, key) {
            if (key === "restart") {
                return () => {
                    const listeners = [...target["listeners"]];
                    const contextListeners = [
                        ...target["contextListeners"],
                    ];
                    const stopListeners = [
                        ...target["stopListeners"],
                    ];
                    const doneListeners = [
                        ...target["doneListeners"],
                    ];
                    const eventListeners = [
                        ...target["eventListeners"],
                    ];
                    const sendListeners = [
                        ...target["sendListeners"],
                    ];
                    target.stop();
                    for (const listener of listeners)
                        target.onTransition(listener);
                    for (const listener of contextListeners)
                        target.onChange(listener);
                    for (const listener of stopListeners)
                        target.onStop(listener);
                    for (const listener of doneListeners)
                        target.onDone(listener);
                    for (const listener of eventListeners)
                        target.onEvent(listener);
                    for (const listener of sendListeners)
                        target.onSend(listener);
                    return target.start();
                };
            }
            else {
                return target[key];
            }
        },
    });
}
exports.interpretEx = interpretEx;

//# sourceMappingURL=StateMachineShared.js.map
