"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultMeterScale = exports.MeterScale = exports.Meter = exports.loadMetersInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "meters.json");
/** @internal */
async function loadMetersInternal() {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("meters", "the database is not an object");
        }
        const meters = new Map();
        for (const [id, meterDefinition] of objects_1.entries(definition)) {
            if (!utils_1.hexKeyRegexNDigits.test(id)) {
                utils_1.throwInvalidConfig("meters", `found non-hex key "${id}" at the root`);
            }
            const idNum = parseInt(id.slice(2), 16);
            meters.set(idNum, new Meter(idNum, meterDefinition));
        }
        return meters;
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("meters");
        }
    }
}
exports.loadMetersInternal = loadMetersInternal;
class Meter {
    constructor(id, definition) {
        this.id = id;
        this.name = definition.name;
        const scales = new Map();
        if (typeguards_1.isObject(definition.scales)) {
            for (const [scaleId, scaleDefinition] of objects_1.entries(definition.scales)) {
                if (!utils_1.hexKeyRegexNDigits.test(scaleId)) {
                    utils_1.throwInvalidConfig("meters", `found non-hex key "${scaleId}" in meter ${shared_1.num2hex(id)}`);
                }
                if (typeof scaleDefinition !== "string") {
                    utils_1.throwInvalidConfig("meters", `The scale definition for "${scaleId}" in meter ${shared_1.num2hex(id)} is not a string!`);
                }
                const scaleIdNum = parseInt(scaleId.slice(2), 16);
                scales.set(scaleIdNum, new MeterScale(scaleIdNum, scaleDefinition));
            }
        }
        this.scales = scales;
    }
}
exports.Meter = Meter;
class MeterScale {
    constructor(key, definition) {
        this.key = key;
        this.label = definition;
    }
}
exports.MeterScale = MeterScale;
function getDefaultMeterScale(scale) {
    return new MeterScale(scale, `Unknown (${shared_1.num2hex(scale)})`);
}
exports.getDefaultMeterScale = getDefaultMeterScale;
//# sourceMappingURL=Meters.js.map