"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.restoreSilence = exports.unsilence = exports.tagify = exports.messageRecordToLines = exports.messageToLines = exports.messageFitsIntoOneLine = exports.shouldLogNode = exports.ZWaveLogContainer = exports.ZWaveLoggerBase = exports.getNodeTag = exports.LOG_PREFIX_WIDTH = exports.LOG_WIDTH = exports.directionPrefixPadding = exports.CONTROL_CHAR_WIDTH = exports.getDirectionPrefix = exports.timestampFormat = void 0;
const shared_1 = require("@zwave-js/shared");
const strings_1 = require("alcalzone-shared/strings");
const path = __importStar(require("path"));
const triple_beam_1 = require("triple-beam");
const winston_1 = __importDefault(require("winston"));
const Colorizer_1 = require("./Colorizer");
const { combine, timestamp, label } = winston_1.default.format;
const loglevels = triple_beam_1.configs.npm.levels;
const isTTY = process.stdout.isTTY;
const isUnitTest = process.env.NODE_ENV === "test";
exports.timestampFormat = "HH:mm:ss.SSS";
const timestampPadding = " ".repeat(exports.timestampFormat.length + 1);
const channelPadding = " ".repeat(7); // 6 chars channel name, 1 space
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
function getDirectionPrefix(direction) {
    return direction === "inbound"
        ? "« "
        : direction === "outbound"
            ? "» "
            : "  ";
}
exports.getDirectionPrefix = getDirectionPrefix;
/** The space the directional arrows, grouping brackets and padding occupies */
exports.CONTROL_CHAR_WIDTH = 2;
exports.directionPrefixPadding = " ".repeat(exports.CONTROL_CHAR_WIDTH);
/**
 * The width of a log line in (visible) characters, excluding the timestamp and
 * label, but including the direction prefix
 */
exports.LOG_WIDTH = 80;
/** The width of the columns containing the timestamp and channel */
exports.LOG_PREFIX_WIDTH = 20;
/** Returns the tag used to log node related messages */
function getNodeTag(nodeId) {
    return "Node " + strings_1.padStart(nodeId.toString(), 3, "0");
}
exports.getNodeTag = getNodeTag;
class ZWaveLoggerBase {
    constructor(loggers, logLabel) {
        this.container = loggers;
        this.logger = this.container.getLogger(logLabel);
    }
}
exports.ZWaveLoggerBase = ZWaveLoggerBase;
class ZWaveLogContainer extends winston_1.default.Container {
    constructor(config = {}) {
        super();
        this.loglevelVisibleCache = new Map();
        this.logConfig = {
            enabled: true,
            level: getTransportLoglevelNumeric(),
            logToFile: !!process.env.LOGTOFILE,
            transports: undefined,
            filename: require.main
                ? path.join(path.dirname(require.main.filename), `zwave-${process.pid}.log`)
                : path.join(__dirname, "../../..", `zwave-${process.pid}.log`),
            forceConsole: false,
        };
        /** Prints a formatted and colorized log message */
        this.logMessagePrinter = {
            transform: ((info) => {
                // The formatter has already split the message into multiple lines
                const messageLines = messageToLines(info.message);
                // Also this can only happen if the user forgot to call the formatter first
                if (info.secondaryTagPadding == undefined)
                    info.secondaryTagPadding = -1;
                // Format the first message line
                let firstLine = [
                    info.primaryTags,
                    messageLines[0],
                    info.secondaryTagPadding < 0
                        ? undefined
                        : " ".repeat(info.secondaryTagPadding),
                    // If the secondary tag padding is zero, the previous segment gets
                    // filtered out and we have one less space than necessary
                    info.secondaryTagPadding === 0 && info.secondaryTags
                        ? " " + info.secondaryTags
                        : info.secondaryTags,
                ]
                    .filter((item) => !!item)
                    .join(" ");
                // The directional arrows and the optional grouping lines must be prepended
                // without adding spaces
                firstLine = `${info.timestamp} ${info.label} ${info.direction}${firstLine}`;
                const lines = [firstLine];
                if (info.multiline) {
                    // Format all message lines but the first
                    lines.push(...messageLines.slice(1).map((line) => 
                    // Skip the columns for the timestamp and the channel name
                    timestampPadding +
                        channelPadding +
                        // Skip the columns for directional arrows
                        exports.directionPrefixPadding +
                        line));
                }
                info[triple_beam_1.MESSAGE] = lines.join("\n");
                return info;
            }),
        };
        /** Formats the log message and calculates the necessary paddings */
        this.logMessageFormatter = {
            transform: ((info) => {
                const messageLines = messageToLines(info.message);
                const firstMessageLineLength = messageLines[0].length;
                info.multiline =
                    messageLines.length > 1 ||
                        !messageFitsIntoOneLine(info, info.message.length);
                // Align postfixes to the right
                if (info.secondaryTags) {
                    // Calculate how many spaces are needed to right-align the postfix
                    // Subtract 1 because the parts are joined by spaces
                    info.secondaryTagPadding = Math.max(
                    // -1 has the special meaning that we don't print any padding,
                    // because the message takes all the available space
                    -1, exports.LOG_WIDTH -
                        1 -
                        calculateFirstLineLength(info, firstMessageLineLength));
                }
                if (info.multiline) {
                    // Break long messages into multiple lines
                    const lines = [];
                    let isFirstLine = true;
                    for (let message of messageLines) {
                        while (message.length) {
                            const cut = Math.min(message.length, isFirstLine
                                ? exports.LOG_WIDTH -
                                    calculateFirstLineLength(info, 0) -
                                    1
                                : exports.LOG_WIDTH - exports.CONTROL_CHAR_WIDTH);
                            isFirstLine = false;
                            lines.push(message.substr(0, cut));
                            message = message.substr(cut);
                        }
                    }
                    info.message = lines.join("\n");
                }
                return info;
            }),
        };
        this.updateConfiguration(config);
    }
    getLogger(label) {
        if (!this.has(label)) {
            this.add(label, {
                transports: this.getConfiguredTransports(),
                format: this.createLoggerFormat(label),
            });
        }
        return this.get(label);
    }
    updateConfiguration(config) {
        var _a;
        this.logConfig = Object.assign(this.logConfig, config);
        if (!((_a = this.logConfig.transports) === null || _a === void 0 ? void 0 : _a.length)) {
            this.logConfig.transports = this.createLogTransports();
        }
        for (const transport of this.logConfig.transports) {
            if (transport === this.consoleTransport) {
                transport.silent = this.isConsoleTransportSilent();
            }
            else if (transport === this.fileTransport) {
                transport.silent = this.isFileTransportSilent();
            }
            else {
                transport.silent = !this.logConfig.enabled;
            }
        }
    }
    getConfiguredTransports() {
        return this.logConfig.transports;
    }
    /** The common logger format for all channels */
    createLoggerFormat(channel) {
        // Only colorize the output if logging to a TTY, otherwise we'll get
        // ansi color codes in logfiles
        const colorize = !this.logConfig.logToFile && (isTTY || isUnitTest);
        const formats = [];
        formats.push(label({ label: channel }), timestamp({ format: exports.timestampFormat }), this.logMessageFormatter);
        if (colorize)
            formats.push(Colorizer_1.colorizer());
        formats.push(this.logMessagePrinter);
        return combine(...formats);
    }
    /** Tests whether a log using the given loglevel will be logged */
    isLoglevelVisible(loglevel) {
        // If we are not connected to a TTY, not unit testing and not logging to a file, we won't see anything
        if (isUnitTest)
            return true;
        if (!isTTY && !this.logConfig.logToFile && !this.logConfig.forceConsole)
            return false;
        if (!this.loglevelVisibleCache.has(loglevel)) {
            this.loglevelVisibleCache.set(loglevel, loglevel in loglevels &&
                loglevels[loglevel] <= this.logConfig.level);
        }
        return this.loglevelVisibleCache.get(loglevel);
    }
    destroy() {
        for (const key in this.loggers) {
            this.close(key);
        }
        this.fileTransport = undefined;
        this.consoleTransport = undefined;
        this.logConfig.transports = [];
    }
    createLogTransports() {
        const ret = [];
        if (this.logConfig.enabled && this.logConfig.logToFile) {
            if (!this.fileTransport) {
                console.log(`Logging to file:
	${this.logConfig.filename}`);
                this.fileTransport = this.createFileTransport();
            }
            ret.push(this.fileTransport);
        }
        else {
            if (!this.consoleTransport) {
                this.consoleTransport = this.createConsoleTransport();
            }
            ret.push(this.consoleTransport);
        }
        return ret;
    }
    createConsoleTransport() {
        return new winston_1.default.transports.Console({
            level: getTransportLoglevel(),
            silent: this.isConsoleTransportSilent(),
        });
    }
    isConsoleTransportSilent() {
        return process.env.NODE_ENV === "test" || !this.logConfig.enabled;
    }
    isFileTransportSilent() {
        return !this.logConfig.enabled;
    }
    createFileTransport() {
        return new winston_1.default.transports.File({
            filename: this.logConfig.filename,
            level: getTransportLoglevel(),
            silent: this.isFileTransportSilent(),
        });
    }
}
exports.ZWaveLogContainer = ZWaveLogContainer;
function getTransportLoglevel() {
    return process.env.LOGLEVEL in loglevels ? process.env.LOGLEVEL : "debug";
}
function getTransportLoglevelNumeric() {
    return loglevels[getTransportLoglevel()];
}
/**
 * Checks the LOG_NODES env variable whether logs should be written for a given node id
 */
function shouldLogNode(nodeId) {
    var _a;
    const activeFilters = ((_a = process.env.LOG_NODES) !== null && _a !== void 0 ? _a : "*").split(",");
    if (activeFilters.includes("*"))
        return true;
    if (activeFilters.includes(nodeId.toString()))
        return true;
    return false;
}
exports.shouldLogNode = shouldLogNode;
/**
 * Calculates the length the first line of a log message would occupy if it is not split
 * @param info The message and information to log
 * @param firstMessageLineLength The length of the first line of the actual message text, not including pre- and postfixes.
 */
function calculateFirstLineLength(info, firstMessageLineLength) {
    return ([
        exports.CONTROL_CHAR_WIDTH - 1,
        firstMessageLineLength,
        (info.primaryTags || "").length,
        (info.secondaryTags || "").length,
    ]
        // filter out empty parts
        .filter((len) => len > 0)
        // simulate adding spaces between parts
        .reduce((prev, val) => prev + (prev > 0 ? 1 : 0) + val));
}
/**
 * Tests if a given message fits into a single log line
 * @param info The message that should be logged
 * @param messageLength The length that should be assumed for the actual message without pre and postfixes.
 * Can be set to 0 to exclude the message from the calculation
 */
function messageFitsIntoOneLine(info, messageLength) {
    const totalLength = calculateFirstLineLength(info, messageLength);
    return totalLength <= exports.LOG_WIDTH;
}
exports.messageFitsIntoOneLine = messageFitsIntoOneLine;
function messageToLines(message) {
    if (typeof message === "string") {
        return message.split("\n");
    }
    else if (message.length > 0) {
        return message;
    }
    else {
        return [""];
    }
}
exports.messageToLines = messageToLines;
/** Splits a message record into multiple lines and auto-aligns key-value pairs */
function messageRecordToLines(message) {
    const entries = Object.entries(message);
    if (!entries.length)
        return [];
    const maxKeyLength = Math.max(...entries.map(([key]) => key.length));
    return shared_1.flatMap(entries, ([key, value]) => `${key}:${" ".repeat(Math.max(maxKeyLength - key.length + 1, 1))}${value}`
        .split("\n")
        .map((line) => line.trimRight()));
}
exports.messageRecordToLines = messageRecordToLines;
/** Wraps an array of strings in square brackets and joins them with spaces */
function tagify(tags) {
    return tags.map((pfx) => `[${pfx}]`).join(" ");
}
exports.tagify = tagify;
/** Unsilences the console transport of a logger and returns the original value */
function unsilence(logger) {
    const consoleTransport = logger.transports.find((t) => t.name === "console");
    if (consoleTransport) {
        const ret = !!consoleTransport.silent;
        consoleTransport.silent = false;
        return ret;
    }
    return false;
}
exports.unsilence = unsilence;
/** Restores the console transport of a logger to its original silence state */
function restoreSilence(logger, original) {
    const consoleTransport = logger.transports.find((t) => t.name === "console");
    if (consoleTransport) {
        consoleTransport.silent = original;
    }
}
exports.restoreSilence = restoreSilence;
//# sourceMappingURL=shared.js.map