/// <reference types="node" />
import type { JsonlDB } from "@alcalzone/jsonl-db";
import { EventEmitter } from "events";
import type { CommandClasses } from "../capabilities/CommandClasses";
import type { ValueMetadata } from "../values/Metadata";
/** Uniquely identifies to which CC, endpoint and property a value belongs to */
export interface ValueID {
    commandClass: CommandClasses;
    endpoint?: number;
    property: string | number;
    propertyKey?: string | number;
}
export interface TranslatedValueID extends ValueID {
    commandClassName: string;
    propertyName?: string;
    propertyKeyName?: string;
}
export interface ValueUpdatedArgs extends ValueID {
    prevValue: unknown;
    newValue: unknown;
}
export interface ValueAddedArgs extends ValueID {
    newValue: unknown;
}
export interface ValueRemovedArgs extends ValueID {
    prevValue: unknown;
}
export interface ValueNotificationArgs extends ValueID {
    value: unknown;
}
export interface MetadataUpdatedArgs extends ValueID {
    metadata: ValueMetadata | undefined;
}
declare type ValueAddedCallback = (args: ValueAddedArgs) => void;
declare type ValueUpdatedCallback = (args: ValueUpdatedArgs) => void;
declare type ValueRemovedCallback = (args: ValueRemovedArgs) => void;
declare type ValueNotificationCallback = (args: ValueNotificationArgs) => void;
declare type MetadataUpdatedCallback = (args: MetadataUpdatedArgs) => void;
interface ValueDBEventCallbacks {
    "value added": ValueAddedCallback;
    "value updated": ValueUpdatedCallback;
    "value removed": ValueRemovedCallback;
    "value notification": ValueNotificationCallback;
    "metadata updated": MetadataUpdatedCallback;
}
declare type ValueDBEvents = Extract<keyof ValueDBEventCallbacks, string>;
export declare function isValueID(param: Record<any, any>): param is ValueID;
export declare function assertValueID(param: Record<any, any>): asserts param is ValueID;
export interface ValueDB {
    on<TEvent extends ValueDBEvents>(event: TEvent, callback: ValueDBEventCallbacks[TEvent]): this;
    once<TEvent extends ValueDBEvents>(event: TEvent, callback: ValueDBEventCallbacks[TEvent]): this;
    removeListener<TEvent extends ValueDBEvents>(event: TEvent, callback: ValueDBEventCallbacks[TEvent]): this;
    removeAllListeners(event?: ValueDBEvents): this;
    emit<TEvent extends ValueDBEvents>(event: TEvent, ...args: Parameters<ValueDBEventCallbacks[TEvent]>): boolean;
}
/**
 * Ensures all Value ID properties are in the same order and there are no extraneous properties.
 * A normalized value ID can be used as a database key */
export declare function normalizeValueID(valueID: ValueID): ValueID;
export declare function valueIdToString(valueID: ValueID): string;
export interface SetValueOptions {
    /** When this is true, no event will be emitted for the value change */
    noEvent?: boolean;
    /** When this is true,  */
    noThrow?: boolean;
    /**
     * When this is `false`, the value will not be stored and a `value notification` event will be emitted instead (implies `noEvent: false`).
     */
    stateful?: boolean;
}
/**
 * The value store for a single node
 */
export declare class ValueDB extends EventEmitter {
    /**
     * @param nodeId The ID of the node this Value DB belongs to
     * @param valueDB The DB instance which stores values
     * @param metadataDB The DB instance which stores metadata
     * @param ownKeys An optional pre-created index of this ValueDB's own keys
     */
    constructor(nodeId: number, valueDB: JsonlDB, metadataDB: JsonlDB<ValueMetadata>, ownKeys?: Set<string>);
    private nodeId;
    private _db;
    private _metadata;
    private _index;
    private buildIndex;
    private valueIdToDBKey;
    private dbKeyToValueId;
    /**
     * Stores a value for a given value id
     */
    setValue(valueId: ValueID, value: unknown, options?: SetValueOptions): void;
    /**
     * Removes a value for a given value id
     */
    removeValue(valueId: ValueID, options?: SetValueOptions): boolean;
    /**
     * Retrieves a value for a given value id
     */
    getValue<T = unknown>(valueId: ValueID): T | undefined;
    /**
     * Checks if a value for a given value id exists in this ValueDB
     */
    hasValue(valueId: ValueID): boolean;
    /** Returns all values whose id matches the given predicate */
    findValues(predicate: (id: ValueID) => boolean): (ValueID & {
        value: unknown;
    })[];
    /** Returns all values that are stored for a given CC */
    getValues(forCC: CommandClasses): (ValueID & {
        value: unknown;
    })[];
    /** Clears all values from the value DB */
    clear(options?: SetValueOptions): void;
    /**
     * Stores metadata for a given value id
     */
    setMetadata(valueId: ValueID, metadata: ValueMetadata | undefined, options?: SetValueOptions): void;
    /**
     * Checks if metadata for a given value id exists in this ValueDB
     */
    hasMetadata(valueId: ValueID): boolean;
    /**
     * Retrieves metadata for a given value id
     */
    getMetadata(valueId: ValueID): ValueMetadata | undefined;
    /** Returns all metadata that is stored for a given CC */
    getAllMetadata(forCC: CommandClasses): (ValueID & {
        metadata: ValueMetadata;
    })[];
    /** Returns all values whose id matches the given predicate */
    findMetadata(predicate: (id: ValueID) => boolean): (ValueID & {
        metadata: ValueMetadata;
    })[];
}
/**
 * Really dumb but very fast way to parse one-lined JSON strings of the following schema
 * {
 *     nodeId: number,
 *     commandClass: number,
 *     endpoint: number,
 *     property: string | number,
 *     propertyKey: string | number,
 * }
 *
 * In benchmarks this was about 58% faster than JSON.parse
 */
export declare function dbKeyToValueIdFast(key: string): {
    nodeId: number;
} & ValueID;
/** Extracts an index for each node from one or more JSONL DBs */
export declare function indexDBsByNode(databases: JsonlDB[]): Map<number, Set<string>>;
export {};
//# sourceMappingURL=ValueDB.d.ts.map