"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssociationCCSupportedGroupingsGet = exports.AssociationCCSupportedGroupingsReport = exports.AssociationCCGet = exports.AssociationCCReport = exports.AssociationCCRemove = exports.AssociationCCSet = exports.AssociationCC = exports.AssociationCCAPI = exports.AssociationCommand = exports.getLifelineGroupIds = exports.getHasLifelineValueId = exports.getGroupCountValueId = exports.getNodeIdsValueId = exports.getMaxNodesValueId = void 0;
const core_1 = require("@zwave-js/core");
const arrays_1 = require("alcalzone-shared/arrays");
const Constants_1 = require("../message/Constants");
const API_1 = require("./API");
const CommandClass_1 = require("./CommandClass");
/** Returns the ValueID used to store the maximum number of nodes of an association group */
function getMaxNodesValueId(groupId) {
    return {
        commandClass: core_1.CommandClasses.Association,
        property: "maxNodes",
        propertyKey: groupId,
    };
}
exports.getMaxNodesValueId = getMaxNodesValueId;
/** Returns the ValueID used to store the node IDs of an association group */
function getNodeIdsValueId(groupId) {
    return {
        commandClass: core_1.CommandClasses.Association,
        property: "nodeIds",
        propertyKey: groupId,
    };
}
exports.getNodeIdsValueId = getNodeIdsValueId;
/** Returns the ValueID used to store the group count of an association group */
function getGroupCountValueId() {
    return {
        commandClass: core_1.CommandClasses.Association,
        property: "groupCount",
    };
}
exports.getGroupCountValueId = getGroupCountValueId;
/** Returns the ValueID used to store whether a node has a lifeline association */
function getHasLifelineValueId() {
    return {
        commandClass: core_1.CommandClasses.Association,
        property: "hasLifeline",
    };
}
exports.getHasLifelineValueId = getHasLifelineValueId;
function getLifelineGroupIds(node) {
    var _a, _b;
    // Some nodes define multiple lifeline groups, so we need to assign us to
    // all of them
    const lifelineGroups = [];
    // If the target node supports Z-Wave+ info that means the lifeline MUST be group #1
    if (node.supportsCC(core_1.CommandClasses["Z-Wave Plus Info"])) {
        lifelineGroups.push(1);
    }
    // We have a device config file that tells us which (additional) association to assign
    if ((_b = (_a = node.deviceConfig) === null || _a === void 0 ? void 0 : _a.associations) === null || _b === void 0 ? void 0 : _b.size) {
        lifelineGroups.push(...[...node.deviceConfig.associations.values()]
            .filter((a) => a.isLifeline)
            .map((a) => a.groupId));
    }
    return arrays_1.distinct(lifelineGroups).sort();
}
exports.getLifelineGroupIds = getLifelineGroupIds;
// All the supported commands
var AssociationCommand;
(function (AssociationCommand) {
    AssociationCommand[AssociationCommand["Set"] = 1] = "Set";
    AssociationCommand[AssociationCommand["Get"] = 2] = "Get";
    AssociationCommand[AssociationCommand["Report"] = 3] = "Report";
    AssociationCommand[AssociationCommand["Remove"] = 4] = "Remove";
    AssociationCommand[AssociationCommand["SupportedGroupingsGet"] = 5] = "SupportedGroupingsGet";
    AssociationCommand[AssociationCommand["SupportedGroupingsReport"] = 6] = "SupportedGroupingsReport";
    // TODO: These two commands are V2. I have no clue how this is supposed to function:
    // SpecificGroupGet = 0x0b,
    // SpecificGroupReport = 0x0c,
    // Here's what the docs have to say:
    // This functionality allows a supporting multi-button device to detect a key press and subsequently advertise
    // the identity of the key. The following sequence of events takes place:
    // * The user activates a special identification sequence and pushes the button to be identified
    // * The device issues a Node Information frame (NIF)
    // * The NIF allows the portable controller to determine the NodeID of the multi-button device
    // * The portable controller issues an Association Specific Group Get Command to the multi-button device
    // * The multi-button device returns an Association Specific Group Report Command that advertises the
    //   association group that represents the most recently detected button
})(AssociationCommand = exports.AssociationCommand || (exports.AssociationCommand = {}));
// @noSetValueAPI
let AssociationCCAPI = class AssociationCCAPI extends API_1.PhysicalCCAPI {
    supportsCommand(cmd) {
        switch (cmd) {
            case AssociationCommand.Get:
            case AssociationCommand.Set:
            case AssociationCommand.Remove:
            case AssociationCommand.SupportedGroupingsGet:
                return true; // This is mandatory
            // Not implemented:
            // case AssociationCommand.SpecificGroupGet:
            // return this.version >= 2;
        }
        return super.supportsCommand(cmd);
    }
    /**
     * Returns the number of association groups a node supports.
     * Association groups are consecutive, starting at 1.
     */
    async getGroupCount() {
        this.assertSupportsCommand(AssociationCommand, AssociationCommand.SupportedGroupingsGet);
        const cc = new AssociationCCSupportedGroupingsGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response)
            return response.groupCount;
    }
    /**
     * Returns information about an association group.
     */
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getGroup(groupId) {
        this.assertSupportsCommand(AssociationCommand, AssociationCommand.Get);
        const cc = new AssociationCCGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            groupId,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return {
                maxNodes: response.maxNodes,
                nodeIds: response.nodeIds,
            };
        }
    }
    /**
     * Adds new nodes to an association group
     */
    async addNodeIds(groupId, ...nodeIds) {
        this.assertSupportsCommand(AssociationCommand, AssociationCommand.Set);
        const cc = new AssociationCCSet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            groupId,
            nodeIds,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    /**
     * Removes nodes from an association group
     */
    async removeNodeIds(options) {
        this.assertSupportsCommand(AssociationCommand, AssociationCommand.Remove);
        const cc = new AssociationCCRemove(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            ...options,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    /**
     * Removes nodes from all association groups
     */
    async removeNodeIdsFromAllGroups(nodeIds) {
        var _a;
        this.assertSupportsCommand(AssociationCommand, AssociationCommand.Remove);
        if (this.version >= 2) {
            // The node supports bulk removal
            return this.removeNodeIds({ nodeIds, groupId: 0 });
        }
        else {
            // We have to remove the node manually from all groups
            const node = this.endpoint.getNodeUnsafe();
            const groupCount = (_a = node.valueDB.getValue(getGroupCountValueId())) !== null && _a !== void 0 ? _a : 0;
            for (let groupId = 1; groupId <= groupCount; groupId++) {
                await this.removeNodeIds({ nodeIds, groupId });
            }
        }
    }
};
AssociationCCAPI = __decorate([
    CommandClass_1.API(core_1.CommandClasses.Association)
], AssociationCCAPI);
exports.AssociationCCAPI = AssociationCCAPI;
let AssociationCC = class AssociationCC extends CommandClass_1.CommandClass {
    constructor(driver, options) {
        super(driver, options);
        this.registerValue(getHasLifelineValueId().property, true);
    }
    determineRequiredCCInterviews() {
        // AssociationCC must be interviewed after Z-Wave+ if that is supported
        return [
            ...super.determineRequiredCCInterviews(),
            core_1.CommandClasses["Z-Wave Plus Info"],
        ];
    }
    skipEndpointInterview() {
        // The associations are managed on the root device
        return true;
    }
    /**
     * Returns the number of association groups reported by the node.
     * This only works AFTER the interview process
     */
    getGroupCountCached() {
        return this.getValueDB().getValue(getGroupCountValueId()) || 0;
    }
    /**
     * Returns the number of nodes an association group supports.
     * This only works AFTER the interview process
     */
    getMaxNodesCached(groupId) {
        return this.getValueDB().getValue(getMaxNodesValueId(groupId)) || 1;
    }
    /**
     * Returns all the destinations of all association groups reported by the node.
     * This only works AFTER the interview process
     */
    getAllDestinationsCached() {
        var _a;
        const ret = new Map();
        const groupCount = this.getGroupCountCached();
        const valueDB = this.getValueDB();
        for (let i = 1; i <= groupCount; i++) {
            // Add all root destinations
            const nodes = (_a = valueDB.getValue(getNodeIdsValueId(i))) !== null && _a !== void 0 ? _a : [];
            ret.set(i, 
            // Filter out duplicates
            arrays_1.distinct(nodes).map((nodeId) => ({ nodeId })));
        }
        return ret;
    }
    async interview(complete = true) {
        var _a;
        const node = this.getNode();
        const endpoint = this.getEndpoint();
        const api = endpoint.commandClasses.Association.withOptions({
            priority: Constants_1.MessagePriority.NodeQuery,
        });
        this.driver.controllerLog.logNode(node.id, {
            endpoint: this.endpointIndex,
            message: `${this.constructor.name}: doing a ${complete ? "complete" : "partial"} interview...`,
            direction: "none",
        });
        // Even if Multi Channel Association is supported, we still need to query the number of
        // normal association groups since some devices report more association groups than
        // multi channel association groups
        let groupCount;
        if (complete) {
            // First find out how many groups are supported
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: "querying number of association groups...",
                direction: "outbound",
            });
            groupCount = await api.getGroupCount();
            if (groupCount != undefined) {
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: `supports ${groupCount} association groups`,
                    direction: "inbound",
                });
            }
            else {
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: "Querying association groups timed out, skipping interview...",
                    level: "warn",
                });
                return;
            }
        }
        else {
            // Partial interview, read the information from cache
            groupCount = this.getGroupCountCached();
        }
        // Skip the remaining quer Association CC in favor of Multi Channel Association if possible
        if (endpoint.commandClasses["Multi Channel Association"].isSupported()) {
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: `${this.constructor.name}: skipping remaining interview because Multi Channel Association is supported...`,
                direction: "none",
            });
            this.interviewComplete = true;
            return;
        }
        // Then query each association group
        for (let groupId = 1; groupId <= groupCount; groupId++) {
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: `querying association group #${groupId}...`,
                direction: "outbound",
            });
            const group = await api.getGroup(groupId);
            if (group != undefined) {
                const logMessage = `received information for association group #${groupId}:
maximum # of nodes: ${group.maxNodes}
currently assigned nodes: ${group.nodeIds.map(String).join(", ")}`;
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: logMessage,
                    direction: "inbound",
                });
            }
        }
        // Assign the controller to all lifeline groups
        const lifelineGroups = getLifelineGroupIds(node);
        const ownNodeId = this.driver.controller.ownNodeId;
        const valueDB = this.getValueDB();
        if (lifelineGroups.length) {
            for (const group of lifelineGroups) {
                // Check if we are already in the lifeline group
                const lifelineValueId = getNodeIdsValueId(group);
                const lifelineNodeIds = (_a = valueDB.getValue(lifelineValueId)) !== null && _a !== void 0 ? _a : [];
                if (!lifelineNodeIds.includes(ownNodeId)) {
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: `Controller missing from lifeline group #${group}, assinging ourselves...`,
                        direction: "outbound",
                    });
                    // Add a new destination
                    await api.addNodeIds(group, ownNodeId);
                    // and refresh it - don't trust that it worked
                    await api.getGroup(group);
                    // TODO: check if it worked
                }
            }
            // Remember that we have a lifeline association
            valueDB.setValue(getHasLifelineValueId(), true);
        }
        else {
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: "No information about Lifeline associations, cannot assign ourselves!",
                direction: "outbound",
                level: "warn",
            });
            // Remember that we have NO lifeline association
            valueDB.setValue(getHasLifelineValueId(), false);
        }
        // Remember that the interview is complete
        this.interviewComplete = true;
    }
};
AssociationCC = __decorate([
    CommandClass_1.commandClass(core_1.CommandClasses.Association),
    CommandClass_1.implementedVersion(3)
], AssociationCC);
exports.AssociationCC = AssociationCC;
let AssociationCCSet = class AssociationCCSet extends AssociationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            if (options.groupId < 1) {
                throw new core_1.ZWaveError("The group id must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            if (options.nodeIds.some((n) => n < 1 || n > core_1.MAX_NODES)) {
                throw new core_1.ZWaveError(`All node IDs must be between 1 and ${core_1.MAX_NODES}!`, core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            this.groupId = options.groupId;
            this.nodeIds = options.nodeIds;
        }
    }
    serialize() {
        this.payload = Buffer.from([this.groupId, ...this.nodeIds]);
        return super.serialize();
    }
    toLogEntry() {
        const message = {
            "group id": this.groupId || "all groups",
            "node ids": this.nodeIds.length
                ? this.nodeIds.join(", ")
                : "all nodes",
        };
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
AssociationCCSet = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.Set)
], AssociationCCSet);
exports.AssociationCCSet = AssociationCCSet;
let AssociationCCRemove = class AssociationCCRemove extends AssociationCC {
    constructor(driver, options) {
        var _a;
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            // Validate options
            if (!options.groupId) {
                if (this.version === 1) {
                    throw new core_1.ZWaveError(`Node ${this.nodeId} only supports AssociationCC V1 which requires the group Id to be set`, core_1.ZWaveErrorCodes.Argument_Invalid);
                }
            }
            else if (options.groupId < 0) {
                throw new core_1.ZWaveError("The group id must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            if ((_a = options.nodeIds) === null || _a === void 0 ? void 0 : _a.some((n) => n < 1 || n > core_1.MAX_NODES)) {
                throw new core_1.ZWaveError(`All node IDs must be between 1 and ${core_1.MAX_NODES}!`, core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            this.groupId = options.groupId;
            this.nodeIds = options.nodeIds;
        }
    }
    serialize() {
        this.payload = Buffer.from([
            this.groupId || 0,
            ...(this.nodeIds || []),
        ]);
        return super.serialize();
    }
    toLogEntry() {
        const message = {
            "group id": this.groupId || "all groups",
            "node ids": this.nodeIds && this.nodeIds.length
                ? this.nodeIds.join(", ")
                : "all nodes",
        };
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
AssociationCCRemove = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.Remove)
], AssociationCCRemove);
exports.AssociationCCRemove = AssociationCCRemove;
let AssociationCCReport = class AssociationCCReport extends AssociationCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 3);
        this._groupId = this.payload[0];
        this._maxNodes = this.payload[1];
        this._reportsToFollow = this.payload[2];
        this._nodeIds = [...this.payload.slice(3)];
    }
    get groupId() {
        return this._groupId;
    }
    get maxNodes() {
        return this._maxNodes;
    }
    get nodeIds() {
        return this._nodeIds;
    }
    get reportsToFollow() {
        return this._reportsToFollow;
    }
    getPartialCCSessionId() {
        // Distinguish sessions by the association group ID
        return { groupId: this._groupId };
    }
    expectMoreMessages() {
        return this._reportsToFollow > 0;
    }
    mergePartialCCs(partials) {
        // Concat the list of nodes
        this._nodeIds = [...partials, this]
            .map((report) => report._nodeIds)
            .reduce((prev, cur) => prev.concat(...cur), []);
        // Persist values
        this.getValueDB().setValue(getMaxNodesValueId(this._groupId), this._maxNodes);
        this.getValueDB().setValue(getNodeIdsValueId(this._groupId), this._nodeIds);
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "group id": this.groupId,
                "max # of nodes": this.maxNodes,
                "node IDs": this.nodeIds.join(", "),
                "reports to follow": this.reportsToFollow,
            },
        };
    }
};
__decorate([
    CommandClass_1.ccValue({ internal: true })
], AssociationCCReport.prototype, "maxNodes", null);
__decorate([
    CommandClass_1.ccValue({ internal: true })
], AssociationCCReport.prototype, "nodeIds", null);
AssociationCCReport = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.Report)
], AssociationCCReport);
exports.AssociationCCReport = AssociationCCReport;
let AssociationCCGet = class AssociationCCGet extends AssociationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            if (options.groupId < 1) {
                throw new core_1.ZWaveError("The group id must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
            }
            this.groupId = options.groupId;
        }
    }
    serialize() {
        this.payload = Buffer.from([this.groupId]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "group id": this.groupId },
        };
    }
};
AssociationCCGet = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.Get),
    CommandClass_1.expectedCCResponse(AssociationCCReport)
], AssociationCCGet);
exports.AssociationCCGet = AssociationCCGet;
let AssociationCCSupportedGroupingsReport = class AssociationCCSupportedGroupingsReport extends AssociationCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 1);
        this._groupCount = this.payload[0];
        this.persistValues();
    }
    get groupCount() {
        return this._groupCount;
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "group count": this.groupCount },
        };
    }
};
__decorate([
    CommandClass_1.ccValue({ internal: true })
], AssociationCCSupportedGroupingsReport.prototype, "groupCount", null);
AssociationCCSupportedGroupingsReport = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.SupportedGroupingsReport)
], AssociationCCSupportedGroupingsReport);
exports.AssociationCCSupportedGroupingsReport = AssociationCCSupportedGroupingsReport;
let AssociationCCSupportedGroupingsGet = class AssociationCCSupportedGroupingsGet extends AssociationCC {
};
AssociationCCSupportedGroupingsGet = __decorate([
    CommandClass_1.CCCommand(AssociationCommand.SupportedGroupingsGet),
    CommandClass_1.expectedCCResponse(AssociationCCSupportedGroupingsReport)
], AssociationCCSupportedGroupingsGet);
exports.AssociationCCSupportedGroupingsGet = AssociationCCSupportedGroupingsGet;

//# sourceMappingURL=AssociationCC.js.map
