"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Driver = void 0;
const jsonl_db_1 = require("@alcalzone/jsonl-db");
const Sentry = __importStar(require("@sentry/node"));
const config_1 = require("@zwave-js/config");
const core_1 = require("@zwave-js/core");
const serial_1 = require("@zwave-js/serial");
const shared_1 = require("@zwave-js/shared");
const async_1 = require("alcalzone-shared/async");
const deferred_promise_1 = require("alcalzone-shared/deferred-promise");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const events_1 = require("events");
const fs_extra_1 = __importDefault(require("fs-extra"));
const path_1 = __importDefault(require("path"));
const serialport_1 = __importDefault(require("serialport"));
const url_1 = require("url");
const util = __importStar(require("util"));
const xstate_1 = require("xstate");
const commandclass_1 = require("../commandclass");
const CommandClass_1 = require("../commandclass/CommandClass");
const DeviceResetLocallyCC_1 = require("../commandclass/DeviceResetLocallyCC");
const EncapsulatingCommandClass_1 = require("../commandclass/EncapsulatingCommandClass");
const ICommandClassContainer_1 = require("../commandclass/ICommandClassContainer");
const MultiChannelCC_1 = require("../commandclass/MultiChannelCC");
const NoOperationCC_1 = require("../commandclass/NoOperationCC");
const SecurityCC_1 = require("../commandclass/SecurityCC");
const SupervisionCC_1 = require("../commandclass/SupervisionCC");
const WakeUpCC_1 = require("../commandclass/WakeUpCC");
const ApplicationCommandRequest_1 = require("../controller/ApplicationCommandRequest");
const ApplicationUpdateRequest_1 = require("../controller/ApplicationUpdateRequest");
const BridgeApplicationCommandRequest_1 = require("../controller/BridgeApplicationCommandRequest");
const Controller_1 = require("../controller/Controller");
const SendDataMessages_1 = require("../controller/SendDataMessages");
const Controller_2 = require("../log/Controller");
const Driver_1 = require("../log/Driver");
const Constants_1 = require("../message/Constants");
const Message_1 = require("../message/Message");
const INodeQuery_1 = require("../node/INodeQuery");
const Types_1 = require("../node/Types");
const SendThreadMachine_1 = require("./SendThreadMachine");
const ThrottlePresets_1 = require("./ThrottlePresets");
const Transaction_1 = require("./Transaction");
// eslint-disable-next-line
const { version: libVersion } = require("../../../package.json");
// This is made with cfonts:
const libNameString = `
███████╗ ██╗    ██╗  █████╗  ██╗   ██╗ ███████╗             ██╗ ███████╗
╚══███╔╝ ██║    ██║ ██╔══██╗ ██║   ██║ ██╔════╝             ██║ ██╔════╝
  ███╔╝  ██║ █╗ ██║ ███████║ ██║   ██║ █████╗   █████╗      ██║ ███████╗
 ███╔╝   ██║███╗██║ ██╔══██║ ╚██╗ ██╔╝ ██╔══╝   ╚════╝ ██   ██║ ╚════██║
███████╗ ╚███╔███╔╝ ██║  ██║  ╚████╔╝  ███████╗        ╚█████╔╝ ███████║
╚══════╝  ╚══╝╚══╝  ╚═╝  ╚═╝   ╚═══╝   ╚══════╝         ╚════╝  ╚══════╝
`;
const defaultOptions = {
    timeouts: {
        ack: 1000,
        byte: 150,
        response: 1600,
        report: 10000,
        nonce: 5000,
        sendDataCallback: 65000,
        refreshValue: 5000,
    },
    attempts: {
        controller: 3,
        sendData: 3,
        retryAfterTransmitReport: false,
        nodeInterview: 5,
    },
    skipInterview: false,
    storage: {
        driver: fs_extra_1.default,
        cacheDir: path_1.default.resolve(__dirname, "../../..", "cache"),
        throttle: "normal",
    },
};
/**
 * Merges the user-defined options with the default options
 */
function applyDefaultOptions(target, source) {
    target = target || {};
    for (const [key, value] of objects_1.entries(source)) {
        if (!(key in target)) {
            target[key] = value;
        }
        else {
            if (typeof value === "object") {
                // merge objects
                target[key] = applyDefaultOptions(target[key], value);
            }
            else if (typeof target[key] === "undefined") {
                // don't override single keys
                target[key] = value;
            }
        }
    }
    return target;
}
/** Ensures that the options are valid */
function checkOptions(options) {
    if (options.timeouts.ack < 1) {
        throw new core_1.ZWaveError(`The ACK timeout must be positive!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.timeouts.byte < 1) {
        throw new core_1.ZWaveError(`The BYTE timeout must be positive!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.timeouts.response < 500 || options.timeouts.response > 5000) {
        throw new core_1.ZWaveError(`The Response timeout must be between 500 and 5000 milliseconds!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.timeouts.report < 1000 || options.timeouts.report > 40000) {
        throw new core_1.ZWaveError(`The Report timeout must be between 1000 and 40000 milliseconds!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.timeouts.nonce < 3000 || options.timeouts.nonce > 20000) {
        throw new core_1.ZWaveError(`The Nonce timeout must be between 3000 and 20000 milliseconds!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.timeouts.sendDataCallback < 10000) {
        throw new core_1.ZWaveError(`The Send Data Callback timeout must be at least 10000 milliseconds!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.networkKey != undefined && options.networkKey.length !== 16) {
        throw new core_1.ZWaveError(`The network key must be a buffer with length 16!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.attempts.controller < 1 || options.attempts.controller > 3) {
        throw new core_1.ZWaveError(`The Controller attempts must be between 1 and 3!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.attempts.sendData < 1 ||
        options.attempts.sendData > SendDataMessages_1.MAX_SEND_ATTEMPTS) {
        throw new core_1.ZWaveError(`The SendData attempts must be between 1 and ${SendDataMessages_1.MAX_SEND_ATTEMPTS}!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
    if (options.attempts.nodeInterview < 1 ||
        options.attempts.nodeInterview > 10) {
        throw new core_1.ZWaveError(`The Node interview attempts must be between 1 and 10!`, core_1.ZWaveErrorCodes.Driver_InvalidOptions);
    }
}
/**
 * The driver is the core of this library. It controls the serial interface,
 * handles transmission and receipt of messages and manages the network cache.
 * Any action you want to perform on the Z-Wave network must go through a driver
 * instance or its associated nodes.
 */
class Driver extends events_1.EventEmitter {
    constructor(port, options) {
        super();
        this.port = port;
        /** A map of handlers for all sorts of requests */
        this.requestHandlers = new Map();
        /** A map of awaited commands */
        this.awaitedCommands = [];
        /** A map of all current supervision sessions that may still receive updates */
        this.supervisionSessions = new Map();
        this._wasStarted = false;
        this._isOpen = false;
        // wotan-enable async-function-assignability
        this._controllerInterviewed = false;
        this._nodesReady = new Set();
        this._nodesReadyEventEmitted = false;
        this.retryNodeInterviewTimeouts = new Map();
        this._wasDestroyed = false;
        this._cleanupHandler = () => {
            void this.destroy();
        };
        this.partialCCSessions = new Map();
        this.lastCallbackId = 0xff;
        this.lastSaveToCache = 0;
        this.saveToCacheInterval = 150;
        this.isSavingToCache = false;
        this.sendNodeToSleepTimers = new Map();
        this.logContainer = new core_1.ZWaveLogContainer(options === null || options === void 0 ? void 0 : options.logConfig);
        this.driverLog = new Driver_1.DriverLogger(this.logContainer);
        this.controllerLog = new Controller_2.ControllerLogger(this.logContainer);
        // merge given options with defaults
        this.options = applyDefaultOptions(options, defaultOptions);
        // And make sure they contain valid values
        checkOptions(this.options);
        this.cacheDir = this.options.storage.cacheDir;
        // Initialize config manager
        this.configManager = new config_1.ConfigManager(this.logContainer);
        // register some cleanup handlers in case the program doesn't get closed cleanly
        this._cleanupHandler = this._cleanupHandler.bind(this);
        process.on("exit", this._cleanupHandler);
        process.on("SIGINT", this._cleanupHandler);
        process.on("uncaughtException", this._cleanupHandler);
        // And initialize but don't start the send thread machine
        const sendThreadMachine = SendThreadMachine_1.createSendThreadMachine({
            sendData: this.writeSerial.bind(this),
            createSendDataAbort: () => new SendDataMessages_1.SendDataAbort(this),
            notifyUnsolicited: (msg) => {
                void this.handleUnsolicitedMessage(msg);
            },
            notifyRetry: (command, lastError, message, attempts, maxAttempts, delay) => {
                var _a;
                if (command === "SendData") {
                    this.controllerLog.logNode((_a = message.getNodeId()) !== null && _a !== void 0 ? _a : 255, `did not respond after ${attempts}/${maxAttempts} attempts. Scheduling next try in ${delay} ms.`, "warn");
                }
                else {
                    // Translate the error into a better one
                    let errorReason;
                    switch (lastError) {
                        case "response timeout":
                            errorReason = "No response from controller";
                            break;
                        case "callback timeout":
                            errorReason = "No callback from controller";
                            break;
                        case "response NOK":
                            errorReason =
                                "The controller response indicated failure";
                            break;
                        case "callback NOK":
                            errorReason =
                                "The controller callback indicated failure";
                            break;
                        case "ACK timeout":
                        case "CAN":
                        case "NAK":
                        default:
                            errorReason =
                                "Failed to execute controller command";
                            break;
                    }
                    this.controllerLog.print(`${errorReason} after ${attempts}/${maxAttempts} attempts. Scheduling next try in ${delay} ms.`, "warn");
                }
            },
            timestamp: core_1.highResTimestamp,
            rejectTransaction: (transaction, error) => {
                // If a node failed to respond in time, it might be sleeping
                if (this.isMissingNodeACK(transaction, error)) {
                    if (this.handleMissingNodeACK(transaction))
                        return;
                }
                transaction.promise.reject(error);
            },
            resolveTransaction: (transaction, result) => {
                transaction.promise.resolve(result);
            },
            logOutgoingMessage: (msg) => {
                this.driverLog.logMessage(msg, {
                    direction: "outbound",
                });
                if (process.env.NODE_ENV !== "test") {
                    // Enrich error data in case something goes wrong
                    Sentry.addBreadcrumb({
                        category: "message",
                        timestamp: Date.now() / 1000,
                        type: "debug",
                        data: {
                            direction: "outbound",
                            msgType: msg.type,
                            functionType: msg.functionType,
                            name: msg.constructor.name,
                            nodeId: msg.getNodeId(),
                            ...msg.toLogEntry(),
                        },
                    });
                }
            },
            log: this.driverLog.print.bind(this.driverLog),
        }, shared_1.pick(this.options, ["timeouts", "attempts"]));
        this.sendThread = xstate_1.interpret(sendThreadMachine);
        // this.sendThread.onTransition((state) => {
        // 	if (state.changed)
        // 		this.driverLog.print(
        // 			`send thread state: ${state.toStrings().slice(-1)[0]}`,
        // 			"verbose",
        // 		);
        // });
    }
    /** @internal */
    get valueDB() {
        return this._valueDB;
    }
    /** @internal */
    get metadataDB() {
        return this._metadataDB;
    }
    /** Encapsulates information about the Z-Wave controller and provides access to its nodes */
    get controller() {
        if (this._controller == undefined) {
            throw new core_1.ZWaveError("The controller is not yet ready!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this._controller;
    }
    /** @internal */
    get securityManager() {
        return this._securityManager;
    }
    /** Enumerates all existing serial ports */
    static async enumerateSerialPorts() {
        const ports = await serialport_1.default.list();
        return ports.map((port) => port.path);
    }
    /** Start the driver */
    // wotan-disable async-function-assignability
    async start() {
        // avoid starting twice
        if (this._wasDestroyed) {
            throw new core_1.ZWaveError("The driver was destroyed. Create a new instance and start that one.", core_1.ZWaveErrorCodes.Driver_Destroyed);
        }
        if (this._wasStarted)
            return Promise.resolve();
        this._wasStarted = true;
        // Enforce that an error handler is attached
        if (this.listenerCount("error") === 0) {
            throw new core_1.ZWaveError(`Before starting the driver, a handler for the "error" event must be attached.`, core_1.ZWaveErrorCodes.Driver_NoErrorHandler);
        }
        const spOpenPromise = deferred_promise_1.createDeferredPromise();
        // Log which version is running
        this.driverLog.print(libNameString, "info");
        this.driverLog.print(`version ${libVersion}`, "info");
        this.driverLog.print("", "info");
        this.driverLog.print("starting driver...");
        this.sendThread.start();
        // Open the serial port
        if (this.port.startsWith("tcp://")) {
            const url = new url_1.URL(this.port);
            this.driverLog.print(`opening serial port ${this.port}`);
            this.serial = new serial_1.ZWaveSocket({
                host: url.hostname,
                port: parseInt(url.port),
            }, this.logContainer);
        }
        else {
            this.driverLog.print(`opening serial port ${this.port}`);
            this.serial = new serial_1.ZWaveSerialPort(this.port, this.logContainer);
        }
        this.serial
            .on("data", this.serialport_onData.bind(this))
            .on("error", (err) => {
            this.driverLog.print(`serial port errored: ${err.message}`, "error");
            if (this._isOpen) {
                this.serialport_onError(err);
            }
            else {
                spOpenPromise.reject(err);
                void this.destroy();
            }
        });
        // If the port is already open, close it first
        if (this.serial.isOpen)
            await this.serial.close();
        // IMPORTANT: Test code expects the open promise to be created and returned synchronously
        // Everything async (inluding opening the serial port) must happen in the setImmediate callback
        // asynchronously open the serial port
        setImmediate(async () => {
            try {
                await this.serial.open();
            }
            catch (e) {
                const message = `Failed to open the serial port: ${e.message}`;
                this.driverLog.print(message, "error");
                spOpenPromise.reject(new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Driver_Failed));
                void this.destroy();
                return;
            }
            this.driverLog.print("serial port opened");
            this._isOpen = true;
            spOpenPromise.resolve();
            await this.writeHeader(serial_1.MessageHeaders.NAK);
            // set unref, so stopping the process doesn't need to wait for the 1500ms
            await async_1.wait(1500, true);
            // Load the necessary configuration
            this.driverLog.print("loading configuration...");
            try {
                await this.configManager.loadDeviceClasses();
                await this.configManager.loadManufacturers();
                await this.configManager.loadDeviceIndex();
                await this.configManager.loadNotifications();
                await this.configManager.loadNamedScales();
                await this.configManager.loadSensorTypes();
                await this.configManager.loadMeters();
                await this.configManager.loadIndicators();
            }
            catch (e) {
                const message = `Failed to load the configuration: ${e.message}`;
                this.driverLog.print(message, "error");
                this.emit("error", new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Driver_Failed));
                void this.destroy();
                return;
            }
            this.driverLog.print("beginning interview...");
            try {
                await this.initializeControllerAndNodes();
            }
            catch (e) {
                let message;
                if (e instanceof core_1.ZWaveError &&
                    e.code === core_1.ZWaveErrorCodes.Controller_MessageDropped) {
                    message = `Failed to initialize the driver, no response from the controller. Are you sure this is a Z-Wave controller?`;
                }
                else {
                    message = `Failed to initialize the driver: ${e instanceof Error ? e.message : String(e)}`;
                }
                this.driverLog.print(message, "error");
                this.emit("error", new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Driver_Failed));
                void this.destroy();
                return;
            }
        });
        return spOpenPromise;
    }
    /** Indicates whether all nodes are ready, i.e. the "all nodes ready" event has been emitted */
    get allNodesReady() {
        return this._nodesReadyEventEmitted;
    }
    /**
     * Initializes the variables for controller and nodes,
     * adds event handlers and starts the interview process.
     */
    async initializeControllerAndNodes() {
        if (this._controller == undefined) {
            this._controller = new Controller_1.ZWaveController(this);
            this._controller
                .on("node added", this.onNodeAdded.bind(this))
                .on("node removed", this.onNodeRemoved.bind(this));
        }
        const initValueDBs = async () => {
            // Always start the value and metadata databases
            const options = {
                ignoreReadErrors: true,
                ...ThrottlePresets_1.throttlePresets[this.options.storage.throttle],
            };
            const valueDBFile = path_1.default.join(this.cacheDir, `${this._controller.homeId.toString(16)}.values.jsonl`);
            this._valueDB = new jsonl_db_1.JsonlDB(valueDBFile, {
                ...options,
                reviver: (key, value) => core_1.deserializeCacheValue(value),
                serializer: (key, value) => core_1.serializeCacheValue(value),
            });
            await this._valueDB.open();
            const metadataDBFile = path_1.default.join(this.cacheDir, `${this._controller.homeId.toString(16)}.metadata.jsonl`);
            this._metadataDB = new jsonl_db_1.JsonlDB(metadataDBFile, options);
            await this._metadataDB.open();
            if (process.env.NO_CACHE === "true") {
                // Since value/metadata DBs are append-only, we need to clear them
                // if the cache should be ignored
                this._valueDB.clear();
                this._metadataDB.clear();
            }
        };
        if (!this.options.skipInterview) {
            // Interview the controller.
            await this._controller.interview(initValueDBs, async () => {
                // Try to restore the network information from the cache
                if (process.env.NO_CACHE !== "true") {
                    await this.restoreNetworkStructureFromCache();
                }
            });
            // No need to initialize databases if skipInterview is true, because it is only used in some
            // Driver unit tests that don't need access to them
        }
        // We need to know the controller node id to set up the security manager
        if (this.options.networkKey) {
            this._securityManager = new core_1.SecurityManager({
                networkKey: this.options.networkKey,
                ownNodeId: this._controller.ownNodeId,
                nonceTimeout: this.options.timeouts.nonce,
            });
        }
        // in any case we need to emit the driver ready event here
        this._controllerInterviewed = true;
        this.driverLog.print("driver ready");
        this.emit("driver ready");
        // Add event handlers for the nodes
        for (const node of this._controller.nodes.values()) {
            this.addNodeEventHandlers(node);
        }
        // Before interviewing nodes reset our knowledge about their ready state
        this._nodesReady.clear();
        this._nodesReadyEventEmitted = false;
        if (!this.options.skipInterview) {
            // Now interview all nodes
            // First complete the controller interview
            const controllerNode = this._controller.nodes.get(this._controller.ownNodeId);
            await this.interviewNode(controllerNode);
            // Then do all the nodes in parallel
            for (const node of this._controller.nodes.values()) {
                if (node.id === this._controller.ownNodeId)
                    continue;
                // don't await the interview, because it may take a very long time
                // if a node is asleep
                void this.interviewNode(node);
            }
        }
    }
    /**
     * @internal
     * Starts or resumes the interview of a Z-Wave node. It is advised to NOT
     * await this method as it can take a very long time (minutes to hours)!
     *
     * WARNING: Do not call this method from application code. To refresh the information
     * for a specific node, use `node.refreshInfo()` instead
     */
    async interviewNode(node) {
        if (node.interviewStage === Types_1.InterviewStage.Complete) {
            node.interviewStage = Types_1.InterviewStage.RestartFromCache;
        }
        // Avoid having multiple restart timeouts active
        if (this.retryNodeInterviewTimeouts.has(node.id)) {
            clearTimeout(this.retryNodeInterviewTimeouts.get(node.id));
            this.retryNodeInterviewTimeouts.delete(node.id);
        }
        // Drop all pending messages that come from a previous interview attempt
        this.rejectTransactions((t) => t.priority === Constants_1.MessagePriority.NodeQuery &&
            t.message.getNodeId() === node.id, "The interview was restarted", core_1.ZWaveErrorCodes.Controller_MessageDropped);
        const maxInterviewAttempts = this.options.attempts.nodeInterview;
        try {
            if (!(await node.interview())) {
                // Find out if we may retry the interview
                if (node.status === Types_1.NodeStatus.Dead) {
                    this.controllerLog.logNode(node.id, `Interview attempt (${node.interviewAttempts}/${maxInterviewAttempts}) failed, node is dead.`, "warn");
                    node.emit("interview failed", node, {
                        errorMessage: "The node is dead",
                        isFinal: true,
                    });
                }
                else if (node.interviewAttempts < maxInterviewAttempts) {
                    // This is most likely because the node is unable to handle our load of requests now. Give it some time
                    const retryTimeout = Math.min(30000, node.interviewAttempts * 5000);
                    this.controllerLog.logNode(node.id, `Interview attempt ${node.interviewAttempts}/${maxInterviewAttempts} failed, retrying in ${retryTimeout} ms...`, "warn");
                    node.emit("interview failed", node, {
                        errorMessage: `Attempt ${node.interviewAttempts}/${maxInterviewAttempts} failed`,
                        isFinal: false,
                        attempt: node.interviewAttempts,
                        maxAttempts: maxInterviewAttempts,
                    });
                    // Schedule the retry and remember the timeout instance
                    this.retryNodeInterviewTimeouts.set(node.id, setTimeout(() => {
                        this.retryNodeInterviewTimeouts.delete(node.id);
                        void this.interviewNode(node);
                    }, retryTimeout).unref());
                }
                else {
                    this.controllerLog.logNode(node.id, `Failed all interview attempts, giving up.`, "warn");
                    node.emit("interview failed", node, {
                        errorMessage: `Maximum interview attempts reached`,
                        isFinal: true,
                        attempt: maxInterviewAttempts,
                        maxAttempts: maxInterviewAttempts,
                    });
                }
            }
        }
        catch (e) {
            if (e instanceof core_1.ZWaveError) {
                if (e.code === core_1.ZWaveErrorCodes.Driver_NotReady ||
                    e.code === core_1.ZWaveErrorCodes.Controller_NodeRemoved) {
                    // This only happens when a node is removed during the interview - we don't log this
                    return;
                }
                this.controllerLog.logNode(node.id, `Error during node interview: ${e.message}`, "error");
            }
            else {
                throw e;
            }
        }
    }
    /** Adds the necessary event handlers for a node instance */
    addNodeEventHandlers(node) {
        node.on("wake up", this.onNodeWakeUp.bind(this))
            .on("sleep", this.onNodeSleep.bind(this))
            .on("alive", this.onNodeAlive.bind(this))
            .on("dead", this.onNodeDead.bind(this))
            .on("interview completed", this.onNodeInterviewCompleted.bind(this))
            .on("ready", this.onNodeReady.bind(this))
            .on("firmware update finished", this.onNodeFirmwareUpdated.bind(this));
    }
    /** Removes a node's event handlers that were added with addNodeEventHandlers */
    removeNodeEventHandlers(node) {
        node.removeAllListeners("wake up")
            .removeAllListeners("sleep")
            .removeAllListeners("alive")
            .removeAllListeners("dead")
            .removeAllListeners("interview completed")
            .removeAllListeners("ready")
            .removeAllListeners("firmware update finished");
    }
    /** Is called when a node wakes up */
    onNodeWakeUp(node, oldStatus) {
        this.controllerLog.logNode(node.id, `The node is ${oldStatus === Types_1.NodeStatus.Unknown ? "" : "now "}awake.`);
        // Make sure to handle the pending messages as quickly as possible
        if (oldStatus === Types_1.NodeStatus.Asleep) {
            this.sendThread.send({
                type: "reduce",
                reducer: ({ message }) => {
                    // Ignore messages that are not for this node
                    if (message.getNodeId() !== node.id)
                        return { type: "keep" };
                    // Resolve pings, so we don't need to send them (we know the node is awake)
                    if (NoOperationCC_1.messageIsPing(message))
                        return { type: "resolve", message: undefined };
                    // Re-queue all other transactions for this node, so they get added in front of the others
                    return { type: "requeue" };
                },
            });
        }
    }
    /** Is called when a node goes to sleep */
    onNodeSleep(node, oldStatus) {
        this.controllerLog.logNode(node.id, `The node is ${oldStatus === Types_1.NodeStatus.Unknown ? "" : "now "}asleep.`);
        // Move all its pending messages to the WakeupQueue
        // This clears the current transaction and continues sending the next messages
        this.moveMessagesToWakeupQueue(node.id);
    }
    /** Is called when a previously dead node starts communicating again */
    onNodeAlive(node, oldStatus) {
        this.controllerLog.logNode(node.id, `The node is ${oldStatus === Types_1.NodeStatus.Unknown ? "" : "now "}alive.`);
        if (oldStatus === Types_1.NodeStatus.Dead &&
            node.interviewStage !== Types_1.InterviewStage.Complete) {
            void this.interviewNode(node);
        }
    }
    /** Is called when a node is marked as dead */
    onNodeDead(node, oldStatus) {
        this.controllerLog.logNode(node.id, `The node is ${oldStatus === Types_1.NodeStatus.Unknown ? "" : "now "}dead.`);
        // This could mean that we need to ignore it in the all nodes ready check,
        // so perform the check again
        this.checkAllNodesReady();
    }
    /** Is called when a node is ready to be used */
    onNodeReady(node) {
        this._nodesReady.add(node.id);
        this.controllerLog.logNode(node.id, "The node is ready to be used");
        this.checkAllNodesReady();
    }
    /** Checks if all nodes are ready and emits the "all nodes ready" event if they are */
    checkAllNodesReady() {
        // Only emit "all nodes ready" once
        if (this._nodesReadyEventEmitted)
            return;
        for (const [id, node] of this.controller.nodes) {
            // Ignore dead nodes or the all nodes ready event will never be emitted without physical user interaction
            if (node.status === Types_1.NodeStatus.Dead)
                continue;
            if (!this._nodesReady.has(id))
                return;
        }
        // All nodes are ready
        this.controllerLog.print("All nodes are ready to be used");
        this.emit("all nodes ready");
        this._nodesReadyEventEmitted = true;
    }
    /** Is called when a node interview is completed */
    onNodeInterviewCompleted(node) {
        this.debounceSendNodeToSleep(node);
    }
    /** This is called when a new node has been added to the network */
    onNodeAdded(node) {
        this.addNodeEventHandlers(node);
        if (!this.options.skipInterview) {
            // Interview the node
            // don't await the interview, because it may take a very long time
            // if a node is asleep
            void this.interviewNode(node);
        }
    }
    /** This is called when a node was removed from the network */
    onNodeRemoved(node) {
        // Remove all listeners
        this.removeNodeEventHandlers(node);
        // purge node values from the DB
        node.valueDB.clear();
        this.rejectAllTransactionsForNode(node.id, "The node was removed from the network", core_1.ZWaveErrorCodes.Controller_NodeRemoved);
        // Asynchronously remove the node from all possible associations, ignore potential errors
        this.controller.removeNodeFromAllAssocations(node.id).catch((err) => {
            this.driverLog.print(`Failed to remove node ${node.id} from all associations: ${err.message}`, "error");
        });
        // Remove the node id from all cached neighbor lists and asynchronously make the affected nodes update their neighbor lists
        for (const otherNode of this.controller.nodes.values()) {
            if (otherNode !== node && otherNode.neighbors.includes(node.id)) {
                otherNode.removeNodeFromCachedNeighbors(node.id);
                otherNode.queryNeighborsInternal().catch((err) => {
                    this.driverLog.print(`Failed to update neighbors for node ${otherNode.id}: ${err.message}`, "warn");
                });
            }
        }
        // And clean up all remaining resources used by the node
        node.destroy();
        // If this was a failed node it could mean that all nodes are now ready
        this.checkAllNodesReady();
    }
    /** This is called when a node's firmware was updated */
    onNodeFirmwareUpdated(node, status, waitTime) {
        // Don't do this for non-successful updates
        if (status < commandclass_1.FirmwareUpdateStatus.OK_WaitingForActivation)
            return;
        // Wait at least 5 seconds
        if (!waitTime)
            waitTime = 5000;
        this.controllerLog.logNode(node.id, `Firmware updated, scheduling interview in ${waitTime} ms...`);
        // We reuse the retryNodeInterviewTimeouts here because they serve a similar purpose
        this.retryNodeInterviewTimeouts.set(node.id, setTimeout(() => {
            this.retryNodeInterviewTimeouts.delete(node.id);
            void node.refreshInfo();
        }, waitTime).unref());
    }
    /** Checks if there are any pending messages for the given node */
    hasPendingMessages(node) {
        const { queue, currentTransaction } = this.sendThread.state.context;
        return (!!queue.find((t) => t.message.getNodeId() === node.id) ||
            (currentTransaction === null || currentTransaction === void 0 ? void 0 : currentTransaction.message.getNodeId()) === node.id);
    }
    /**
     * Retrieves the maximum version of a command class the given endpoint supports.
     * Returns 0 when the CC is not supported. Also returns 0 when the node was not found.
     * Falls back to querying the root endpoint if an endpoint was not found on the node
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint in question
     */
    getSupportedCCVersionForEndpoint(cc, nodeId, endpointIndex = 0) {
        if (this._controller == undefined ||
            !this.controller.nodes.has(nodeId)) {
            return 0;
        }
        const node = this.controller.nodes.get(nodeId);
        const endpoint = node.getEndpoint(endpointIndex);
        if (endpoint)
            return endpoint.getCCVersion(cc);
        // We sometimes receive messages from an endpoint, but can't find that endpoint.
        // In that case fall back to the root endpoint to determine the supported version.
        return node.getCCVersion(cc);
    }
    /**
     * Retrieves the maximum version of a command class that can be used to communicate with a node.
     * Returns 1 if the node claims that it does not support a CC.
     * Throws if the CC is not implemented in this library yet.
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint for which the CC version should be retrieved
     */
    getSafeCCVersionForNode(cc, nodeId, endpointIndex = 0) {
        const supportedVersion = this.getSupportedCCVersionForEndpoint(cc, nodeId, endpointIndex);
        if (supportedVersion === 0) {
            // For unsupported CCs use version 1, no matter what
            return 1;
        }
        else {
            // For supported versions find the maximum version supported by both the
            // node and this library
            const implementedVersion = CommandClass_1.getImplementedVersion(cc);
            if (implementedVersion !== 0 &&
                implementedVersion !== Number.POSITIVE_INFINITY) {
                return Math.min(supportedVersion, implementedVersion);
            }
            throw new core_1.ZWaveError("Cannot retrieve the version of a CC that is not implemented", core_1.ZWaveErrorCodes.CC_NotSupported);
        }
    }
    /**
     * Performs a hard reset on the controller. This wipes out all configuration!
     *
     * The returned Promise resolves when the hard reset has been performed.
     * It does not wait for the initialization process which is started afterwards.
     */
    async hardReset() {
        this.ensureReady(true);
        // Calling ensureReady with true ensures that _controller is defined
        await this._controller.hardReset();
        // Clean up
        this.rejectTransactions(() => true, `The controller was hard-reset`);
        this.sendNodeToSleepTimers.forEach((timeout) => clearTimeout(timeout));
        this.sendNodeToSleepTimers.clear();
        this.retryNodeInterviewTimeouts.forEach((timeout) => clearTimeout(timeout));
        this.retryNodeInterviewTimeouts.clear();
        this._controllerInterviewed = false;
        void this.initializeControllerAndNodes();
    }
    /**
     * Ensures that the driver is ready to communicate (serial port open and not destroyed).
     * If desired, also checks that the controller interview has been completed.
     */
    ensureReady(includingController = false) {
        if (!this._wasStarted || !this._isOpen || this._wasDestroyed) {
            throw new core_1.ZWaveError("The driver is not ready or has been destroyed", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        if (includingController && !this._controllerInterviewed) {
            throw new core_1.ZWaveError("The controller is not ready yet", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
    }
    /** Indicates whether the driver is ready, i.e. the "driver ready" event was emitted */
    get ready() {
        return (this._wasStarted &&
            this._isOpen &&
            !this._wasDestroyed &&
            this._controllerInterviewed);
    }
    /**
     * Terminates the driver instance and closes the underlying serial connection.
     * Must be called under any circumstances.
     */
    async destroy() {
        var _a, _b, _c;
        // Ensure this is only called once
        if (this._wasDestroyed)
            return;
        this._wasDestroyed = true;
        this.driverLog.print("destroying driver instance...");
        // First stop the send thread machine and close the serial port, so nothing happens anymore
        if (this.sendThread.initialized)
            this.sendThread.stop();
        if (this.serial != undefined) {
            if (this.serial.isOpen)
                await this.serial.close();
            this.serial = undefined;
        }
        try {
            // Attempt to save the network to cache
            await this.saveNetworkToCacheInternal();
        }
        catch (e) {
            this.driverLog.print(`Saving the network to cache failed: ${e.message}`, "error");
        }
        try {
            // Attempt to close the value DBs
            await ((_a = this._valueDB) === null || _a === void 0 ? void 0 : _a.close());
            await ((_b = this._metadataDB) === null || _b === void 0 ? void 0 : _b.close());
        }
        catch (e) {
            this.driverLog.print(`Closing the value DBs failed: ${e.message}`, "error");
        }
        // Remove all timeouts
        for (const timeout of [
            this.saveToCacheTimer,
            ...this.sendNodeToSleepTimers.values(),
            ...this.retryNodeInterviewTimeouts.values(),
        ]) {
            if (timeout)
                clearTimeout(timeout);
        }
        // Destroy all nodes
        (_c = this._controller) === null || _c === void 0 ? void 0 : _c.nodes.forEach((n) => n.destroy());
        process.removeListener("exit", this._cleanupHandler);
        process.removeListener("SIGINT", this._cleanupHandler);
        process.removeListener("uncaughtException", this._cleanupHandler);
        // destroy loggers as the very last thing
        this.logContainer.destroy();
    }
    serialport_onError(err) {
        this.emit("error", err);
    }
    /**
     * Is called when the serial port has received a single-byte message or a complete message buffer
     */
    async serialport_onData(data) {
        var _a;
        if (typeof data === "number") {
            switch (data) {
                // single-byte messages - just forward them to the send thread
                case serial_1.MessageHeaders.ACK: {
                    this.sendThread.send("ACK");
                    return;
                }
                case serial_1.MessageHeaders.NAK: {
                    this.sendThread.send("NAK");
                    return;
                }
                case serial_1.MessageHeaders.CAN: {
                    this.sendThread.send("CAN");
                    return;
                }
            }
        }
        let msg;
        try {
            msg = Message_1.Message.from(this, data);
            // all good, send ACK
            await this.writeHeader(serial_1.MessageHeaders.ACK);
        }
        catch (e) {
            const response = this.handleDecodeError(e, data);
            if (response)
                await this.writeHeader(response);
        }
        // If the message could be decoded, forward it to the send thread
        if (msg) {
            if (ICommandClassContainer_1.isCommandClassContainer(msg)) {
                // SecurityCCCommandEncapsulationNonceGet is two commands in one, but
                // we're not set up to handle things like this. Reply to the nonce get
                // and handle the encapsulation part normally
                if (msg.command instanceof
                    SecurityCC_1.SecurityCCCommandEncapsulationNonceGet) {
                    void ((_a = msg.getNodeUnsafe()) === null || _a === void 0 ? void 0 : _a.handleSecurityNonceGet());
                }
                // Assemble partial CCs on the driver level. Only forward complete messages to the send thread machine
                if (!this.assemblePartialCCs(msg))
                    return;
            }
            this.driverLog.logMessage(msg, { direction: "inbound" });
            if (process.env.NODE_ENV !== "test") {
                // Enrich error data in case something goes wrong
                Sentry.addBreadcrumb({
                    category: "message",
                    timestamp: Date.now() / 1000,
                    type: "debug",
                    data: {
                        direction: "inbound",
                        msgType: msg.type,
                        functionType: msg.functionType,
                        name: msg.constructor.name,
                        nodeId: msg.getNodeId(),
                        ...msg.toLogEntry(),
                    },
                });
            }
            this.sendThread.send({ type: "message", message: msg });
        }
    }
    /** Handles a decoding error and returns the desired reply to the stick */
    handleDecodeError(e, data) {
        if (e instanceof core_1.ZWaveError) {
            switch (e.code) {
                case core_1.ZWaveErrorCodes.PacketFormat_Invalid:
                case core_1.ZWaveErrorCodes.PacketFormat_Checksum:
                case core_1.ZWaveErrorCodes.PacketFormat_Truncated:
                    this.driverLog.print(`Dropping message because it contains invalid data`, "warn");
                    return serial_1.MessageHeaders.NAK;
                case core_1.ZWaveErrorCodes.Deserialization_NotImplemented:
                case core_1.ZWaveErrorCodes.CC_NotImplemented:
                    this.driverLog.print(`Dropping message because it could not be deserialized: ${e.message}`, "warn");
                    return serial_1.MessageHeaders.ACK;
                case core_1.ZWaveErrorCodes.Driver_NotReady:
                    this.driverLog.print(`Dropping message because the driver is not ready to handle it yet.`, "warn");
                    return serial_1.MessageHeaders.ACK;
                case core_1.ZWaveErrorCodes.PacketFormat_InvalidPayload:
                    this.driverLog.print(`Dropping message with invalid data${typeof e.context === "string"
                        ? ` (Reason: ${e.context})`
                        : ""}:
0x${data.toString("hex")}`, "warn");
                    return serial_1.MessageHeaders.ACK;
                case core_1.ZWaveErrorCodes.Driver_NoSecurity:
                    this.driverLog.print(`Dropping message because network key is not set or the driver is not yet ready to receive secure messages.`, "warn");
                    return serial_1.MessageHeaders.ACK;
            }
        }
        else {
            if (/database is not open/.test(e.message)) {
                // The JSONL-DB is not open yet
                this.driverLog.print(`Dropping message because the driver is not ready to handle it yet.`, "warn");
                return serial_1.MessageHeaders.ACK;
            }
        }
        // Pass all other errors through
        throw e;
    }
    /** Checks if a transaction failed because a node didn't respond in time */
    isMissingNodeACK(transaction, e) {
        return (
        // If the node does not acknowledge our request, it is either asleep or dead
        e.code === core_1.ZWaveErrorCodes.Controller_CallbackNOK &&
            transaction.message instanceof SendDataMessages_1.SendDataRequest &&
            // Ignore pre-transmit handshakes because the actual transaction will be retried
            transaction.priority !== Constants_1.MessagePriority.PreTransmitHandshake);
    }
    /**
     * Handles the case that a node failed to respond in time.
     * Returns `true` if the transaction failure was handled, `false` if it needs to be rejected.
     */
    handleMissingNodeACK(transaction) {
        const node = transaction.message.getNodeUnsafe();
        if (!node)
            return false; // This should never happen, but whatever
        if (!transaction.changeNodeStatusOnTimeout) {
            // The sender of this transaction doesn't want it to change the status of the node
            return false;
        }
        else if (node.supportsCC(core_1.CommandClasses["Wake Up"])) {
            this.controllerLog.logNode(node.id, `The node did not respond after ${transaction.message.maxSendAttempts} attempts.
It is probably asleep, moving its messages to the wakeup queue.`, "warn");
            // Mark the node as asleep
            // The handler for the asleep status will move the messages to the wakeup queue
            // We need to re-add the current transaction if that is allowed because otherwise it will be dropped silently
            if (this.mayMoveToWakeupQueue(transaction)) {
                this.sendThread.send({ type: "add", transaction });
            }
            else {
                transaction.promise.reject(new core_1.ZWaveError(`The node is asleep`, core_1.ZWaveErrorCodes.Controller_MessageDropped));
            }
            node.markAsAsleep();
            return true;
        }
        else {
            const errorMsg = `Node ${node.id} did not respond after ${transaction.message.maxSendAttempts} attempts, it is presumed dead`;
            this.controllerLog.logNode(node.id, errorMsg, "warn");
            node.markAsDead();
            this.rejectAllTransactionsForNode(node.id, errorMsg);
            // The above call will reject the transaction, no need to do it again
            return false;
        }
    }
    /**
     * Assembles partial CCs of in a message body. Returns `true` when the message is complete and can be handled further.
     * If the message expects another partial one, this returns `false`.
     */
    assemblePartialCCs(msg) {
        let command = msg.command;
        let sessionId;
        // We search for the every CC that provides us with a session ID
        // There might be newly-completed CCs that contain a partial CC,
        // so investigate the entire CC encapsulation stack.
        while (true) {
            sessionId = command.getPartialCCSessionId();
            if (sessionId) {
                // This CC belongs to a partial session
                const partialSessionKey = JSON.stringify({
                    nodeId: msg.getNodeId(),
                    ccId: msg.command.ccId,
                    ccCommand: msg.command.ccCommand,
                    ...sessionId,
                });
                if (!this.partialCCSessions.has(partialSessionKey)) {
                    this.partialCCSessions.set(partialSessionKey, []);
                }
                const session = this.partialCCSessions.get(partialSessionKey);
                if (command.expectMoreMessages()) {
                    // this is not the final one, store it
                    session.push(command);
                    // and don't handle the command now
                    this.driverLog.logMessage(msg, {
                        secondaryTags: ["partial"],
                        direction: "inbound",
                    });
                    return false;
                }
                else {
                    // this is the final one, merge the previous responses
                    this.partialCCSessions.delete(partialSessionKey);
                    try {
                        command.mergePartialCCs(session);
                    }
                    catch (e) {
                        if (e instanceof core_1.ZWaveError) {
                            switch (e.code) {
                                case core_1.ZWaveErrorCodes.Deserialization_NotImplemented:
                                case core_1.ZWaveErrorCodes.CC_NotImplemented:
                                    this.driverLog.print(`Dropping message because it could not be deserialized: ${e.message}`, "warn");
                                    // Don't continue handling this message
                                    return false;
                                case core_1.ZWaveErrorCodes.PacketFormat_InvalidPayload:
                                    this.driverLog.print(`Could not assemble partial CCs because the payload is invalid. Dropping them.`, "warn");
                                    // Don't continue handling this message
                                    return false;
                                case core_1.ZWaveErrorCodes.Driver_NotReady:
                                    this.driverLog.print(`Could not assemble partial CCs because the driver is not ready yet. Dropping them`, "warn");
                                    // Don't continue handling this message
                                    return false;
                            }
                        }
                        throw e;
                    }
                    // Assembling this CC was successful - but it might contain another partial CC
                }
            }
            else {
                // No partial CC, just continue
            }
            // If this is an encapsulating CC, we need to look one level deeper
            if (EncapsulatingCommandClass_1.isEncapsulatingCommandClass(command)) {
                command = command.encapsulated;
            }
            else {
                break;
            }
        }
        return true;
    }
    /**
     * Is called when a message is received that does not belong to any ongoing transactions
     * @param msg The decoded message
     */
    async handleUnsolicitedMessage(msg) {
        if (msg.type === Constants_1.MessageType.Request) {
            // This is a request we might have registered handlers for
            try {
                await this.handleRequest(msg);
            }
            catch (e) {
                if (e instanceof core_1.ZWaveError &&
                    e.code === core_1.ZWaveErrorCodes.Driver_NotReady) {
                    this.driverLog.print(`Cannot handle message because the driver is not ready to handle it yet.`, "warn");
                }
                else {
                    throw e;
                }
            }
        }
        else {
            this.driverLog.transactionResponse(msg, undefined, "unexpected");
            this.driverLog.print("unexpected response, discarding...", "warn");
        }
    }
    /**
     * Registers a handler for messages that are not handled by the driver as part of a message exchange.
     * The handler function needs to return a boolean indicating if the message has been handled.
     * Registered handlers are called in sequence until a handler returns `true`.
     *
     * @param fnType The function type to register the handler for
     * @param handler The request handler callback
     * @param oneTime Whether the handler should be removed after its first successful invocation
     */
    registerRequestHandler(fnType, handler, oneTime = false) {
        const handlers = this.requestHandlers.has(fnType)
            ? this.requestHandlers.get(fnType)
            : [];
        const entry = { invoke: handler, oneTime };
        handlers.push(entry);
        this.driverLog.print(`added${oneTime ? " one-time" : ""} request handler for ${Constants_1.FunctionType[fnType]} (${shared_1.num2hex(fnType)})...
${handlers.length} registered`);
        this.requestHandlers.set(fnType, handlers);
    }
    /**
     * Unregisters a message handler that has been added with `registerRequestHandler`
     * @param fnType The function type to unregister the handler for
     * @param handler The previously registered request handler callback
     */
    unregisterRequestHandler(fnType, handler) {
        const handlers = this.requestHandlers.has(fnType)
            ? this.requestHandlers.get(fnType)
            : [];
        for (let i = 0, entry = handlers[i]; i < handlers.length; i++) {
            // remove the handler if it was found
            if (entry.invoke === handler) {
                handlers.splice(i, 1);
                break;
            }
        }
        this.driverLog.print(`removed request handler for ${Constants_1.FunctionType[fnType]} (${fnType})...
${handlers.length} left`);
        this.requestHandlers.set(fnType, handlers);
    }
    /**
     * Checks whether a CC may be handled or should be ignored.
     * This method expects `cc` to be unwrapped.
     */
    mayHandleUnsolicitedCommand(cc) {
        // This should only be necessary for unsolicited commands, since the response matching
        // is pretty strict and looks at the encapsulation order
        // From SDS11847:
        // A controlling node MUST discard a received Report/Notification type command if it is
        // not received using S0 encapsulation and the corresponding Command Class is supported securely only
        if (cc.secure && cc.ccId !== core_1.CommandClasses.Security) {
            const commandName = cc.constructor.name;
            if (commandName.endsWith("Report") ||
                commandName.endsWith("Notification")) {
                // Check whether there was a S0 encapsulation
                if (cc.isEncapsulatedWith(core_1.CommandClasses.Security))
                    return true;
                // none found, don't accept the CC
                this.controllerLog.logNode(cc.nodeId, `command must be encrypted but was received without Security encapsulation - discarding it...`, "warn");
                return false;
            }
        }
        return true;
    }
    /**
     * Is called when a Request-type message was received
     */
    async handleRequest(msg) {
        let handlers;
        // For further actions, we are only interested in the innermost CC
        if (ICommandClassContainer_1.isCommandClassContainer(msg))
            this.unwrapCommands(msg);
        if (INodeQuery_1.isNodeQuery(msg) || ICommandClassContainer_1.isCommandClassContainer(msg)) {
            const node = msg.getNodeUnsafe();
            if (node) {
                // We have received an unsolicited message from a dead node, bring it back to life
                if (node.status === Types_1.NodeStatus.Dead) {
                    node.markAsAlive();
                }
            }
        }
        if (ICommandClassContainer_1.isCommandClassContainer(msg)) {
            const node = msg.getNodeUnsafe();
            // If we receive an encrypted message but assume the node is insecure, change our assumption
            if ((node === null || node === void 0 ? void 0 : node.isSecure) === false &&
                (msg.command.ccId === core_1.CommandClasses.Security ||
                    msg.command.isEncapsulatedWith(core_1.CommandClasses.Security))) {
                node.isSecure = true;
                // Force a new interview
                void node.refreshInfo();
            }
            // Check if we may even handle the command
            if (!this.mayHandleUnsolicitedCommand(msg.command))
                return;
        }
        if (msg instanceof ApplicationCommandRequest_1.ApplicationCommandRequest ||
            msg instanceof BridgeApplicationCommandRequest_1.BridgeApplicationCommandRequest) {
            // we handle ApplicationCommandRequests differently because they are handled by the nodes directly
            const nodeId = msg.command.nodeId;
            // cannot handle ApplicationCommandRequests without a controller
            if (this._controller == undefined) {
                this.driverLog.print(`  the controller is not ready yet, discarding...`, "warn");
                return;
            }
            else if (!this.controller.nodes.has(nodeId)) {
                this.driverLog.print(`  the node is unknown or not initialized yet, discarding...`, "warn");
                return;
            }
            const node = this.controller.nodes.get(nodeId);
            // Check if we need to handle the command ourselves
            if (msg.command.ccId === core_1.CommandClasses["Device Reset Locally"] &&
                msg.command instanceof DeviceResetLocallyCC_1.DeviceResetLocallyCCNotification) {
                this.controllerLog.logNode(msg.command.nodeId, {
                    message: `The node was reset locally, removing it`,
                    direction: "inbound",
                });
                if (!(await this.controller.isFailedNode(msg.command.nodeId))) {
                    try {
                        // Force a ping of the node, so it gets added to the failed nodes list
                        node.markAsAwake();
                        await node.commandClasses["No Operation"].send();
                    }
                    catch (_a) {
                        // this is expected
                    }
                }
                try {
                    // ...because we can only remove failed nodes
                    await this.controller.removeFailedNode(msg.command.nodeId);
                }
                catch (e) {
                    this.controllerLog.logNode(msg.command.nodeId, {
                        message: `removing the node failed: ${e}`,
                        level: "error",
                    });
                }
            }
            else if (msg.command.ccId === core_1.CommandClasses.Supervision &&
                msg.command instanceof SupervisionCC_1.SupervisionCCReport &&
                this.supervisionSessions.has(msg.command.sessionId)) {
                // Supervision commands are handled here
                this.controllerLog.logNode(msg.command.nodeId, {
                    message: `Received update for a Supervision session`,
                    direction: "inbound",
                });
                // Call the update handler
                this.supervisionSessions.get(msg.command.sessionId)(msg.command.status, msg.command.duration);
                // If this was a final report, remove the handler
                if (!msg.command.moreUpdatesFollow) {
                    this.supervisionSessions.delete(msg.command.sessionId);
                }
            }
            else {
                // check if someone is waiting for this command
                for (const entry of this.awaitedCommands) {
                    if (entry.predicate(msg.command)) {
                        // resolve the promise - this will remove the entry from the list
                        entry.promise.resolve(msg.command);
                        return;
                    }
                }
                // noone is waiting, dispatch the command to the node itself
                await node.handleCommand(msg.command);
            }
            return;
        }
        else if (msg instanceof ApplicationUpdateRequest_1.ApplicationUpdateRequest) {
            if (msg instanceof ApplicationUpdateRequest_1.ApplicationUpdateRequestNodeInfoReceived) {
                const node = msg.getNodeUnsafe();
                if (node) {
                    this.controllerLog.logNode(node.id, {
                        message: "Received updated node info",
                        direction: "inbound",
                    });
                    node.updateNodeInfo(msg.nodeInformation);
                    // Tell the send thread that we received a NIF from the node
                    this.sendThread.send({ type: "NIF", nodeId: node.id });
                    return;
                }
            }
        }
        else {
            // TODO: This deserves a nicer formatting
            this.driverLog.print(`handling request ${Constants_1.FunctionType[msg.functionType]} (${msg.functionType})`);
            handlers = this.requestHandlers.get(msg.functionType);
        }
        if (handlers != undefined && handlers.length > 0) {
            this.driverLog.print(`  ${handlers.length} handler${handlers.length !== 1 ? "s" : ""} registered!`);
            // loop through all handlers and find the first one that returns true to indicate that it handled the message
            for (let i = 0; i < handlers.length; i++) {
                this.driverLog.print(`  invoking handler #${i}`);
                // Invoke the handler and remember its result
                const handler = handlers[i];
                let handlerResult = handler.invoke(msg);
                if (handlerResult instanceof Promise) {
                    handlerResult = await handlerResult;
                }
                if (handlerResult) {
                    this.driverLog.print(`    the message was handled`);
                    if (handler.oneTime) {
                        this.driverLog.print("  one-time handler was successfully called, removing it...");
                        handlers.splice(i, 1);
                    }
                    // don't invoke any more handlers
                    break;
                }
            }
        }
        else {
            this.driverLog.print("  no handlers registered!", "warn");
        }
    }
    /**
     * Returns the next callback ID. Callback IDs are used to correllate requests
     * to the controller/nodes with its response
     */
    getNextCallbackId() {
        this.lastCallbackId = (this.lastCallbackId + 1) & 0xff;
        if (this.lastCallbackId < 1)
            this.lastCallbackId = 1;
        return this.lastCallbackId;
    }
    encapsulateCommands(msg) {
        // The encapsulation order (from outside to inside) is as follows:
        // 5. Any one of the following combinations:
        //   a. Security (S0 or S2) followed by transport service
        //   b. Transport Service
        //   c. Security (S0 or S2)
        //   d. CRC16
        // b and d are mutually exclusive, security is not
        // 4. Multi Channel
        // 3. Supervision
        // 2. Multi Command
        // 1. Encapsulated Command Class (payload), e.g. Basic Set
        // TODO: 2.
        // 3.
        if (SupervisionCC_1.SupervisionCC.requiresEncapsulation(msg.command)) {
            msg.command = SupervisionCC_1.SupervisionCC.encapsulate(this, msg.command);
        }
        // 4.
        if (MultiChannelCC_1.MultiChannelCC.requiresEncapsulation(msg.command)) {
            msg.command = MultiChannelCC_1.MultiChannelCC.encapsulate(this, msg.command);
        }
        // 5.
        if (SecurityCC_1.SecurityCC.requiresEncapsulation(msg.command)) {
            msg.command = SecurityCC_1.SecurityCC.encapsulate(this, msg.command);
        }
    }
    unwrapCommands(msg) {
        // Unwrap encapsulating CCs until we get to the core
        while (EncapsulatingCommandClass_1.isEncapsulatingCommandClass(msg.command) ||
            EncapsulatingCommandClass_1.isMultiEncapsulatingCommandClass(msg.command)) {
            const unwrapped = msg.command.encapsulated;
            if (typeguards_1.isArray(unwrapped)) {
                this.driverLog.print(`Received a command that contains multiple CommandClasses. This is not supported yet! Discarding the message...`, "warn");
                return;
            }
            msg.command = unwrapped;
        }
    }
    // wotan-disable no-misused-generics
    /**
     * Sends a message to the Z-Wave stick.
     * @param msg The message to send
     * @param options (optional) Options regarding the message transmission
     */
    async sendMessage(msg, options = {}) {
        this.ensureReady();
        let node;
        // Don't send messages to dead nodes
        if (INodeQuery_1.isNodeQuery(msg) || ICommandClassContainer_1.isCommandClassContainer(msg)) {
            node = msg.getNodeUnsafe();
            if (!NoOperationCC_1.messageIsPing(msg) && (node === null || node === void 0 ? void 0 : node.status) === Types_1.NodeStatus.Dead) {
                // Instead of throwing immediately, try to ping the node first - if it responds, continue
                if (!(await node.ping())) {
                    throw new core_1.ZWaveError(`The message cannot be sent because node ${node.id} is dead`, core_1.ZWaveErrorCodes.Controller_MessageDropped);
                }
            }
        }
        if (options.priority == undefined)
            options.priority = Message_1.getDefaultPriority(msg);
        if (options.priority == undefined) {
            const className = msg.constructor.name;
            const msgTypeName = Constants_1.FunctionType[msg.functionType];
            throw new core_1.ZWaveError(`No default priority has been defined for ${className} (${msgTypeName}), so you have to provide one for your message`, core_1.ZWaveErrorCodes.Driver_NoPriority);
        }
        if (options.supportCheck == undefined)
            options.supportCheck = true;
        if (options.supportCheck &&
            this._controller != undefined &&
            !this._controller.isFunctionSupported(msg.functionType)) {
            throw new core_1.ZWaveError(`Your hardware does not support the ${Constants_1.FunctionType[msg.functionType]} function`, core_1.ZWaveErrorCodes.Driver_NotSupported);
        }
        // When sending a message to a node that is known to be sleeping,
        // the priority must be WakeUp, so the message gets deprioritized
        // in comparison with messages to awake nodes.
        // However there are a few exceptions...
        if (!!node &&
            // Pings can be used to check if a node is really asleep, so they should be sent regardless
            !NoOperationCC_1.messageIsPing(msg) &&
            // Nodes that are asleep should have their messages queued for wakeup
            (!node.isAwake() ||
                // Nodes that can sleep should use the WakeUp priority so their messages
                // get handled immediately
                node.supportsCC(core_1.CommandClasses["Wake Up"])) &&
            // If we move multicasts to the wakeup queue, it is unlikely
            // that there is ever a points where all targets are awake
            !(msg instanceof SendDataMessages_1.SendDataMulticastRequest) &&
            // Handshake messages are meant to be sent immediately
            options.priority !== Constants_1.MessagePriority.Handshake &&
            options.priority !== Constants_1.MessagePriority.PreTransmitHandshake) {
            options.priority = Constants_1.MessagePriority.WakeUp;
        }
        // create the transaction and enqueue it
        const promise = deferred_promise_1.createDeferredPromise();
        const transaction = new Transaction_1.Transaction(this, msg, promise, options.priority);
        if (options.changeNodeStatusOnMissingACK != undefined) {
            transaction.changeNodeStatusOnTimeout =
                options.changeNodeStatusOnMissingACK;
        }
        transaction.tag = options.tag;
        // start sending now (maybe)
        this.sendThread.send({ type: "add", transaction });
        // If the transaction should expire, start the timeout
        let expirationTimeout;
        if (options.expire) {
            expirationTimeout = setTimeout(() => {
                this.sendThread.send({
                    type: "reduce",
                    reducer: (t) => {
                        if (t === transaction)
                            return {
                                type: "reject",
                                message: `The message has expired`,
                                code: core_1.ZWaveErrorCodes.Controller_MessageExpired,
                            };
                        return { type: "keep" };
                    },
                });
            }, options.expire).unref();
        }
        try {
            const ret = await promise;
            if (expirationTimeout)
                clearTimeout(expirationTimeout);
            if (node) {
                if (node.supportsCC(core_1.CommandClasses["Wake Up"])) {
                    // If the node is not meant to be kept awake, try to send it back to sleep
                    if (!node.keepAwake) {
                        this.debounceSendNodeToSleep(node);
                    }
                    // The node must be awake because it answered
                    node.markAsAwake();
                }
                else if (node.status !== Types_1.NodeStatus.Alive) {
                    // The node status was unknown or dead - in either case it must be alive because it answered
                    node.markAsAlive();
                }
            }
            return ret;
        }
        catch (e) {
            if (e instanceof core_1.ZWaveError) {
                if (
                // If a controller command failed (that is not SendData), pass the response/callback through
                (e.code === core_1.ZWaveErrorCodes.Controller_ResponseNOK ||
                    e.code === core_1.ZWaveErrorCodes.Controller_CallbackNOK) &&
                    e.context instanceof Message_1.Message &&
                    e.context.functionType !== Constants_1.FunctionType.SendData &&
                    e.context.functionType !== Constants_1.FunctionType.SendDataMulticast) {
                    return e.context;
                }
            }
            throw e;
        }
    }
    // wotan-enable no-misused-generics
    /**
     * Sends a command to a Z-Wave node. If the node returns a command in response, that command will be the return value.
     * If the command expects no response **or** the response times out, nothing will be returned
     * @param command The command to send. It will be encapsulated in a SendData[Multicast]Request.
     * @param options (optional) Options regarding the message transmission
     */
    // wotan-disable-next-line no-misused-generics
    async sendCommand(command, options = {}) {
        let msg;
        if (command.isSinglecast()) {
            msg = new SendDataMessages_1.SendDataRequest(this, { command });
        }
        else if (command.isMulticast()) {
            msg = new SendDataMessages_1.SendDataMulticastRequest(this, { command });
        }
        else {
            throw new core_1.ZWaveError(`A CC must either be singlecast or multicast`, core_1.ZWaveErrorCodes.Argument_Invalid);
        }
        // Specify the number of send attempts for the request
        if (options.maxSendAttempts != undefined) {
            msg.maxSendAttempts = options.maxSendAttempts;
        }
        // Automatically encapsulate commands before sending
        this.encapsulateCommands(msg);
        try {
            const resp = await this.sendMessage(msg, options);
            // And unwrap the response if there was any
            if (ICommandClassContainer_1.isCommandClassContainer(resp)) {
                this.unwrapCommands(resp);
                return resp.command;
            }
        }
        catch (e) {
            // A timeout always has to be expected. In this case return nothing.
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout) {
                if (command.isSinglecast()) {
                    this.controllerLog.logNode(command.nodeId, e.message, "warn");
                }
            }
            else {
                // We don't want to swallow any other errors
                throw e;
            }
        }
    }
    /**
     * Sends a supervised command to a Z-Wave node. When status updates are requested, the passed callback will be executed for every non-final update.
     * @param command The command to send
     * @param options (optional) Options regarding the message transmission
     */
    async sendSupervisedCommand(command, options = {
        requestStatusUpdates: false,
    }) {
        var _a;
        // Check if the target supports this command
        if (!((_a = command.getNode()) === null || _a === void 0 ? void 0 : _a.supportsCC(core_1.CommandClasses.Supervision))) {
            throw new core_1.ZWaveError(`Node ${command.nodeId} does not support the Supervision CC!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        // Create the encapsulating CC so we have a session ID
        command = SupervisionCC_1.SupervisionCC.encapsulate(this, command, options.requestStatusUpdates);
        const resp = await this.sendCommand(command, options);
        if (!resp)
            return;
        // If future updates are expected, listen for them
        if (options.requestStatusUpdates && resp.moreUpdatesFollow) {
            this.supervisionSessions.set(command.sessionId, options.onUpdate);
        }
        // In any case, return the status
        return {
            status: resp.status,
            remainingDuration: resp.duration,
        };
    }
    /**
     * Sends a supervised command to a Z-Wave node if the Supervision CC is supported. If not, a normal command is sent.
     * This does not return any Report values, so only use this for Set-type commands.
     *
     * @param command The command to send
     * @param options (optional) Options regarding the message transmission
     */
    async trySendCommandSupervised(command, options) {
        var _a;
        if ((_a = command.getNode()) === null || _a === void 0 ? void 0 : _a.supportsCC(core_1.CommandClasses.Supervision)) {
            return this.sendSupervisedCommand(command, options);
        }
        else {
            await this.sendCommand(command, options);
        }
    }
    /**
     * Sends a low-level message like ACK, NAK or CAN immediately
     * @param message The low-level message to send
     */
    writeHeader(header) {
        // ACK, CAN, NAK
        return this.writeSerial(Buffer.from([header]));
    }
    /** Sends a raw datagram to the serialport (if that is open) */
    async writeSerial(data) {
        var _a;
        return (_a = this.serial) === null || _a === void 0 ? void 0 : _a.writeAsync(data);
    }
    /**
     * Waits until a command is received or a timeout has elapsed. Returns the received command.
     * @param timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected
     * @param predicate A predicate function to test all incoming command classes
     */
    // wotan-disable-next-line no-misused-generics
    waitForCommand(predicate, timeout) {
        return new Promise((resolve, reject) => {
            const entry = {
                predicate,
                promise: deferred_promise_1.createDeferredPromise(),
                timeout: undefined,
            };
            this.awaitedCommands.push(entry);
            const removeEntry = () => {
                if (entry.timeout)
                    clearTimeout(entry.timeout);
                const index = this.awaitedCommands.indexOf(entry);
                if (index !== -1)
                    this.awaitedCommands.splice(index, 1);
            };
            // When the timeout elapses, remove the wait entry and reject the returned Promise
            entry.timeout = setTimeout(() => {
                removeEntry();
                reject(new core_1.ZWaveError(`Received no matching command within the provided timeout!`, core_1.ZWaveErrorCodes.Controller_NodeTimeout));
            }, timeout).unref();
            // When the promise is resolved, remove the wait entry and resolve the returned Promise
            void entry.promise.then((cc) => {
                removeEntry();
                resolve(cc);
            });
        });
    }
    /** Checks if a message is allowed to go into the wakeup queue */
    mayMoveToWakeupQueue(transaction) {
        const msg = transaction.message;
        switch (true) {
            // Pings and handshake responses will block the send queue until wakeup,
            // so they must be dropped
            case NoOperationCC_1.messageIsPing(msg):
            case transaction.priority === Constants_1.MessagePriority.Handshake:
            // Outgoing handshake requests are very likely not valid after wakeup, so drop them too
            case transaction.priority === Constants_1.MessagePriority.PreTransmitHandshake:
            // We also don't want to immediately send the node to sleep when it wakes up
            case ICommandClassContainer_1.isCommandClassContainer(msg) &&
                msg.command instanceof WakeUpCC_1.WakeUpCCNoMoreInformation:
            // compat queries because they will be recreated when the node wakes up
            case transaction.tag === "compat":
                return false;
        }
        return true;
    }
    /** Moves all messages for a given node into the wakeup queue */
    moveMessagesToWakeupQueue(nodeId) {
        const reject = {
            type: "reject",
            message: `The node is asleep`,
            code: core_1.ZWaveErrorCodes.Controller_MessageDropped,
        };
        const requeue = {
            type: "requeue",
            priority: Constants_1.MessagePriority.WakeUp,
        };
        const reducer = (transaction, _source) => {
            const msg = transaction.message;
            if (msg.getNodeId() !== nodeId)
                return { type: "keep" };
            // Drop all messages that are not allowed in the wakeup queue
            // For all other messages, change the priority to wakeup
            return this.mayMoveToWakeupQueue(transaction) ? requeue : reject;
        };
        // Apply the reducer to the send thread
        this.sendThread.send({ type: "reduce", reducer });
    }
    /**
     * @internal
     * Rejects all pending transactions that match a predicate and removes them from the send queue
     */
    rejectTransactions(predicate, errorMsg = `The message has been removed from the queue`, errorCode = core_1.ZWaveErrorCodes.Controller_MessageDropped) {
        const reducer = (transaction) => {
            if (predicate(transaction)) {
                return {
                    type: "reject",
                    message: errorMsg,
                    code: errorCode,
                };
            }
            else {
                return { type: "keep" };
            }
        };
        // Apply the reducer to the send thread
        this.sendThread.send({ type: "reduce", reducer });
    }
    /**
     * @internal
     * Rejects all pending transactions for a node and removes them from the send queue
     */
    rejectAllTransactionsForNode(nodeId, errorMsg = `The node is dead`, errorCode = core_1.ZWaveErrorCodes.Controller_MessageDropped) {
        this.rejectTransactions((t) => t.message.getNodeId() === nodeId, errorMsg, errorCode);
    }
    /** Re-sorts the send queue */
    sortSendQueue() {
        this.sendThread.send("sortQueue");
    }
    /**
     * Does the work for saveNetworkToCache. This is not throttled, so any call
     * to this method WILL save the network.
     */
    async saveNetworkToCacheInternal() {
        if (!this._controller || !this.controller.homeId)
            return;
        await this.options.storage.driver.ensureDir(this.cacheDir);
        const cacheFile = path_1.default.join(this.cacheDir, this.controller.homeId.toString(16) + ".json");
        const serializedObj = this.controller.serialize();
        const jsonString = shared_1.stringify(serializedObj);
        await this.options.storage.driver.writeFile(cacheFile, jsonString, "utf8");
    }
    /**
     * Saves the current configuration and collected data about the controller and all nodes to a cache file.
     * For performance reasons, these calls may be throttled.
     */
    async saveNetworkToCache() {
        // TODO: Detect if the network needs to be saved at all
        if (!this._controller || !this.controller.homeId)
            return;
        // Ensure this method isn't being executed too often
        if (this.isSavingToCache ||
            Date.now() - this.lastSaveToCache < this.saveToCacheInterval) {
            // Schedule a save in a couple of ms to collect changes
            if (!this.saveToCacheTimer) {
                this.saveToCacheTimer = setTimeout(() => void this.saveNetworkToCache(), this.saveToCacheInterval);
            }
            return;
        }
        else {
            this.saveToCacheTimer = undefined;
        }
        this.isSavingToCache = true;
        await this.saveNetworkToCacheInternal();
        this.isSavingToCache = false;
        this.lastSaveToCache = Date.now();
    }
    /**
     * Restores a previously stored Z-Wave network state from cache to speed up the startup process
     */
    async restoreNetworkStructureFromCache() {
        if (!this._controller || !this.controller.homeId)
            return;
        const cacheFile = path_1.default.join(this.cacheDir, `${this.controller.homeId.toString(16)}.json`);
        if (!(await this.options.storage.driver.pathExists(cacheFile)))
            return;
        try {
            this.driverLog.print(`Cache file for homeId ${shared_1.num2hex(this.controller.homeId)} found, attempting to restore the network from cache...`);
            const cacheString = await this.options.storage.driver.readFile(cacheFile, "utf8");
            await this.controller.deserialize(JSON.parse(cacheString));
            this.driverLog.print(`Restoring the network from cache was successful!`);
        }
        catch (e) {
            const message = `Restoring the network from cache failed: ${e}`;
            this.emit("error", new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Driver_InvalidCache));
            this.driverLog.print(message, "error");
        }
    }
    /**
     * @internal
     * Marks a node for a later sleep command. Every call refreshes the period until the node actually goes to sleep
     */
    debounceSendNodeToSleep(node) {
        // TODO: This should be a single command to the send thread
        // Delete old timers if any exist
        if (this.sendNodeToSleepTimers.has(node.id)) {
            clearTimeout(this.sendNodeToSleepTimers.get(node.id));
        }
        // Sends a node to sleep if it has no more messages.
        const sendNodeToSleep = (node) => {
            this.sendNodeToSleepTimers.delete(node.id);
            if (!this.hasPendingMessages(node)) {
                void node.sendNoMoreInformation().catch(() => {
                    /* ignore errors */
                });
            }
        };
        // If a sleeping node has no messages pending, we may send it back to sleep
        if (node.supportsCC(core_1.CommandClasses["Wake Up"]) &&
            !this.hasPendingMessages(node)) {
            this.sendNodeToSleepTimers.set(node.id, setTimeout(() => sendNodeToSleep(node), 1000).unref());
        }
    }
    /** Computes the maximum net CC payload size for the given CC or SendDataRequest */
    computeNetCCPayloadSize(commandOrMsg) {
        // Recreate the correct encapsulation structure
        const msg = commandOrMsg instanceof SendDataMessages_1.SendDataRequest
            ? commandOrMsg
            : new SendDataMessages_1.SendDataRequest(this, { command: commandOrMsg });
        this.encapsulateCommands(msg);
        return msg.command.getMaxPayloadLength(msg.getMaxPayloadLength());
    }
    // This does not all need to be printed to the console
    [util.inspect.custom]() {
        return "[Driver]";
    }
}
exports.Driver = Driver;

//# sourceMappingURL=Driver.js.map
