"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Endpoint = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const CommandClass_1 = require("../commandclass/CommandClass");
const ZWavePlusCC_1 = require("../commandclass/ZWavePlusCC");
/**
 * Represents a physical endpoint of a Z-Wave node. This can either be the root
 * device itself (index 0) or a more specific endpoint like a single plug.
 *
 * Each endpoint may have different capabilities (supported/controlled CCs)
 */
class Endpoint {
    constructor(
    /** The id of the node this endpoint belongs to */
    nodeId, 
    /** The driver instance this endpoint belongs to */
    driver, 
    /** The index of this endpoint. 0 for the root device, 1+ otherwise */
    index, supportedCCs) {
        this.nodeId = nodeId;
        this.driver = driver;
        this.index = index;
        this._implementedCommandClasses = new Map();
        this._commandClassAPIs = new Map();
        this._commandClassAPIsProxy = new Proxy(this._commandClassAPIs, {
            get: (target, ccNameOrId) => {
                // Avoid ultra-weird error messages during testing
                if (process.env.NODE_ENV === "test" &&
                    typeof ccNameOrId === "string" &&
                    (ccNameOrId === "$$typeof" ||
                        ccNameOrId === "constructor" ||
                        ccNameOrId.includes("@@__IMMUTABLE"))) {
                    return undefined;
                }
                if (typeof ccNameOrId === "symbol") {
                    // Allow access to the iterator symbol
                    if (ccNameOrId === Symbol.iterator) {
                        return this.commandClassesIterator;
                    }
                    else if (ccNameOrId === Symbol.toStringTag) {
                        return "[object Object]";
                    }
                    // ignore all other symbols
                    return undefined;
                }
                else {
                    // typeof ccNameOrId === "string"
                    let ccId;
                    // The command classes are exposed to library users by their name or the ID
                    if (/\d+/.test(ccNameOrId)) {
                        // Since this is a property accessor, ccNameOrID is passed as a string,
                        // even when it was a number (CommandClasses)
                        ccId = +ccNameOrId;
                    }
                    else {
                        // If a name was given, retrieve the corresponding ID
                        ccId = core_1.CommandClasses[ccNameOrId];
                        if (ccId == undefined) {
                            throw new core_1.ZWaveError(`Command Class ${ccNameOrId} is not implemented! If you are sure that the name/id is correct, consider opening an issue at https://github.com/AlCalzone/node-zwave-js`, core_1.ZWaveErrorCodes.CC_NotImplemented);
                        }
                    }
                    // When accessing a CC API for the first time, we need to create it
                    if (!target.has(ccId)) {
                        const api = this.createAPI(ccId);
                        target.set(ccId, api);
                    }
                    return target.get(ccId);
                }
            },
        });
        /**
         * Used to iterate over the commandClasses API without throwing errors by accessing unsupported CCs
         */
        this.commandClassesIterator = function* () {
            for (const cc of this.implementedCommandClasses.keys()) {
                if (this.supportsCC(cc))
                    yield this.commandClasses[cc];
            }
        }.bind(this);
        if (supportedCCs != undefined) {
            for (const cc of supportedCCs) {
                this.addCC(cc, { isSupported: true });
            }
        }
    }
    /** Resets all stored information of this endpoint */
    reset() {
        this._implementedCommandClasses.clear();
        this._commandClassAPIs.clear();
    }
    /**
     * @internal
     * Information about the implemented Command Classes of this endpoint.
     */
    get implementedCommandClasses() {
        return this._implementedCommandClasses;
    }
    /**
     * Adds a CC to the list of command classes implemented by the endpoint or updates the information.
     * You shouldn't need to call this yourself.
     * @param info The information about the command class. This is merged with existing information.
     */
    addCC(cc, info) {
        var _a;
        let ccInfo = (_a = this._implementedCommandClasses.get(cc)) !== null && _a !== void 0 ? _a : {
            isSupported: false,
            isControlled: false,
            secure: false,
            version: 0,
        };
        ccInfo = Object.assign(ccInfo, info);
        this._implementedCommandClasses.set(cc, ccInfo);
    }
    /** Removes a CC from the list of command classes implemented by the endpoint */
    removeCC(cc) {
        this._implementedCommandClasses.delete(cc);
    }
    /** Tests if this endpoint supports the given CommandClass */
    supportsCC(cc) {
        var _a;
        return !!((_a = this._implementedCommandClasses.get(cc)) === null || _a === void 0 ? void 0 : _a.isSupported);
    }
    /** Tests if this endpoint supports or controls the given CC only securely */
    isCCSecure(cc) {
        var _a;
        return !!((_a = this._implementedCommandClasses.get(cc)) === null || _a === void 0 ? void 0 : _a.secure);
    }
    /** Tests if this endpoint controls the given CommandClass */
    controlsCC(cc) {
        var _a;
        return !!((_a = this._implementedCommandClasses.get(cc)) === null || _a === void 0 ? void 0 : _a.isControlled);
    }
    /** Removes the BasicCC from the supported CCs if any other actuator CCs are supported */
    hideBasicCCInFavorOfActuatorCCs() {
        // This behavior is defined in SDS14223
        if (this.supportsCC(core_1.CommandClasses.Basic) &&
            core_1.actuatorCCs.some((cc) => this.supportsCC(cc))) {
            // We still want to know if BasicCC is controlled, so only mark it as not supported
            this.addCC(core_1.CommandClasses.Basic, { isSupported: false });
            // If the record is now only a dummy, remove the CC
            if (!this.supportsCC(core_1.CommandClasses.Basic) &&
                !this.controlsCC(core_1.CommandClasses.Basic)) {
                this.removeCC(core_1.CommandClasses.Basic);
            }
        }
    }
    /**
     * Retrieves the version of the given CommandClass this endpoint implements.
     * Returns 0 if the CC is not supported.
     */
    getCCVersion(cc) {
        var _a;
        const ccInfo = this._implementedCommandClasses.get(cc);
        const ret = (_a = ccInfo === null || ccInfo === void 0 ? void 0 : ccInfo.version) !== null && _a !== void 0 ? _a : 0;
        // A controlling node interviewing a Multi Channel End Point MUST request the End Point’s
        // Command Class version from the Root Device if the End Point does not advertise support
        // for the Version Command Class.
        if (ret === 0 &&
            this.index > 0 &&
            !this.supportsCC(core_1.CommandClasses.Version)) {
            return this.getNodeUnsafe().getCCVersion(cc);
        }
        return ret;
    }
    /**
     * Creates an instance of the given CC and links it to this endpoint.
     * Throws if the CC is neither supported nor controlled by the endpoint.
     */
    createCCInstance(cc) {
        const ccId = typeof cc === "number" ? cc : CommandClass_1.getCommandClassStatic(cc);
        if (!this.supportsCC(ccId) && !this.controlsCC(ccId)) {
            throw new core_1.ZWaveError(`Cannot create an instance of the unsupported CC ${core_1.CommandClasses[ccId]} (${shared_1.num2hex(ccId)})`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        return this.createCCInstanceInternal(cc);
    }
    /**
     * Creates an instance of the given CC and links it to this endpoint.
     * Returns undefined if the CC is neither supported nor controlled by the endpoint.
     */
    createCCInstanceUnsafe(cc) {
        const ccId = typeof cc === "number" ? cc : CommandClass_1.getCommandClassStatic(cc);
        if (this.supportsCC(ccId) || this.controlsCC(ccId)) {
            return this.createCCInstanceInternal(cc);
        }
    }
    /**
     * @internal
     * Create an instance of the given CC without checking whether it is supported.
     * Applications should not use this directly.
     */
    createCCInstanceInternal(cc) {
        const Constructor = typeof cc === "number" ? CommandClass_1.getCCConstructor(cc) : cc;
        if (Constructor) {
            return new Constructor(this.driver, {
                nodeId: this.nodeId,
                endpoint: this.index,
            });
        }
    }
    /** Returns instances for all CCs this endpoint supports, that should be interviewed, and that are implemented in this library */
    getSupportedCCInstances() {
        let supportedCCInstances = [...this.implementedCommandClasses.keys()]
            // Don't interview CCs the node or endpoint only controls
            .filter((cc) => this.supportsCC(cc))
            // Filter out CCs we don't implement
            .map((cc) => this.createCCInstance(cc))
            .filter((instance) => !!instance);
        // For endpoint interviews, we skip some CCs
        if (this.index > 0) {
            supportedCCInstances = supportedCCInstances.filter((instance) => !instance.skipEndpointInterview());
        }
        return supportedCCInstances;
    }
    /** Builds the dependency graph used to automatically determine the order of CC interviews */
    buildCCInterviewGraph() {
        const supportedCCs = this.getSupportedCCInstances()
            .map((instance) => instance.ccId)
            // Security must be interviewed before all others, so we do that outside of the loop
            .filter((ccId) => ccId !== core_1.CommandClasses.Security &&
            ccId !== core_1.CommandClasses["Security 2"]);
        // Create GraphNodes from all supported CCs
        const ret = supportedCCs.map((cc) => new core_1.GraphNode(cc));
        // Create the dependencies
        for (const node of ret) {
            const instance = this.createCCInstance(node.value);
            for (const requiredCCId of instance.determineRequiredCCInterviews()) {
                const requiredCC = ret.find((instance) => instance.value === requiredCCId);
                if (requiredCC)
                    node.edges.add(requiredCC);
            }
        }
        return ret;
    }
    /**
     * @internal
     * Creates an API instance for a given command class. Throws if no API is defined.
     * @param ccId The command class to create an API instance for
     */
    createAPI(ccId) {
        const APIConstructor = CommandClass_1.getAPI(ccId);
        const ccName = core_1.CommandClasses[ccId];
        if (APIConstructor == undefined) {
            throw new core_1.ZWaveError(`Command Class ${ccName} (${shared_1.num2hex(ccId)}) has no associated API!`, core_1.ZWaveErrorCodes.CC_NoAPI);
        }
        const apiInstance = new APIConstructor(this.driver, this);
        return new Proxy(apiInstance, {
            get: (target, property) => {
                // Forbid access to the API if it is not supported by the node
                if (property !== "ccId" &&
                    property !== "endpoint" &&
                    property !== "isSupported" &&
                    !target.isSupported()) {
                    throw new core_1.ZWaveError(`Node ${this.nodeId}${this.index === 0 ? "" : ` (endpoint ${this.index})`} does not support the Command Class ${ccName}!`, core_1.ZWaveErrorCodes.CC_NotSupported);
                }
                return target[property];
            },
        });
    }
    /**
     * Provides access to simplified APIs that are taylored to specific CCs.
     * Make sure to check support of each API using `API.isSupported()` since
     * all other API calls will throw if the API is not supported
     */
    get commandClasses() {
        return this._commandClassAPIsProxy;
    }
    /**
     * Returns the node this endpoint belongs to (or undefined if the node doesn't exist)
     */
    getNodeUnsafe() {
        return this.driver.controller.nodes.get(this.nodeId);
    }
    /** Z-Wave+ Icon (for management) */
    get installerIcon() {
        var _a;
        return (_a = this.getNodeUnsafe()) === null || _a === void 0 ? void 0 : _a.getValue(ZWavePlusCC_1.getInstallerIconValueId(this.index));
    }
    /** Z-Wave+ Icon (for end users) */
    get userIcon() {
        var _a;
        return (_a = this.getNodeUnsafe()) === null || _a === void 0 ? void 0 : _a.getValue(ZWavePlusCC_1.getUserIconValueId(this.index));
    }
}
exports.Endpoint = Endpoint;

//# sourceMappingURL=Endpoint.js.map
