"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZWaveNode = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const strings_1 = require("alcalzone-shared/strings");
const typeguards_1 = require("alcalzone-shared/typeguards");
const crypto_1 = require("crypto");
const events_1 = require("events");
const AssociationCC_1 = require("../commandclass/AssociationCC");
const BasicCC_1 = require("../commandclass/BasicCC");
const CentralSceneCC_1 = require("../commandclass/CentralSceneCC");
const ClockCC_1 = require("../commandclass/ClockCC");
const CommandClass_1 = require("../commandclass/CommandClass");
const FirmwareUpdateMetaDataCC_1 = require("../commandclass/FirmwareUpdateMetaDataCC");
const HailCC_1 = require("../commandclass/HailCC");
const ICommandClassContainer_1 = require("../commandclass/ICommandClassContainer");
const ManufacturerSpecificCC_1 = require("../commandclass/ManufacturerSpecificCC");
const MultiChannelCC_1 = require("../commandclass/MultiChannelCC");
const NodeNamingCC_1 = require("../commandclass/NodeNamingCC");
const NotificationCC_1 = require("../commandclass/NotificationCC");
const SecurityCC_1 = require("../commandclass/SecurityCC");
const VersionCC_1 = require("../commandclass/VersionCC");
const WakeUpCC_1 = require("../commandclass/WakeUpCC");
const ZWavePlusCC_1 = require("../commandclass/ZWavePlusCC");
const ApplicationUpdateRequest_1 = require("../controller/ApplicationUpdateRequest");
const GetNodeProtocolInfoMessages_1 = require("../controller/GetNodeProtocolInfoMessages");
const GetRoutingInfoMessages_1 = require("../controller/GetRoutingInfoMessages");
const StateMachineShared_1 = require("../driver/StateMachineShared");
const DeviceClass_1 = require("./DeviceClass");
const Endpoint_1 = require("./Endpoint");
const NodeReadyMachine_1 = require("./NodeReadyMachine");
const NodeStatusMachine_1 = require("./NodeStatusMachine");
const RequestNodeInfoMessages_1 = require("./RequestNodeInfoMessages");
const Types_1 = require("./Types");
/**
 * A ZWaveNode represents a node in a Z-Wave network. It is also an instance
 * of its root endpoint (index 0)
 */
let ZWaveNode = class ZWaveNode extends Endpoint_1.Endpoint {
    constructor(id, driver, deviceClass, supportedCCs = [], controlledCCs = [], valueDB) {
        // Define this node's intrinsic endpoint as the root device (0)
        super(id, driver, 0);
        this.id = id;
        this._status = Types_1.NodeStatus.Unknown;
        this._ready = false;
        this._neighbors = [];
        this._nodeInfoReceived = false;
        /** Cache for this node's endpoint instances */
        this._endpointInstances = new Map();
        /**
         * This tells us which interview stage was last completed
         */
        this.interviewStage = Types_1.InterviewStage.None;
        this._interviewAttempts = 0;
        this._hasEmittedNoNetworkKeyError = false;
        this.manualRefreshTimers = new Map();
        this.hasLoggedNoNetworkKey = false;
        /**
         * Allows automatically resetting notification values to idle if the node does not do it itself
         */
        this.notificationIdleTimeouts = new Map();
        /**
         * Whether the node should be kept awake when there are no pending messages.
         */
        this.keepAwake = false;
        this.isSendingNoMoreInformation = false;
        this._valueDB = valueDB !== null && valueDB !== void 0 ? valueDB : new core_1.ValueDB(id, driver.valueDB, driver.metadataDB);
        for (const event of [
            "value added",
            "value updated",
            "value removed",
            "value notification",
            "metadata updated",
        ]) {
            this._valueDB.on(event, this.translateValueEvent.bind(this, event));
        }
        this._deviceClass = deviceClass;
        // Add mandatory CCs
        if (deviceClass) {
            for (const cc of deviceClass.mandatorySupportedCCs) {
                this.addCC(cc, { isSupported: true });
            }
            for (const cc of deviceClass.mandatoryControlledCCs) {
                this.addCC(cc, { isControlled: true });
            }
        }
        // Add optional CCs
        for (const cc of supportedCCs)
            this.addCC(cc, { isSupported: true });
        for (const cc of controlledCCs)
            this.addCC(cc, { isControlled: true });
        // Create and hook up the status machine
        this.statusMachine = StateMachineShared_1.interpretEx(NodeStatusMachine_1.createNodeStatusMachine(this));
        this.statusMachine.onTransition((state) => {
            if (state.changed) {
                this.onStatusChange(NodeStatusMachine_1.nodeStatusMachineStateToNodeStatus(state.value));
            }
        });
        this.statusMachine.start();
        this.readyMachine = StateMachineShared_1.interpretEx(NodeReadyMachine_1.createNodeReadyMachine());
        this.readyMachine.onTransition((state) => {
            if (state.changed) {
                this.onReadyChange(state.value === "ready");
            }
        });
        this.readyMachine.start();
    }
    /**
     * Cleans up all resources used by this node
     */
    destroy() {
        var _a;
        // Stop all state machines
        this.statusMachine.stop();
        this.readyMachine.stop();
        // Remove all timeouts
        for (const timeout of [
            (_a = this.centralSceneKeyHeldDownContext) === null || _a === void 0 ? void 0 : _a.timeout,
            ...this.notificationIdleTimeouts.values(),
            ...this.manualRefreshTimers.values(),
        ]) {
            if (timeout)
                clearTimeout(timeout);
        }
    }
    /**
     * Enhances a value id so it can be consumed better by applications
     */
    translateValueID(valueId) {
        // Try to retrieve the speaking CC name
        const commandClassName = core_1.getCCName(valueId.commandClass);
        const ret = {
            commandClassName,
            ...valueId,
        };
        const ccInstance = this.createCCInstanceInternal(valueId.commandClass);
        if (!ccInstance) {
            throw new core_1.ZWaveError(`Cannot translate a value ID for the non-implemented CC ${core_1.getCCName(valueId.commandClass)}`, core_1.ZWaveErrorCodes.CC_NotImplemented);
        }
        // Retrieve the speaking property name
        ret.propertyName = ccInstance.translateProperty(valueId.property, valueId.propertyKey);
        // Try to retrieve the speaking property key
        if (valueId.propertyKey != undefined) {
            const propertyKey = ccInstance.translatePropertyKey(valueId.property, valueId.propertyKey);
            ret.propertyKeyName = propertyKey;
        }
        return ret;
    }
    /**
     * Enhances the raw event args of the ValueDB so it can be consumed better by applications
     */
    translateValueEvent(eventName, arg) {
        var _a, _b;
        // Try to retrieve the speaking CC name
        const outArg = this.translateValueID(arg);
        // If this is a metadata event, make sure we return the merged metadata
        if ("metadata" in outArg) {
            outArg.metadata = this.getValueMetadata(arg);
        }
        // Log the value change
        const ccInstance = this.createCCInstanceInternal(arg.commandClass);
        const isInternalValue = ccInstance && ccInstance.isInternalValue(arg.property);
        // I don't like the splitting and any but its the easiest solution here
        const [changeTarget, changeType] = eventName.split(" ");
        const logArgument = {
            ...outArg,
            nodeId: this.nodeId,
            internal: isInternalValue,
        };
        if (changeTarget === "value") {
            this.driver.controllerLog.value(changeType, logArgument);
        }
        else if (changeTarget === "metadata") {
            this.driver.controllerLog.metadataUpdated(logArgument);
        }
        //Don't expose value events for internal value IDs...
        if (isInternalValue)
            return;
        // ... and root values ID that mirrors endpoint functionality
        if (
        // Only root endpoint values need to be filtered
        !arg.endpoint &&
            // Only application CCs need to be filtered
            core_1.applicationCCs.includes(arg.commandClass) &&
            // and only if a config file does not force us to expose the root endpoint
            !((_b = (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.compat) === null || _b === void 0 ? void 0 : _b.preserveRootApplicationCCValueIDs)) {
            // Iterate through all possible non-root endpoints of this node and
            // check if there is a value ID that mirrors root endpoint functionality
            for (let endpoint = 1; endpoint <= this.getEndpointCount(); endpoint++) {
                const possiblyMirroredValueID = {
                    // same CC, property and key
                    ...shared_1.pick(arg, ["commandClass", "property", "propertyKey"]),
                    // but different endpoint
                    endpoint,
                };
                if (this.valueDB.hasValue(possiblyMirroredValueID))
                    return;
            }
        }
        // And pass the translated event to our listeners
        this.emit(eventName, this, outArg);
    }
    onStatusChange(newStatus) {
        // Ignore duplicate events
        if (newStatus === this._status)
            return;
        const oldStatus = this._status;
        this._status = newStatus;
        if (this._status === Types_1.NodeStatus.Asleep) {
            this.emit("sleep", this, oldStatus);
        }
        else if (this._status === Types_1.NodeStatus.Awake) {
            this.emit("wake up", this, oldStatus);
        }
        else if (this._status === Types_1.NodeStatus.Dead) {
            this.emit("dead", this, oldStatus);
        }
        else if (this._status === Types_1.NodeStatus.Alive) {
            this.emit("alive", this, oldStatus);
        }
        // To be marked ready, a node must be known to be not dead.
        // This means that listening nodes must have communicated with us and
        // sleeping nodes are assumed to be ready
        this.readyMachine.send(this._status !== Types_1.NodeStatus.Unknown &&
            this._status !== Types_1.NodeStatus.Dead
            ? "NOT_DEAD"
            : "MAYBE_DEAD");
    }
    /**
     * Which status the node is believed to be in
     */
    get status() {
        return this._status;
    }
    /**
     * @internal
     * Marks this node as dead (if applicable)
     */
    markAsDead() {
        this.statusMachine.send("DEAD");
    }
    /**
     * @internal
     * Marks this node as alive (if applicable)
     */
    markAsAlive() {
        this.statusMachine.send("ALIVE");
    }
    /**
     * @internal
     * Marks this node as asleep (if applicable)
     */
    markAsAsleep() {
        this.statusMachine.send("ASLEEP");
    }
    /**
     * @internal
     * Marks this node as awake (if applicable)
     */
    markAsAwake() {
        this.statusMachine.send("AWAKE");
    }
    onReadyChange(ready) {
        // Ignore duplicate events
        if (ready === this._ready)
            return;
        this._ready = ready;
        if (ready)
            this.emit("ready", this);
    }
    /**
     * Whether the node is ready to be used
     */
    get ready() {
        return this._ready;
    }
    get deviceClass() {
        return this._deviceClass;
    }
    get isListening() {
        return this._isListening;
    }
    get isFrequentListening() {
        return this._isFrequentListening;
    }
    get isRouting() {
        return this._isRouting;
    }
    get maxBaudRate() {
        return this._maxBaudRate;
    }
    get isSecure() {
        return this._isSecure;
    }
    set isSecure(value) {
        this._isSecure = value;
    }
    /** The Z-Wave protocol version this node implements */
    get version() {
        return this._version;
    }
    get isBeaming() {
        return this._isBeaming;
    }
    get manufacturerId() {
        return this.getValue(ManufacturerSpecificCC_1.getManufacturerIdValueId());
    }
    get productId() {
        return this.getValue(ManufacturerSpecificCC_1.getProductIdValueId());
    }
    get productType() {
        return this.getValue(ManufacturerSpecificCC_1.getProductTypeValueId());
    }
    get firmwareVersion() {
        var _a;
        // We're only interested in the first (main) firmware
        return (_a = this.getValue(VersionCC_1.getFirmwareVersionsValueId())) === null || _a === void 0 ? void 0 : _a[0];
    }
    get zwavePlusVersion() {
        return this.getValue(ZWavePlusCC_1.getZWavePlusVersionValueId());
    }
    get nodeType() {
        return this.getValue(ZWavePlusCC_1.getNodeTypeValueId());
    }
    get roleType() {
        return this.getValue(ZWavePlusCC_1.getRoleTypeValueId());
    }
    /**
     * The user-defined name of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the name locally. To permanently change the name of the node, use
     * the `commandClasses` API.
     */
    get name() {
        return this.getValue(NodeNamingCC_1.getNodeNameValueId());
    }
    set name(value) {
        if (value != undefined) {
            this._valueDB.setValue(NodeNamingCC_1.getNodeNameValueId(), value);
        }
        else {
            this._valueDB.removeValue(NodeNamingCC_1.getNodeNameValueId());
        }
    }
    /**
     * The user-defined location of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the location locally. To permanently change the location of the node, use
     * the `commandClasses` API.
     */
    get location() {
        return this.getValue(NodeNamingCC_1.getNodeLocationValueId());
    }
    set location(value) {
        if (value != undefined) {
            this._valueDB.setValue(NodeNamingCC_1.getNodeLocationValueId(), value);
        }
        else {
            this._valueDB.removeValue(NodeNamingCC_1.getNodeLocationValueId());
        }
    }
    /**
     * Contains additional information about this node, loaded from a config file
     */
    get deviceConfig() {
        return this._deviceConfig;
    }
    get label() {
        var _a;
        return (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.label;
    }
    /** The IDs of all direct neighbors of this node */
    get neighbors() {
        return this._neighbors;
    }
    /**
     * Provides access to this node's values
     * @internal
     */
    get valueDB() {
        return this._valueDB;
    }
    /**
     * Retrieves a stored value for a given value id.
     * This does not request an updated value from the node!
     */
    /* wotan-disable-next-line no-misused-generics */
    getValue(valueId) {
        return this._valueDB.getValue(valueId);
    }
    /**
     * Retrieves metadata for a given value id.
     * This can be used to enhance the user interface of an application
     */
    getValueMetadata(valueId) {
        const { commandClass, property } = valueId;
        return {
            // Merge static metadata
            ...CommandClass_1.getCCValueMetadata(commandClass, property),
            // with potentially existing dynamic metadata
            ...this._valueDB.getMetadata(valueId),
        };
    }
    /** Returns a list of all value names that are defined on all endpoints of this node */
    getDefinedValueIDs() {
        var _a, _b;
        let ret = [];
        for (const endpoint of this.getAllEndpoints()) {
            for (const [cc, info] of endpoint.implementedCommandClasses) {
                // Don't return value IDs which are only controlled
                if (!info.isSupported)
                    continue;
                const ccInstance = endpoint.createCCInstanceUnsafe(cc);
                if (ccInstance) {
                    ret.push(...ccInstance.getDefinedValueIDs());
                }
            }
        }
        if (!((_b = (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.compat) === null || _b === void 0 ? void 0 : _b.preserveRootApplicationCCValueIDs)) {
            // Application command classes of the Root Device capabilities that are also advertised by at
            // least one End Point SHOULD be filtered out by controlling nodes before presenting the functionalities
            // via service discovery mechanisms like mDNS or to users in a GUI.
            ret = this.filterRootApplicationCCValueIDs(ret);
        }
        // Translate the remaining value IDs before exposing them to applications
        return ret.map((id) => this.translateValueID(id));
    }
    shouldHideValueID(valueId, allValueIds) {
        // Non-root endpoint values don't need to be filtered
        if (!!valueId.endpoint)
            return false;
        // Non-application CCs don't need to be filtered
        if (!core_1.applicationCCs.includes(valueId.commandClass))
            return false;
        // Filter out root values if an identical value ID exists for another endpoint
        const valueExistsOnAnotherEndpoint = allValueIds.some((other) => 
        // same CC
        other.commandClass === valueId.commandClass &&
            // non-root endpoint
            !!other.endpoint &&
            // same property and key
            other.property === valueId.property &&
            other.propertyKey === valueId.propertyKey);
        return valueExistsOnAnotherEndpoint;
    }
    /**
     * Removes all Value IDs from an array that belong to a root endpoint and have a corresponding
     * Value ID on a non-root endpoint
     */
    filterRootApplicationCCValueIDs(allValueIds) {
        return allValueIds.filter((vid) => !this.shouldHideValueID(vid, allValueIds));
    }
    /**
     * Updates a value for a given property of a given CommandClass on the node.
     * This will communicate with the node!
     */
    async setValue(valueId, value) {
        // Try to retrieve the corresponding CC API
        try {
            // Access the CC API by name
            const endpointInstance = this.getEndpoint(valueId.endpoint || 0);
            if (!endpointInstance)
                return false;
            const api = endpointInstance.commandClasses[valueId.commandClass];
            // Check if the setValue method is implemented
            if (!api.setValue)
                return false;
            // And call it
            await api.setValue({
                property: valueId.property,
                propertyKey: valueId.propertyKey,
            }, value);
            // If the call did not throw, assume that the call was successful and remember the new value
            this._valueDB.setValue(valueId, value, { noEvent: true });
            return true;
        }
        catch (e) {
            // Define which errors during setValue are expected and won't crash
            // the driver:
            if (e instanceof core_1.ZWaveError) {
                let handled = false;
                let emitErrorEvent = false;
                switch (e.code) {
                    // This CC or API is not implemented
                    case core_1.ZWaveErrorCodes.CC_NotImplemented:
                    case core_1.ZWaveErrorCodes.CC_NoAPI:
                        handled = true;
                        break;
                    // A user tried to set an invalid value
                    case core_1.ZWaveErrorCodes.Argument_Invalid:
                        handled = true;
                        emitErrorEvent = true;
                        break;
                }
                if (emitErrorEvent)
                    this.driver.emit("error", e);
                if (handled)
                    return false;
            }
            throw e;
        }
    }
    /**
     * Requests a value for a given property of a given CommandClass by polling the node.
     * **Warning:** Some value IDs share a command, so make sure not to blindly call this for every property
     */
    // wotan-disable-next-line no-misused-generics
    pollValue(valueId) {
        // Try to retrieve the corresponding CC API
        const endpointInstance = this.getEndpoint(valueId.endpoint || 0);
        if (!endpointInstance) {
            throw new core_1.ZWaveError(`Endpoint ${valueId.endpoint} does not exist on Node ${this.id}`, core_1.ZWaveErrorCodes.Argument_Invalid);
        }
        const api = endpointInstance.commandClasses[valueId.commandClass];
        // Check if the pollValue method is implemented
        if (!api.pollValue) {
            throw new core_1.ZWaveError(`The pollValue API is not implemented for CC ${core_1.getCCName(valueId.commandClass)}!`, core_1.ZWaveErrorCodes.CC_NoAPI);
        }
        // And call it
        return api.pollValue({
            property: valueId.property,
            propertyKey: valueId.propertyKey,
        });
    }
    get endpointCountIsDynamic() {
        return this.getValue({
            commandClass: core_1.CommandClasses["Multi Channel"],
            property: "countIsDynamic",
        });
    }
    get endpointsHaveIdenticalCapabilities() {
        return this.getValue({
            commandClass: core_1.CommandClasses["Multi Channel"],
            property: "identicalCapabilities",
        });
    }
    get individualEndpointCount() {
        return this.getValue({
            commandClass: core_1.CommandClasses["Multi Channel"],
            property: "individualCount",
        });
    }
    get aggregatedEndpointCount() {
        return this.getValue({
            commandClass: core_1.CommandClasses["Multi Channel"],
            property: "aggregatedCount",
        });
    }
    getEndpointCCs(index) {
        return this.getValue(MultiChannelCC_1.getEndpointCCsValueId(this.endpointsHaveIdenticalCapabilities ? 1 : index));
    }
    /** Returns the current endpoint count of this node */
    getEndpointCount() {
        return ((this.individualEndpointCount || 0) +
            (this.aggregatedEndpointCount || 0));
    }
    /** Whether the Multi Channel CC has been interviewed and all endpoint information is known */
    get isMultiChannelInterviewComplete() {
        return !!this.getValue({
            commandClass: core_1.CommandClasses["Multi Channel"],
            endpoint: 0,
            property: "interviewComplete",
        });
    }
    getEndpoint(index) {
        if (index < 0)
            throw new core_1.ZWaveError("The endpoint index must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
        // Zero is the root endpoint - i.e. this node
        if (index === 0)
            return this;
        // Check if the requested endpoint exists on the physical node
        if (index > this.getEndpointCount())
            return undefined;
        // Check if the Multi Channel CC interview for this node is completed,
        // because we don't have all the information before that
        if (!this.isMultiChannelInterviewComplete) {
            this.driver.driverLog.print(`Node ${this.nodeId}, Endpoint ${index}: Trying to access endpoint instance before Multi Channel interview`, "error");
            return undefined;
        }
        // Create an endpoint instance if it does not exist
        if (!this._endpointInstances.has(index)) {
            this._endpointInstances.set(index, new Endpoint_1.Endpoint(this.id, this.driver, index, this.getEndpointCCs(index)));
        }
        return this._endpointInstances.get(index);
    }
    /** Returns a list of all endpoints of this node, including the root endpoint (index 0) */
    getAllEndpoints() {
        const ret = [this];
        // Check if the Multi Channel CC interview for this node is completed,
        // because we don't have all the endpoint information before that
        if (this.isMultiChannelInterviewComplete) {
            for (let i = 1; i <= this.getEndpointCount(); i++) {
                // Iterating over the endpoint count ensures that we don't get undefined
                ret.push(this.getEndpoint(i));
            }
        }
        return ret;
    }
    /** How many attempts to interview this node have already been made */
    get interviewAttempts() {
        return this._interviewAttempts;
    }
    /** Utility function to check if this node is the controller */
    isControllerNode() {
        return this.id === this.driver.controller.ownNodeId;
    }
    /**
     * Resets all information about this node and forces a fresh interview.
     *
     * WARNING: Take care NOT to call this method when the node is already being interviewed.
     * Otherwise the node information may become inconsistent.
     */
    async refreshInfo() {
        // preserve the node name and location, since they might not be stored on the node
        const name = this.name;
        const location = this.location;
        this._interviewAttempts = 0;
        this.interviewStage = Types_1.InterviewStage.None;
        this._nodeInfoReceived = false;
        this._ready = false;
        this._deviceClass = undefined;
        this._isListening = undefined;
        this._isFrequentListening = undefined;
        this._isRouting = undefined;
        this._maxBaudRate = undefined;
        this._isSecure = undefined;
        this._version = undefined;
        this._isBeaming = undefined;
        this._deviceConfig = undefined;
        this._neighbors = [];
        this._hasEmittedNoNetworkKeyError = false;
        this._valueDB.clear({ noEvent: true });
        this._endpointInstances.clear();
        super.reset();
        // Restart all state machines
        this.readyMachine.restart();
        this.statusMachine.restart();
        // Restore the previously saved name/location
        if (name != undefined)
            this.name = name;
        if (location != undefined)
            this.location = location;
        // Also remove the information from the cache
        await this.driver.saveNetworkToCache();
        // Don't keep the node awake after the interview
        this.keepAwake = false;
        void this.driver.interviewNode(this);
    }
    /**
     * @internal
     * Interviews this node. Returns true when it succeeded, false otherwise
     *
     * WARNING: Do not call this method from application code. To refresh the information
     * for a specific node, use `node.refreshInfo()` instead
     */
    async interview() {
        if (this.interviewStage === Types_1.InterviewStage.Complete) {
            this.driver.controllerLog.logNode(this.id, `skipping interview because it is already completed`);
            return true;
        }
        else {
            this.driver.controllerLog.interviewStart(this);
        }
        // Remember that we tried to interview this node
        this._interviewAttempts++;
        // Wrapper around interview methods to return false in case of a communication error
        // This way the single methods don't all need to have the same error handler
        const tryInterviewStage = async (method) => {
            try {
                await method();
                return true;
            }
            catch (e) {
                if (e instanceof core_1.ZWaveError &&
                    (e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout ||
                        e.code === core_1.ZWaveErrorCodes.Controller_ResponseNOK ||
                        e.code === core_1.ZWaveErrorCodes.Controller_CallbackNOK ||
                        e.code === core_1.ZWaveErrorCodes.Controller_MessageDropped)) {
                    return false;
                }
                throw e;
            }
        };
        // The interview is done in several stages. At each point, the interview process might be aborted
        // due to a stage failing. The reached stage is saved, so we can continue it later without
        // repeating stages unnecessarily
        if (this.interviewStage === Types_1.InterviewStage.None) {
            // do a full interview starting with the protocol info
            this.driver.controllerLog.logNode(this.id, `new node, doing a full interview...`);
            await this.queryProtocolInfo();
        }
        if (this.interviewStage === Types_1.InterviewStage.ProtocolInfo) {
            // Ping node to check if it is alive/asleep/...
            // TODO: #739, point 3 -> Do this automatically for the first message
            await this.ping();
            if (!(await tryInterviewStage(() => this.queryNodeInfo()))) {
                return false;
            }
        }
        // The node is deemed ready when has been interviewed completely at least once
        if (this.interviewStage === Types_1.InterviewStage.RestartFromCache) {
            // Mark listening nodes as potentially ready. The first message will determine if it is
            this.readyMachine.send("RESTART_INTERVIEW_FROM_CACHE");
            // Ping node to check if it is alive/asleep/...
            // TODO: #739, point 3 -> Do this automatically for the first message
            await this.ping();
        }
        // At this point the basic interview of new nodes is done. Start here when re-interviewing known nodes
        // to get updated information about command classes
        if (this.interviewStage === Types_1.InterviewStage.RestartFromCache ||
            this.interviewStage === Types_1.InterviewStage.NodeInfo) {
            // Only advance the interview if it was completed, otherwise abort
            if (await this.interviewCCs()) {
                await this.setInterviewStage(Types_1.InterviewStage.CommandClasses);
            }
            else {
                return false;
            }
        }
        if (this.interviewStage === Types_1.InterviewStage.CommandClasses) {
            // Load a config file for this node if it exists and overwrite the previously reported information
            await this.overwriteConfig();
        }
        if (this.interviewStage === Types_1.InterviewStage.OverwriteConfig) {
            // Request a list of this node's neighbors
            if (!(await tryInterviewStage(() => this.queryNeighbors()))) {
                return false;
            }
        }
        await this.setInterviewStage(Types_1.InterviewStage.Complete);
        this.readyMachine.send("INTERVIEW_DONE");
        // Regularly query listening nodes for updated values
        this.scheduleManualValueRefreshesForListeningNodes();
        // Tell listeners that the interview is completed
        // The driver will then send this node to sleep
        this.emit("interview completed", this);
        return true;
    }
    /** Updates this node's interview stage and saves to cache when appropriate */
    async setInterviewStage(completedStage) {
        this.interviewStage = completedStage;
        // Also save to the cache after certain stages
        switch (completedStage) {
            case Types_1.InterviewStage.ProtocolInfo:
            case Types_1.InterviewStage.NodeInfo:
            case Types_1.InterviewStage.CommandClasses:
            case Types_1.InterviewStage.Complete:
                await this.driver.saveNetworkToCache();
        }
        this.driver.controllerLog.interviewStage(this);
    }
    /** Step #1 of the node interview */
    async queryProtocolInfo() {
        this.driver.controllerLog.logNode(this.id, {
            message: "querying protocol info...",
            direction: "outbound",
        });
        const resp = await this.driver.sendMessage(new GetNodeProtocolInfoMessages_1.GetNodeProtocolInfoRequest(this.driver, {
            requestedNodeId: this.id,
        }));
        if (process.env.NODE_ENV !== "test" &&
            !(resp instanceof GetNodeProtocolInfoMessages_1.GetNodeProtocolInfoResponse)) {
            // eslint-disable-next-line @typescript-eslint/no-var-requires
            const Sentry = require("@sentry/node");
            Sentry.captureMessage("Response to GetNodeProtocolInfoRequest is not a GetNodeProtocolInfoResponse", {
                contexts: {
                    message: {
                        name: resp.constructor.name,
                        type: resp.type,
                        functionType: resp
                            .functionType,
                        ...resp.toLogEntry(),
                    },
                },
            });
        }
        this._deviceClass = resp.deviceClass;
        for (const cc of this._deviceClass.mandatorySupportedCCs) {
            this.addCC(cc, { isSupported: true });
        }
        for (const cc of this._deviceClass.mandatoryControlledCCs) {
            this.addCC(cc, { isControlled: true });
        }
        this._isListening = resp.isListening;
        this._isFrequentListening = resp.isFrequentListening;
        this._isRouting = resp.isRouting;
        this._maxBaudRate = resp.maxBaudRate;
        // Many nodes don't have this flag set, even if they are included securely
        // So we treat false as "unknown"
        this._isSecure = resp.isSecure || core_1.unknownBoolean;
        this._version = resp.version;
        this._isBeaming = resp.isBeaming;
        const logMessage = `received response for protocol info:
basic device class:    ${this._deviceClass.basic.label}
generic device class:  ${this._deviceClass.generic.label}
specific device class: ${this._deviceClass.specific.label}
is a listening device: ${this.isListening}
is frequent listening: ${this.isFrequentListening}
is a routing device:   ${this.isRouting}
is a secure device:    ${this.isSecure}
is a beaming device:   ${this.isBeaming}
maximum baud rate:     ${this.maxBaudRate} kbps
version:               ${this.version}`;
        this.driver.controllerLog.logNode(this.id, {
            message: logMessage,
            direction: "inbound",
        });
        if (!this.isListening && !this.isFrequentListening) {
            // This is a "sleeping" device which must support the WakeUp CC.
            // We are requesting the supported CCs later, but those commands may need to go into the
            // wakeup queue. Thus we need to mark WakeUp as supported
            this.addCC(core_1.CommandClasses["Wake Up"], {
                isSupported: true,
            });
            // Assume the node is awake, after all we're communicating with it.
            this.markAsAwake();
        }
        await this.setInterviewStage(Types_1.InterviewStage.ProtocolInfo);
    }
    /** Node interview: pings the node to see if it responds */
    async ping() {
        if (this.isControllerNode()) {
            this.driver.controllerLog.logNode(this.id, "not pinging the controller");
        }
        else {
            this.driver.controllerLog.logNode(this.id, {
                message: "pinging the node...",
                direction: "outbound",
            });
            try {
                await this.commandClasses["No Operation"].send();
                this.driver.controllerLog.logNode(this.id, {
                    message: "ping successful",
                    direction: "inbound",
                });
            }
            catch (e) {
                this.driver.controllerLog.logNode(this.id, `ping failed: ${e.message}`);
                return false;
            }
        }
        return true;
    }
    /**
     * Step #5 of the node interview
     * Request node info
     */
    async queryNodeInfo() {
        if (this.isControllerNode()) {
            this.driver.controllerLog.logNode(this.id, "not querying node info from the controller");
        }
        else {
            this.driver.controllerLog.logNode(this.id, {
                message: "querying node info...",
                direction: "outbound",
            });
            const resp = await this.driver.sendMessage(new RequestNodeInfoMessages_1.RequestNodeInfoRequest(this.driver, { nodeId: this.id }));
            if (resp instanceof RequestNodeInfoMessages_1.RequestNodeInfoResponse && !resp.wasSent) {
                // TODO: handle this in SendThreadMachine
                this.driver.controllerLog.logNode(this.id, `Querying the node info failed`, "error");
                throw new core_1.ZWaveError(`Querying the node info failed`, core_1.ZWaveErrorCodes.Controller_ResponseNOK);
            }
            else if (resp instanceof ApplicationUpdateRequest_1.ApplicationUpdateRequestNodeInfoRequestFailed) {
                // TODO: handle this in SendThreadMachine
                this.driver.controllerLog.logNode(this.id, `Querying the node info failed`, "error");
                throw new core_1.ZWaveError(`Querying the node info failed`, core_1.ZWaveErrorCodes.Controller_CallbackNOK);
            }
            else if (resp instanceof ApplicationUpdateRequest_1.ApplicationUpdateRequestNodeInfoReceived) {
                const logLines = [
                    "node info received",
                    "supported CCs:",
                ];
                for (const cc of resp.nodeInformation.supportedCCs) {
                    const ccName = core_1.CommandClasses[cc];
                    logLines.push(`· ${ccName ? ccName : shared_1.num2hex(cc)}`);
                }
                logLines.push("controlled CCs:");
                for (const cc of resp.nodeInformation.controlledCCs) {
                    const ccName = core_1.CommandClasses[cc];
                    logLines.push(`· ${ccName ? ccName : shared_1.num2hex(cc)}`);
                }
                this.driver.controllerLog.logNode(this.id, {
                    message: logLines.join("\n"),
                    direction: "inbound",
                });
                this.updateNodeInfo(resp.nodeInformation);
            }
        }
        await this.setInterviewStage(Types_1.InterviewStage.NodeInfo);
    }
    /**
     * Loads the device configuration for this node from a config file
     */
    async loadDeviceConfig() {
        // But the configuration definitions might change
        if (this.manufacturerId != undefined &&
            this.productType != undefined &&
            this.productId != undefined) {
            // Try to load the config file
            this.driver.controllerLog.logNode(this.id, "trying to load device config");
            this._deviceConfig = await this.driver.configManager.lookupDevice(this.manufacturerId, this.productType, this.productId, this.firmwareVersion);
            if (this._deviceConfig) {
                this.driver.controllerLog.logNode(this.id, "device config loaded");
            }
            else {
                this.driver.controllerLog.logNode(this.id, "no device config loaded", "warn");
            }
        }
    }
    /** Step #? of the node interview */
    async interviewCCs() {
        var _a;
        const interviewEndpoint = async (endpoint, cc) => {
            var _a, _b;
            let instance;
            try {
                instance = endpoint.createCCInstance(cc);
            }
            catch (e) {
                if (e instanceof core_1.ZWaveError &&
                    e.code === core_1.ZWaveErrorCodes.CC_NotSupported) {
                    // The CC is no longer supported. This can happen if the node tells us
                    // something different in the Version interview than it did in its NIF
                    return "continue";
                }
                // we want to pass all other errors through
                throw e;
            }
            if (endpoint.isCCSecure(cc) && !this.driver.securityManager) {
                // The CC is only supported securely, but the network key is not set up
                // Skip the CC
                this.driver.controllerLog.logNode(this.id, `Skipping interview for secure CC ${core_1.getCCName(cc)} because no network key is configured!`, "error");
                return "continue";
            }
            try {
                await instance.interview(!instance.interviewComplete);
            }
            catch (e) {
                if (e instanceof core_1.ZWaveError &&
                    (e.code === core_1.ZWaveErrorCodes.Controller_MessageDropped ||
                        e.code === core_1.ZWaveErrorCodes.Controller_CallbackNOK ||
                        e.code === core_1.ZWaveErrorCodes.Controller_ResponseNOK ||
                        e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout)) {
                    // We had a CAN or timeout during the interview
                    // or the node is presumed dead. Abort the process
                    return false;
                }
                // we want to pass all other errors through
                throw e;
            }
            try {
                if (cc === core_1.CommandClasses.Version && endpoint.index === 0) {
                    // After the version CC interview of the root endpoint, we have enough info to load the correct device config file
                    await this.loadDeviceConfig();
                    if ((_b = (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.compat) === null || _b === void 0 ? void 0 : _b.treatBasicSetAsEvent) {
                        // To create the compat event value, we need to force a Basic CC interview
                        this.addCC(core_1.CommandClasses.Basic, {
                            isSupported: true,
                            version: 1,
                        });
                    }
                }
                await this.driver.saveNetworkToCache();
            }
            catch (e) {
                this.driver.controllerLog.print(`${core_1.getCCName(cc)}: Error after interview:\n${e.message}`, "error");
            }
        };
        // Always interview Security first because it changes the interview order
        if (this.supportsCC(core_1.CommandClasses.Security) &&
            // At this point we're not sure if the node is included securely
            this._isSecure !== false) {
            // Security is always supported *securely*
            this.addCC(core_1.CommandClasses.Security, { secure: true });
            if (!this.driver.securityManager) {
                if (!this._hasEmittedNoNetworkKeyError) {
                    // Cannot interview a secure device securely without a network key
                    const errorMessage = `supports Security, but no network key was configured. Continuing interview non-securely.`;
                    this.driver.controllerLog.logNode(this.nodeId, errorMessage, "error");
                    this.driver.emit("error", new core_1.ZWaveError(`Node ${strings_1.padStart(this.id.toString(), 3, "0")} ${errorMessage}`, core_1.ZWaveErrorCodes.Controller_NodeInsecureCommunication));
                    this._hasEmittedNoNetworkKeyError = true;
                }
            }
            else {
                const action = await interviewEndpoint(this, core_1.CommandClasses.Security);
                if (this._isSecure === true && action === false) {
                    // The node is definitely included securely, but we got no response to the interview question
                    return false;
                }
                else if (action === false || action === "continue") {
                    // Assume that the node is not actually included securely
                    this.driver.controllerLog.logNode(this.nodeId, `The node is not included securely. Continuing interview non-securely.`);
                    this._isSecure = false;
                }
                else {
                    // We got a response, so we know the node is included securely
                    if (this._isSecure !== true) {
                        this.driver.controllerLog.logNode(this.nodeId, `The node is included securely.`);
                    }
                    this._isSecure = true;
                }
            }
        }
        else if (!this.supportsCC(core_1.CommandClasses.Security) &&
            this._isSecure === core_1.unknownBoolean) {
            // Remember that this node is not secure
            this._isSecure = false;
        }
        // Don't offer or interview the Basic CC if any actuator CC is supported - except if the config files forbid us
        // to map the Basic CC to other CCs or expose Basic Set as an event
        const compat = (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.compat;
        if (!(compat === null || compat === void 0 ? void 0 : compat.disableBasicMapping) && !(compat === null || compat === void 0 ? void 0 : compat.treatBasicSetAsEvent)) {
            this.hideBasicCCInFavorOfActuatorCCs();
        }
        // We determine the correct interview order by topologically sorting a dependency graph
        const rootInterviewGraph = this.buildCCInterviewGraph();
        let rootInterviewOrder;
        // In order to avoid emitting unnecessary value events for the root endpoint,
        // we defer the application CC interview until after the other endpoints
        // have been interviewed
        const deferApplicationCCs = (cc1, cc2) => {
            const cc1IsApplicationCC = core_1.applicationCCs.includes(cc1);
            const cc2IsApplicationCC = core_1.applicationCCs.includes(cc2);
            return ((cc1IsApplicationCC ? 1 : 0) -
                (cc2IsApplicationCC ? 1 : 0));
        };
        try {
            rootInterviewOrder = core_1.topologicalSort(rootInterviewGraph, deferApplicationCCs);
        }
        catch (e) {
            // This interview cannot be done
            throw new core_1.ZWaveError("The CC interview cannot be completed because there are circular dependencies between CCs!", core_1.ZWaveErrorCodes.CC_Invalid);
        }
        // Now that we know the correct order, do the interview in sequence
        let rootCCIndex = 0;
        for (; rootCCIndex < rootInterviewOrder.length; rootCCIndex++) {
            const cc = rootInterviewOrder[rootCCIndex];
            // Once we reach the application CCs, pause the root endpoint interview
            if (core_1.applicationCCs.includes(cc))
                break;
            const action = await interviewEndpoint(this, cc);
            if (action === "continue")
                continue;
            else if (typeof action === "boolean")
                return action;
        }
        // Now query ALL endpoints
        for (let endpointIndex = 1; endpointIndex <= this.getEndpointCount(); endpointIndex++) {
            const endpoint = this.getEndpoint(endpointIndex);
            if (!endpoint)
                continue;
            // Always interview Security first because it changes the interview order
            if (endpoint.supportsCC(core_1.CommandClasses.Security) &&
                // The root endpoint has been interviewed, so we know it the device supports security
                this._isSecure === true &&
                // Only interview SecurityCC if the network key was set
                this.driver.securityManager) {
                // Security is always supported *securely*
                endpoint.addCC(core_1.CommandClasses.Security, { secure: true });
                const action = await interviewEndpoint(endpoint, core_1.CommandClasses.Security);
                if (typeof action === "boolean")
                    return action;
            }
            // Don't offer or interview the Basic CC if any actuator CC is supported - except if the config files forbid us
            // to map the Basic CC to other CCs or expose Basic Set as an event
            if (!(compat === null || compat === void 0 ? void 0 : compat.disableBasicMapping) && !(compat === null || compat === void 0 ? void 0 : compat.treatBasicSetAsEvent)) {
                endpoint.hideBasicCCInFavorOfActuatorCCs();
            }
            const endpointInterviewGraph = endpoint.buildCCInterviewGraph();
            let endpointInterviewOrder;
            try {
                endpointInterviewOrder = core_1.topologicalSort(endpointInterviewGraph);
            }
            catch (e) {
                // This interview cannot be done
                throw new core_1.ZWaveError("The CC interview cannot be completed because there are circular dependencies between CCs!", core_1.ZWaveErrorCodes.CC_Invalid);
            }
            // Now that we know the correct order, do the interview in sequence
            for (const cc of endpointInterviewOrder) {
                const action = await interviewEndpoint(endpoint, cc);
                if (action === "continue")
                    continue;
                else if (typeof action === "boolean")
                    return action;
            }
        }
        // Continue with the application CCs for the root endpoint
        for (; rootCCIndex < rootInterviewOrder.length; rootCCIndex++) {
            const cc = rootInterviewOrder[rootCCIndex];
            const action = await interviewEndpoint(this, cc);
            if (action === "continue")
                continue;
            else if (typeof action === "boolean")
                return action;
        }
        return true;
    }
    /**
     * @internal
     * Handles the receipt of a NIF / NodeUpdatePayload
     */
    updateNodeInfo(nodeInfo) {
        if (!this._nodeInfoReceived) {
            for (const cc of nodeInfo.supportedCCs)
                this.addCC(cc, { isSupported: true });
            for (const cc of nodeInfo.controlledCCs)
                this.addCC(cc, { isControlled: true });
            this._nodeInfoReceived = true;
        }
        // As the NIF is sent on wakeup, treat this as a sign that the node is awake
        this.markAsAwake();
        // SDS14223 Unless unsolicited <XYZ> Report Commands are received,
        // a controlling node MUST probe the current values when the
        // supporting node issues a Wake Up Notification Command for sleeping nodes.
        // This is not the handler for wakeup notifications, but some legacy devices send this
        // message whenever there's an update
        if (this.requiresManualValueRefresh()) {
            this.driver.controllerLog.logNode(this.nodeId, {
                message: `Node does not send unsolicited updates, refreshing actuator and sensor values...`,
            });
            void this.refreshValues();
        }
    }
    /** Returns whether a manual refresh of non-static values is likely necessary for this node */
    requiresManualValueRefresh() {
        // If there was no lifeline configured, we assume that the controller
        // does not receive unsolicited updates from the node
        return (this.interviewStage === Types_1.InterviewStage.Complete &&
            !this.supportsCC(core_1.CommandClasses["Z-Wave Plus Info"]) &&
            !this.valueDB.getValue(AssociationCC_1.getHasLifelineValueId()));
    }
    /**
     * Schedules the regular refreshes of some CC values
     */
    scheduleManualValueRefreshesForListeningNodes() {
        var _a;
        // Only schedule this for listening nodes. Sleeping nodes are queried on wakeup
        if (this.supportsCC(core_1.CommandClasses["Wake Up"]))
            return;
        // Only schedule this if we don't expect any unsolicited updates
        if (!this.requiresManualValueRefresh())
            return;
        // TODO: The timespan definitions should be on the CCs themselves (probably as decorators)
        this.scheduleManualValueRefresh(core_1.CommandClasses.Battery, 
        // The specs say once per month, but that's a bit too unfrequent IMO
        // Also the maximum that setInterval supports is ~24.85 days
        core_1.timespan.days(7));
        this.scheduleManualValueRefresh(core_1.CommandClasses.Meter, core_1.timespan.hours(6));
        this.scheduleManualValueRefresh(core_1.CommandClasses["Multilevel Sensor"], core_1.timespan.hours(6));
        if (this.supportsCC(core_1.CommandClasses.Notification) &&
            ((_a = this.createCCInstance(NotificationCC_1.NotificationCC)) === null || _a === void 0 ? void 0 : _a.notificationMode) === "pull") {
            this.scheduleManualValueRefresh(core_1.CommandClasses.Notification, core_1.timespan.hours(6));
        }
    }
    /**
     * Is used to schedule a manual value refresh for nodes that don't send unsolicited commands
     */
    scheduleManualValueRefresh(cc, timeout) {
        // // Avoid triggering the refresh multiple times
        // this.cancelManualValueRefresh(cc);
        this.manualRefreshTimers.set(cc, setInterval(() => {
            void this.refreshCCValues(cc);
        }, timeout).unref());
    }
    cancelManualValueRefresh(cc) {
        if (this.manualRefreshTimers.has(cc)) {
            const timeout = this.manualRefreshTimers.get(cc);
            clearTimeout(timeout);
            this.manualRefreshTimers.delete(cc);
        }
    }
    /**
     * Refreshes all non-static values of a single CC from this node.
     * WARNING: It is not recommended to await this method!
     */
    async refreshCCValues(cc) {
        for (const endpoint of this.getAllEndpoints()) {
            const instance = endpoint.createCCInstanceUnsafe(cc);
            if (instance) {
                // Don't do a complete interview, only dynamic values
                try {
                    await instance.interview(false);
                }
                catch (e) {
                    this.driver.controllerLog.logNode(this.id, `failed to refresh values for ${core_1.getCCName(cc)}, endpoint ${endpoint.index}: ${e.message}`, "error");
                }
            }
        }
    }
    /**
     * Refreshes all non-static values from this node.
     * WARNING: It is not recommended to await this method!
     */
    async refreshValues() {
        for (const endpoint of this.getAllEndpoints()) {
            for (const cc of endpoint.getSupportedCCInstances()) {
                // Only query actuator and sensor CCs
                if (!core_1.actuatorCCs.includes(cc.ccId) &&
                    !core_1.sensorCCs.includes(cc.ccId)) {
                    continue;
                }
                // Don't do a complete interview, only dynamic values
                try {
                    await cc.interview(false);
                }
                catch (e) {
                    this.driver.controllerLog.logNode(this.id, `failed to refresh values for ${core_1.getCCName(cc.ccId)}, endpoint ${endpoint.index}: ${e.message}`, "error");
                }
            }
        }
    }
    /** Overwrites the reported configuration with information from a config file */
    async overwriteConfig() {
        var _a, _b, _c, _d;
        if (this.isControllerNode()) {
            // The device config was not loaded prior to this step because the Version CC is not interviewed.
            // Therefore do it here.
            await this.loadDeviceConfig();
        }
        if (this.deviceConfig) {
            // Add CCs the device config file tells us to
            const addCCs = (_a = this.deviceConfig.compat) === null || _a === void 0 ? void 0 : _a.addCCs;
            if (addCCs) {
                for (const [cc, { endpoints }] of addCCs) {
                    for (const [ep, info] of endpoints) {
                        (_b = this.getEndpoint(ep)) === null || _b === void 0 ? void 0 : _b.addCC(cc, info);
                    }
                }
            }
            // And remove those that it marks as unsupported
            const removeCCs = (_c = this.deviceConfig.compat) === null || _c === void 0 ? void 0 : _c.removeCCs;
            if (removeCCs) {
                for (const [cc, endpoints] of removeCCs) {
                    if (endpoints === "*") {
                        for (const ep of this.getAllEndpoints()) {
                            ep.removeCC(cc);
                        }
                    }
                    else {
                        for (const ep of endpoints) {
                            (_d = this.getEndpoint(ep)) === null || _d === void 0 ? void 0 : _d.removeCC(cc);
                        }
                    }
                }
            }
        }
        await this.setInterviewStage(Types_1.InterviewStage.OverwriteConfig);
    }
    /** @internal */
    async queryNeighborsInternal() {
        this.driver.controllerLog.logNode(this.id, {
            message: "requesting node neighbors...",
            direction: "outbound",
        });
        try {
            const resp = await this.driver.sendMessage(new GetRoutingInfoMessages_1.GetRoutingInfoRequest(this.driver, {
                nodeId: this.id,
                removeBadLinks: false,
                removeNonRepeaters: false,
            }));
            if (process.env.NODE_ENV !== "test" &&
                !(resp instanceof GetRoutingInfoMessages_1.GetRoutingInfoResponse)) {
                // eslint-disable-next-line @typescript-eslint/no-var-requires
                const Sentry = require("@sentry/node");
                Sentry.captureMessage("Response to GetRoutingInfoRequest is not a GetRoutingInfoResponse", {
                    contexts: {
                        message: {
                            name: resp.constructor
                                .name,
                            type: resp.type,
                            functionType: resp
                                .functionType,
                            ...resp.toLogEntry(),
                        },
                    },
                });
            }
            this._neighbors = resp.nodeIds;
            this.driver.controllerLog.logNode(this.id, {
                message: `  node neighbors received: ${this._neighbors.join(", ")}`,
                direction: "inbound",
            });
        }
        catch (e) {
            this.driver.controllerLog.logNode(this.id, `  requesting the node neighbors failed: ${e.message}`, "error");
            throw e;
        }
    }
    /**
     * @internal
     * Temporarily updates the node's neighbor list by removing a node from it
     */
    removeNodeFromCachedNeighbors(nodeId) {
        this._neighbors = this._neighbors.filter((id) => id !== nodeId);
    }
    /** Queries a node for its neighbor nodes during the node interview */
    async queryNeighbors() {
        await this.queryNeighborsInternal();
        await this.setInterviewStage(Types_1.InterviewStage.Neighbors);
    }
    /**
     * @internal
     * Handles a CommandClass that was received from this node
     */
    handleCommand(command) {
        // If the node sent us an unsolicited update, our initial assumption
        // was wrong. Stop querying it regularly for updates
        this.cancelManualValueRefresh(command.ccId);
        // If this is a report for the root endpoint and the node does not support Multi Channel Association CC V3+,
        // we need to map it to endpoint 1
        if (command.endpointIndex === 0 &&
            command.constructor.name.endsWith("Report") &&
            this.getEndpointCount() >= 1 &&
            // Don't check for MCA support or devices without it won't be handled
            // Instead rely on the version. If MCA is not supported, this will be 0
            this.getCCVersion(core_1.CommandClasses["Multi Channel Association"]) < 3) {
            // Force the CC to store its values again under endpoint 1
            command.endpointIndex = 1;
            command.persistValues();
        }
        if (command instanceof BasicCC_1.BasicCC) {
            return this.handleBasicCommand(command);
        }
        else if (command instanceof CentralSceneCC_1.CentralSceneCCNotification) {
            return this.handleCentralSceneNotification(command);
        }
        else if (command instanceof WakeUpCC_1.WakeUpCCWakeUpNotification) {
            return this.handleWakeUpNotification();
        }
        else if (command instanceof NotificationCC_1.NotificationCCReport) {
            return this.handleNotificationReport(command);
        }
        else if (command instanceof ClockCC_1.ClockCCReport) {
            return this.handleClockReport(command);
        }
        else if (command instanceof SecurityCC_1.SecurityCCNonceGet) {
            return this.handleSecurityNonceGet();
        }
        else if (command instanceof SecurityCC_1.SecurityCCNonceReport) {
            return this.handleSecurityNonceReport(command);
        }
        else if (command instanceof HailCC_1.HailCC) {
            return this.handleHail(command);
        }
        else if (command instanceof FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCCGet) {
            return this.handleFirmwareUpdateGet(command);
        }
        else if (command instanceof FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCCStatusReport) {
            return this.handleFirmwareUpdateStatusReport(command);
        }
        // Ignore all commands that don't need to be handled
        if (command.constructor.name.endsWith("Report")) {
            // Reports are either a response to a Get command or
            // automatically store their values in the Value DB.
            // No need to manually handle them - other than what we've already done
            return;
        }
        this.driver.controllerLog.logNode(this.id, {
            message: `TODO: no handler for application command`,
            direction: "inbound",
        });
    }
    /**
     * @internal
     * Handles a nonce request
     */
    async handleSecurityNonceGet() {
        // Only reply if secure communication is set up
        if (!this.driver.securityManager) {
            if (!this.hasLoggedNoNetworkKey) {
                this.hasLoggedNoNetworkKey = true;
                this.driver.controllerLog.logNode(this.id, {
                    message: `cannot reply to NonceGet because no network key was configured!`,
                    direction: "inbound",
                    level: "warn",
                });
            }
            return;
        }
        // When a node asks us for a nonce, it must support Security CC
        this.addCC(core_1.CommandClasses.Security, {
            isSupported: true,
            version: 1,
            // Security CC is always secure
            secure: true,
        });
        // Ensure that we're not flooding the queue with unnecessary NonceReports (GH#1059)
        const { queue, currentTransaction } = this.driver["sendThread"].state.context;
        const isNonceReport = (t) => !!t &&
            t.message.getNodeId() === this.nodeId &&
            ICommandClassContainer_1.isCommandClassContainer(t.message) &&
            t.message.command instanceof SecurityCC_1.SecurityCCNonceReport;
        if (isNonceReport(currentTransaction) ||
            queue.find((t) => isNonceReport(t))) {
            this.driver.controllerLog.logNode(this.id, {
                message: "in the process of replying to a NonceGet, won't send another NonceReport",
                level: "warn",
            });
            return;
        }
        // Delete all previous nonces we sent the node, since they should no longer be used
        this.driver.securityManager.deleteAllNoncesForReceiver(this.id);
        // Now send the current nonce
        try {
            await this.commandClasses.Security.sendNonce();
        }
        catch (e) {
            this.driver.controllerLog.logNode(this.id, {
                message: `failed to send nonce: ${e}`,
                direction: "inbound",
            });
        }
    }
    /**
     * Is called when a nonce report is received that does not belong to any transaction.
     * The received nonce reports are stored as "free" nonces
     */
    handleSecurityNonceReport(command) {
        const secMan = this.driver.securityManager;
        if (!secMan)
            return;
        secMan.setNonce({
            issuer: this.id,
            nonceId: secMan.getNonceId(command.nonce),
        }, {
            nonce: command.nonce,
            receiver: this.driver.controller.ownNodeId,
        }, { free: true });
    }
    handleHail(_command) {
        // treat this as a sign that the node is awake
        this.markAsAwake();
        this.driver.controllerLog.logNode(this.nodeId, {
            message: `Hail received from node, refreshing actuator and sensor values...`,
        });
        void this.refreshValues();
    }
    /** Handles the receipt of a Central Scene notifification */
    handleCentralSceneNotification(command) {
        // Did we already receive this command?
        if (command.sequenceNumber ===
            this.lastCentralSceneNotificationSequenceNumber) {
            return;
        }
        else {
            this.lastCentralSceneNotificationSequenceNumber =
                command.sequenceNumber;
        }
        /*
        If the Slow Refresh field is false:
        - A new Key Held Down notification MUST be sent every 200ms until the key is released.
        - The Sequence Number field MUST be updated at each notification transmission.
        - If not receiving a new Key Held Down notification within 400ms, a controlling node SHOULD use an adaptive timeout approach as described in 4.17.1:
        A controller SHOULD apply an adaptive approach based on the reception of the Key Released Notification.
        Initially, the controller SHOULD time out if not receiving any Key Held Down Notification refresh after
        400ms and consider this to be a Key Up Notification. If, however, the controller subsequently receives a
        Key Released Notification, the controller SHOULD consider the sending node to be operating with the Slow
        Refresh capability enabled.

        If the Slow Refresh field is true:
        - A new Key Held Down notification MUST be sent every 55 seconds until the key is released.
        - The Sequence Number field MUST be updated at each notification refresh.
        - If not receiving a new Key Held Down notification within 60 seconds after the most recent Key Held Down
        notification, a receiving node MUST respond as if it received a Key Release notification.
        */
        const setSceneValue = (sceneNumber, key) => {
            const valueId = CentralSceneCC_1.getSceneValueId(sceneNumber);
            this.valueDB.setValue(valueId, key, { stateful: false });
        };
        const forceKeyUp = () => {
            // force key up event
            setSceneValue(this.centralSceneKeyHeldDownContext.sceneNumber, CentralSceneCC_1.CentralSceneKeys.KeyReleased);
            // clear old timer
            clearTimeout(this.centralSceneKeyHeldDownContext.timeout);
            // clear the key down context
            this.centralSceneKeyHeldDownContext = undefined;
        };
        if (this.centralSceneKeyHeldDownContext &&
            this.centralSceneKeyHeldDownContext.sceneNumber !==
                command.sceneNumber) {
            // The user pressed another button, force release
            forceKeyUp();
        }
        if (command.keyAttribute === CentralSceneCC_1.CentralSceneKeys.KeyHeldDown) {
            // Set or refresh timer to force a release of the key
            if (this.centralSceneKeyHeldDownContext) {
                clearTimeout(this.centralSceneKeyHeldDownContext.timeout);
            }
            this.centralSceneKeyHeldDownContext = {
                sceneNumber: command.sceneNumber,
                // Unref'ing long running timers allows the process to exit mid-timeout
                timeout: setTimeout(forceKeyUp, command.slowRefresh ? 60000 : 400).unref(),
            };
        }
        else if (command.keyAttribute === CentralSceneCC_1.CentralSceneKeys.KeyReleased) {
            // Stop the release timer
            if (this.centralSceneKeyHeldDownContext) {
                clearTimeout(this.centralSceneKeyHeldDownContext.timeout);
                this.centralSceneKeyHeldDownContext = undefined;
            }
        }
        setSceneValue(command.sceneNumber, command.keyAttribute);
        this.driver.controllerLog.logNode(this.id, {
            message: `received CentralScene notification ${shared_1.stringify(command)}`,
            direction: "inbound",
        });
    }
    /** Handles the receipt of a Wake Up notification */
    handleWakeUpNotification() {
        var _a, _b, _c;
        this.driver.controllerLog.logNode(this.id, {
            message: `received wakeup notification`,
            direction: "inbound",
        });
        // It can happen that the node has not told us that it supports the Wake Up CC
        // https://sentry.io/share/issue/6a681729d7db46d591f1dcadabe8d02e/
        // To avoid a crash, mark it as supported
        this.addCC(core_1.CommandClasses["Wake Up"], {
            isSupported: true,
            version: 1,
        });
        this.markAsAwake();
        // From the specs:
        // A controlling node SHOULD read the Wake Up Interval of a supporting node when the delays between
        // Wake Up periods are larger than what was last set at the supporting node.
        const now = Date.now();
        if (this.lastWakeUp) {
            // we've already measured the wake up interval, so we can check whether a refresh is necessary
            const wakeUpInterval = (_a = this.getValue(WakeUpCC_1.getWakeUpIntervalValueId())) !== null && _a !== void 0 ? _a : 0;
            // The wakeup interval is specified in seconds. Also add 5 seconds tolerance to avoid
            // unnecessary queries since there might be some delay
            if ((now - this.lastWakeUp) / 1000 > wakeUpInterval + 5) {
                this.commandClasses["Wake Up"].getInterval().catch(() => {
                    // Don't throw if there's an error
                });
            }
        }
        this.lastWakeUp = now;
        // Some devices expect us to query them on wake up in order to function correctly
        if ((_c = (_b = this._deviceConfig) === null || _b === void 0 ? void 0 : _b.compat) === null || _c === void 0 ? void 0 : _c.queryOnWakeup) {
            // Don't wait
            void this.compatDoWakeupQueries();
        }
        // In case there are no messages in the queue, the node may go back to sleep very soon
        this.driver.debounceSendNodeToSleep(this);
    }
    async compatDoWakeupQueries() {
        var _a, _b;
        if (!((_b = (_a = this._deviceConfig) === null || _a === void 0 ? void 0 : _a.compat) === null || _b === void 0 ? void 0 : _b.queryOnWakeup))
            return;
        this.driver.controllerLog.logNode(this.id, {
            message: `expects some queries after wake up, so it shall receive`,
            direction: "none",
        });
        for (const [ccName, apiMethod, ...args] of this._deviceConfig.compat
            .queryOnWakeup) {
            this.driver.controllerLog.logNode(this.id, {
                message: `compat query "${ccName}"::${apiMethod}(${args
                    .map((arg) => JSON.stringify(arg))
                    .join(", ")})`,
                direction: "none",
            });
            // Try to access the API - if it doesn't work, skip this option
            let API;
            try {
                API = this.commandClasses[ccName].withOptions({
                    // Tag the resulting transactions as compat queries
                    tag: "compat",
                });
            }
            catch (_c) {
                this.driver.controllerLog.logNode(this.id, {
                    message: `could not access API, skipping query`,
                    direction: "none",
                    level: "warn",
                });
                continue;
            }
            if (!API.isSupported()) {
                this.driver.controllerLog.logNode(this.id, {
                    message: `API not supported, skipping query`,
                    direction: "none",
                    level: "warn",
                });
                continue;
            }
            else if (!API[apiMethod]) {
                this.driver.controllerLog.logNode(this.id, {
                    message: `method ${apiMethod} not found on API, skipping query`,
                    direction: "none",
                    level: "warn",
                });
                continue;
            }
            // Retrieve the method
            // eslint-disable-next-line
            const method = API[apiMethod].bind(API);
            // And replace "smart" arguments with their corresponding value
            const methodArgs = args.map((arg) => {
                if (typeguards_1.isObject(arg)) {
                    const valueId = {
                        commandClass: API.ccId,
                        ...arg,
                    };
                    return this.getValue(valueId);
                }
                return arg;
            });
            // Do the API call and ignore/log any errors
            try {
                await method(...methodArgs);
                this.driver.controllerLog.logNode(this.id, {
                    message: `API call successful`,
                    direction: "none",
                });
            }
            catch (e) {
                this.driver.controllerLog.logNode(this.id, {
                    message: `error during API call: ${e}`,
                    direction: "none",
                    level: "warn",
                });
            }
        }
    }
    /** Handles the receipt of a BasicCC Set or Report */
    handleBasicCommand(command) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Retrieve the endpoint the command is coming from
        const sourceEndpoint = (_b = this.getEndpoint((_a = command.endpointIndex) !== null && _a !== void 0 ? _a : 0)) !== null && _b !== void 0 ? _b : this;
        // Depending on the generic device class, we may need to map the basic command to other CCs
        let mappedTargetCC;
        // Do not map the basic CC if the device config forbids it
        if (!((_d = (_c = this._deviceConfig) === null || _c === void 0 ? void 0 : _c.compat) === null || _d === void 0 ? void 0 : _d.disableBasicMapping)) {
            switch ((_e = this.deviceClass) === null || _e === void 0 ? void 0 : _e.generic.key) {
                case 0x20: // Binary Sensor
                    mappedTargetCC = sourceEndpoint.createCCInstanceUnsafe(core_1.CommandClasses["Binary Sensor"]);
                    break;
                // TODO: Which sensor type to use here?
                // case GenericDeviceClasses["Multilevel Sensor"]:
                // 	mappedTargetCC = this.createCCInstanceUnsafe(
                // 		CommandClasses["Multilevel Sensor"],
                // 	);
                // 	break;
                case 0x10: // Binary Switch
                    mappedTargetCC = sourceEndpoint.createCCInstanceUnsafe(core_1.CommandClasses["Binary Switch"]);
                    break;
                case 0x11: // Multilevel Switch
                    mappedTargetCC = sourceEndpoint.createCCInstanceUnsafe(core_1.CommandClasses["Multilevel Switch"]);
                    break;
            }
        }
        if (command instanceof BasicCC_1.BasicCCReport) {
            // Try to set the mapped value on the target CC
            const didSetMappedValue = typeof command.currentValue === "number" && (mappedTargetCC === null || mappedTargetCC === void 0 ? void 0 : mappedTargetCC.setMappedBasicValue(command.currentValue));
            // Otherwise fall back to setting it ourselves
            if (!didSetMappedValue) {
                // Store the value in the value DB now
                command.persistValues();
                // Since the node sent us a Basic report, we are sure that it is at least supported
                // If this is the only supported actuator CC, add it to the support list,
                // so the information lands in the network cache
                if (!core_1.actuatorCCs.some((cc) => sourceEndpoint.supportsCC(cc))) {
                    sourceEndpoint.addCC(core_1.CommandClasses.Basic, {
                        isControlled: true,
                    });
                }
            }
        }
        else if (command instanceof BasicCC_1.BasicCCSet) {
            // Treat BasicCCSet as value events if desired
            if ((_g = (_f = this._deviceConfig) === null || _f === void 0 ? void 0 : _f.compat) === null || _g === void 0 ? void 0 : _g.treatBasicSetAsEvent) {
                this.driver.controllerLog.logNode(this.id, {
                    message: "treating BasicCC Set as a value event",
                });
                this._valueDB.setValue(BasicCC_1.getCompatEventValueId(command.endpointIndex), command.targetValue, {
                    stateful: false,
                });
                return;
            }
            // Some devices send their current state using `BasicCCSet`s to their associations
            // instead of using reports. We still interpret them like reports
            this.driver.controllerLog.logNode(this.id, {
                message: "treating BasicCC Set as a report",
            });
            // Try to set the mapped value on the target CC
            const didSetMappedValue = mappedTargetCC === null || mappedTargetCC === void 0 ? void 0 : mappedTargetCC.setMappedBasicValue(command.targetValue);
            // Otherwise fall back to setting it ourselves
            if (!didSetMappedValue) {
                // Sets cannot store their value automatically, so store the values manually
                this._valueDB.setValue(BasicCC_1.getCurrentValueValueId(command.endpointIndex), command.targetValue);
                // Since the node sent us a Basic command, we are sure that it is at least controlled
                // Add it to the support list, so the information lands in the network cache
                if (!sourceEndpoint.controlsCC(core_1.CommandClasses.Basic)) {
                    sourceEndpoint.addCC(core_1.CommandClasses.Basic, {
                        isControlled: true,
                    });
                }
            }
        }
    }
    /** Schedules a notification value to be reset */
    scheduleNotificationIdleReset(valueId, handler) {
        this.clearNotificationIdleReset(valueId);
        const key = core_1.valueIdToString(valueId);
        this.notificationIdleTimeouts.set(key, 
        // Unref'ing long running timeouts allows to quit the application before the timeout elapses
        setTimeout(handler, 5 * 3600 * 1000 /* 5 minutes */).unref());
    }
    /** Removes a scheduled notification reset */
    clearNotificationIdleReset(valueId) {
        const key = core_1.valueIdToString(valueId);
        if (this.notificationIdleTimeouts.has(key)) {
            clearTimeout(this.notificationIdleTimeouts.get(key));
            this.notificationIdleTimeouts.delete(key);
        }
    }
    /**
     * Handles the receipt of a Notification Report
     */
    handleNotificationReport(command) {
        if (command.notificationType == undefined) {
            this.driver.controllerLog.logNode(this.id, {
                message: `received unsupported notification ${shared_1.stringify(command)}`,
                direction: "inbound",
            });
            return;
        }
        // Look up the received notification in the config
        const notificationConfig = this.driver.configManager.lookupNotification(command.notificationType);
        if (notificationConfig) {
            // This is a known notification (status or event)
            const property = notificationConfig.name;
            /** Returns a single notification state to idle */
            const setStateIdle = (prevValue) => {
                const valueConfig = notificationConfig.lookupValue(prevValue);
                // Only known variables may be reset to idle
                if (!valueConfig || valueConfig.type !== "state")
                    return;
                // Some properties may not be reset to idle
                if (!valueConfig.idle)
                    return;
                const propertyKey = valueConfig.variableName;
                const valueId = {
                    commandClass: command.ccId,
                    endpoint: command.endpointIndex,
                    property,
                    propertyKey,
                };
                // Since the node has reset the notification itself, we don't need the idle reset
                this.clearNotificationIdleReset(valueId);
                this.valueDB.setValue(valueId, 0 /* idle */);
            };
            const value = command.notificationEvent;
            if (value === 0) {
                // Generic idle notification, this contains a value to be reset
                if (Buffer.isBuffer(command.eventParameters) &&
                    command.eventParameters.length) {
                    // The target value is the first byte of the event parameters
                    setStateIdle(command.eventParameters[0]);
                }
                else {
                    // Reset all values to idle
                    const nonIdleValues = this.valueDB
                        .getValues(core_1.CommandClasses.Notification)
                        .filter((v) => (v.endpoint || 0) === command.endpointIndex &&
                        v.property === property &&
                        typeof v.value === "number" &&
                        v.value !== 0);
                    for (const v of nonIdleValues) {
                        setStateIdle(v.value);
                    }
                }
                return;
            }
            let propertyKey;
            // Find out which property we need to update
            const valueConfig = notificationConfig.lookupValue(value);
            let allowIdleReset;
            if (!valueConfig) {
                // This is an unknown value, collect it in an unknown bucket
                propertyKey = "unknown";
                // We don't know what this notification refers to, so we don't force a reset
                allowIdleReset = false;
            }
            else if (valueConfig.type === "state") {
                propertyKey = valueConfig.variableName;
                allowIdleReset = valueConfig.idle;
            }
            else {
                this.emit("notification", this, valueConfig.label, command.eventParameters);
                return;
            }
            // Now that we've gathered all we need to know, update the value in our DB
            const valueId = {
                commandClass: command.ccId,
                endpoint: command.endpointIndex,
                property,
                propertyKey,
            };
            this.valueDB.setValue(valueId, value);
            // Nodes before V8 don't necessarily reset the notification to idle
            // Set a fallback timer in case the node does not reset it.
            if (allowIdleReset &&
                this.driver.getSafeCCVersionForNode(core_1.CommandClasses.Notification, this.id) <= 7) {
                this.scheduleNotificationIdleReset(valueId, () => setStateIdle(value));
            }
        }
        else {
            // This is an unknown notification
            const property = `UNKNOWN_${shared_1.num2hex(command.notificationType)}`;
            const valueId = {
                commandClass: command.ccId,
                endpoint: command.endpointIndex,
                property,
            };
            this.valueDB.setValue(valueId, command.notificationEvent);
            // We don't know what this notification refers to, so we don't force a reset
        }
    }
    handleClockReport(command) {
        // A Z-Wave Plus node SHOULD issue a Clock Report Command via the Lifeline Association Group if they
        // suspect to have inaccurate time and/or weekdays (e.g. after battery removal).
        // A controlling node SHOULD compare the received time and weekday with its current time and set the
        // time again at the supporting node if a deviation is observed (e.g. different weekday or more than a
        // minute difference)
        const now = new Date();
        // local time
        const hours = now.getHours();
        let minutes = now.getMinutes();
        // A sending node knowing the current time with seconds precision SHOULD round its
        // current time to the nearest minute when sending this command.
        if (now.getSeconds() >= 30) {
            minutes = (minutes + 1) % 60;
        }
        // Sunday is 0 in JS, but 7 in Z-Wave
        let weekday = now.getDay();
        if (weekday === 0)
            weekday = 7;
        if (command.weekday !== weekday ||
            command.hour !== hours ||
            command.minute !== minutes) {
            const endpoint = command.getEndpoint();
            if (!endpoint)
                return;
            this.driver.controllerLog.logNode(this.nodeId, `detected a deviation of the node's clock, updating it...`);
            endpoint.commandClasses.Clock.set(hours, minutes, weekday).catch(() => {
                // Don't throw when the update fails
            });
        }
    }
    /**
     * Starts an OTA firmware update process for this node.
     *
     * **WARNING: Use at your own risk! We don't take any responsibility if your devices don't work after an update.**
     *
     * @param data The firmware image
     * @param target The firmware target (i.e. chip) to upgrade. 0 updates the Z-Wave chip, >=1 updates others if they exist
     */
    async beginFirmwareUpdate(data, target = 0) {
        var _a;
        // Don't start the process twice
        if (this._firmwareUpdateStatus) {
            throw new core_1.ZWaveError(`Failed to start the update: A firmware upgrade is already in progress!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_Busy);
        }
        this._firmwareUpdateStatus = {
            data,
            fragmentSize: 0,
            numFragments: 0,
            progress: 0,
            abort: false,
        };
        const version = this.getCCVersion(core_1.CommandClasses["Firmware Update Meta Data"]);
        const api = this.commandClasses["Firmware Update Meta Data"];
        // Check if this update is possible
        const meta = await api.getMetaData();
        if (!meta) {
            throw new core_1.ZWaveError(`The node did not respond in time`, core_1.ZWaveErrorCodes.Controller_NodeTimeout);
        }
        if (target === 0 && !meta.firmwareUpgradable) {
            throw new core_1.ZWaveError(`The Z-Wave chip firmware is not upgradable`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_NotUpgradable);
        }
        else if (version < 3 && target !== 0) {
            throw new core_1.ZWaveError(`Upgrading different firmware targets requires version 3+`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_TargetNotFound);
        }
        else if (target < 0 ||
            (target > 0 && meta.additionalFirmwareIDs.length < target)) {
            throw new core_1.ZWaveError(`Firmware target #${target} not found!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_TargetNotFound);
        }
        // Determine the fragment size
        const fcc = new FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCC(this.driver, {
            nodeId: this.id,
        });
        const maxNetPayloadSize = this.driver.computeNetCCPayloadSize(fcc) -
            2 - // report number
            (version >= 2 ? 2 : 0); // checksum
        // Use the smallest allowed payload
        this._firmwareUpdateStatus.fragmentSize = Math.min(maxNetPayloadSize, (_a = meta.maxFragmentSize) !== null && _a !== void 0 ? _a : Number.POSITIVE_INFINITY);
        this._firmwareUpdateStatus.numFragments = Math.ceil(data.length / this._firmwareUpdateStatus.fragmentSize);
        this.driver.controllerLog.logNode(this.id, {
            message: `Starting firmware update...`,
            direction: "outbound",
        });
        // Request the node to start the upgrade
        // TODO: Should manufacturer id and firmware id be provided externally?
        const status = await api.requestUpdate({
            manufacturerId: meta.manufacturerId,
            firmwareId: meta.firmwareId,
            firmwareTarget: target,
            fragmentSize: this._firmwareUpdateStatus.fragmentSize,
            checksum: core_1.CRC16_CCITT(data),
        });
        switch (status) {
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_AuthenticationExpected:
                throw new core_1.ZWaveError(`Failed to start the update: A manual authentication event (e.g. button push) was expected!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToStart);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_BatteryLow:
                throw new core_1.ZWaveError(`Failed to start the update: The battery level is too low!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToStart);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_FirmwareUpgradeInProgress:
                throw new core_1.ZWaveError(`Failed to start the update: A firmware upgrade is already in progress!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_Busy);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_InvalidManufacturerOrFirmwareID:
                throw new core_1.ZWaveError(`Failed to start the update: Invalid manufacturer or firmware id!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToStart);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_InvalidHardwareVersion:
                throw new core_1.ZWaveError(`Failed to start the update: Invalid hardware version!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToStart);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_NotUpgradable:
                throw new core_1.ZWaveError(`Failed to start the update: Firmware target #${target} is not upgradable!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_NotUpgradable);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.Error_FragmentSizeTooLarge:
                throw new core_1.ZWaveError(`Failed to start the update: The chosen fragment size is too large!`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToStart);
            case FirmwareUpdateMetaDataCC_1.FirmwareUpdateRequestStatus.OK:
                // All good, we have started!
                // Keep the node awake until the update is done.
                this.keepAwake = true;
                // Timeout the update when no get request has been received for a while
                this._firmwareUpdateStatus.getTimeout = setTimeout(() => this.timeoutFirmwareUpdate(), 30000).unref();
                return;
        }
    }
    /**
     * Aborts an active firmware update process
     */
    async abortFirmwareUpdate() {
        // Don't stop the process twice
        if (!this._firmwareUpdateStatus || this._firmwareUpdateStatus.abort) {
            return;
        }
        else if (this._firmwareUpdateStatus.numFragments > 0 &&
            this._firmwareUpdateStatus.progress ===
                this._firmwareUpdateStatus.numFragments) {
            throw new core_1.ZWaveError(`The firmware update was transmitted completely, cannot abort anymore.`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToAbort);
        }
        this.driver.controllerLog.logNode(this.id, {
            message: `Aborting firmware update...`,
            direction: "outbound",
        });
        this._firmwareUpdateStatus.abort = true;
        try {
            await this.driver.waitForCommand((cc) => cc.nodeId === this.nodeId &&
                cc instanceof FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCCStatusReport &&
                cc.status === FirmwareUpdateMetaDataCC_1.FirmwareUpdateStatus.Error_TransmissionFailed, 5000);
            this.driver.controllerLog.logNode(this.id, {
                message: `Firmware update aborted`,
                direction: "inbound",
            });
            // Clean up
            this._firmwareUpdateStatus = undefined;
            this.keepAwake = false;
        }
        catch (e) {
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout) {
                throw new core_1.ZWaveError(`The node did not acknowledge the aborted update`, core_1.ZWaveErrorCodes.FirmwareUpdateCC_FailedToAbort);
            }
            throw e;
        }
    }
    async sendCorruptedFirmwareUpdateReport(reportNum, fragment) {
        var _a;
        if (!fragment) {
            let fragmentSize = (_a = this._firmwareUpdateStatus) === null || _a === void 0 ? void 0 : _a.fragmentSize;
            if (!fragmentSize) {
                // We don't know the fragment size, so we send a fragment with the maximum possible size
                const fcc = new FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCC(this.driver, {
                    nodeId: this.id,
                });
                fragmentSize =
                    this.driver.computeNetCCPayloadSize(fcc) -
                        2 - // report number
                        (fcc.version >= 2 ? 2 : 0); // checksum
            }
            fragment = crypto_1.randomBytes(fragmentSize);
        }
        else {
            // If we already have data, corrupt it
            for (let i = 0; i < fragment.length; i++) {
                fragment[i] = fragment[i] ^ 0xff;
            }
        }
        await this.commandClasses["Firmware Update Meta Data"].sendFirmwareFragment(reportNum, true, fragment);
    }
    handleFirmwareUpdateGet(command) {
        if (this._firmwareUpdateStatus == undefined) {
            this.driver.controllerLog.logNode(this.id, {
                message: `Received Firmware Update Get, but no firmware update is in progress. Forcing the node to abort...`,
                direction: "inbound",
            });
            this.sendCorruptedFirmwareUpdateReport(command.reportNumber).catch(() => {
                /* ignore */
            });
            return;
        }
        else if (command.reportNumber > this._firmwareUpdateStatus.numFragments) {
            this.driver.controllerLog.logNode(this.id, {
                message: `Received Firmware Update Get for an out-of-bounds fragment. Forcing the node to abort...`,
                direction: "inbound",
            });
            this.sendCorruptedFirmwareUpdateReport(command.reportNumber).catch(() => {
                /* ignore */
            });
            return;
        }
        // Refresh the get timeout
        if (this._firmwareUpdateStatus.getTimeout) {
            // console.warn("refreshed get timeout");
            this._firmwareUpdateStatus.getTimeout = this._firmwareUpdateStatus.getTimeout
                .refresh()
                .unref();
        }
        // When a node requests a firmware update fragment, it must be awake
        try {
            this.markAsAwake();
        }
        catch (_a) {
            /* ignore */
        }
        // Send the response(s) in the background
        void (async () => {
            var _a;
            const { numFragments, data, fragmentSize, abort, } = this._firmwareUpdateStatus;
            for (let num = command.reportNumber; num < command.reportNumber + command.numReports; num++) {
                // Check if the node requested more fragments than are left
                if (num > numFragments) {
                    break;
                }
                const fragment = data.slice((num - 1) * fragmentSize, num * fragmentSize);
                if (abort) {
                    await this.sendCorruptedFirmwareUpdateReport(num, fragment);
                    return;
                }
                else {
                    this.driver.controllerLog.logNode(this.id, {
                        message: `Sending firmware fragment ${num} / ${numFragments}`,
                        direction: "outbound",
                    });
                    const isLast = num === numFragments;
                    if (isLast) {
                        // Don't send the node to sleep now
                        this.keepAwake = true;
                    }
                    await this.commandClasses["Firmware Update Meta Data"].sendFirmwareFragment(num, isLast, fragment);
                    // Remember the progress
                    this._firmwareUpdateStatus.progress = num;
                    // And notify listeners
                    this.emit("firmware update progress", this, num, numFragments);
                    // If that was the last one wait for status report from the node and restart interview
                    if (isLast) {
                        // The update was completed, we don't need to timeout get requests anymore
                        if ((_a = this._firmwareUpdateStatus) === null || _a === void 0 ? void 0 : _a.getTimeout) {
                            clearTimeout(this._firmwareUpdateStatus.getTimeout);
                            this._firmwareUpdateStatus.getTimeout = undefined;
                        }
                        void this.finishFirmwareUpdate().catch(() => {
                            /* ignore */
                        });
                    }
                }
            }
        })().catch((e) => {
            this.driver.controllerLog.logNode(this.nodeId, `Error while sending firmware fragment: ${e.message}`, "error");
        });
    }
    timeoutFirmwareUpdate() {
        // In some cases it can happen that the device stops requesting update frames
        // We need to timeout the update in this case so it can be restarted
        this.driver.controllerLog.logNode(this.id, {
            message: `Firmware update timed out`,
            direction: "none",
            level: "warn",
        });
        // clean up
        this._firmwareUpdateStatus = undefined;
        this.keepAwake = false;
        // Notify listeners
        this.emit("firmware update finished", this, FirmwareUpdateMetaDataCC_1.FirmwareUpdateStatus.Error_Timeout);
    }
    handleFirmwareUpdateStatusReport(report) {
        // If no firmware update is in progress, we don't care
        if (!this._firmwareUpdateStatus)
            return;
        const { status, waitTime } = report;
        // Actually, OK_WaitingForActivation should never happen since we don't allow
        // delayed activation in the RequestGet command
        const success = status >= FirmwareUpdateMetaDataCC_1.FirmwareUpdateStatus.OK_WaitingForActivation;
        this.driver.controllerLog.logNode(this.id, {
            message: `Firmware update ${success ? "completed" : "failed"} with status ${shared_1.getEnumMemberName(FirmwareUpdateMetaDataCC_1.FirmwareUpdateStatus, status)}`,
            direction: "inbound",
        });
        // clean up
        this._firmwareUpdateStatus = undefined;
        this.keepAwake = false;
        // Notify listeners
        this.emit("firmware update finished", this, status, success ? waitTime : undefined);
    }
    async finishFirmwareUpdate() {
        try {
            const report = await this.driver.waitForCommand((cc) => cc.nodeId === this.nodeId &&
                cc instanceof FirmwareUpdateMetaDataCC_1.FirmwareUpdateMetaDataCCStatusReport, 
            // Wait up to 5 minutes. It should never take that long, but the specs
            // don't say anything specific
            5 * 60000);
            this.handleFirmwareUpdateStatusReport(report);
        }
        catch (e) {
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Controller_NodeTimeout) {
                this.driver.controllerLog.logNode(this.id, `The node did not acknowledge the completed update`, "warn");
                // clean up
                this._firmwareUpdateStatus = undefined;
                this.keepAwake = false;
                // Notify listeners
                this.emit("firmware update finished", this, FirmwareUpdateMetaDataCC_1.FirmwareUpdateStatus.Error_Timeout);
            }
            throw e;
        }
    }
    /**
     * @internal
     * Serializes this node in order to store static data in a cache
     */
    serialize() {
        var _a;
        const ret = {
            id: this.id,
            interviewStage: this.interviewStage >= Types_1.InterviewStage.RestartFromCache
                ? Types_1.InterviewStage[Types_1.InterviewStage.Complete]
                : Types_1.InterviewStage[this.interviewStage],
            deviceClass: this.deviceClass && {
                basic: this.deviceClass.basic.key,
                generic: this.deviceClass.generic.key,
                specific: this.deviceClass.specific.key,
            },
            neighbors: [...this._neighbors].sort(),
            isListening: this.isListening,
            isFrequentListening: this.isFrequentListening,
            isRouting: this.isRouting,
            maxBaudRate: this.maxBaudRate,
            isSecure: (_a = this.isSecure) !== null && _a !== void 0 ? _a : core_1.unknownBoolean,
            isBeaming: this.isBeaming,
            version: this.version,
            commandClasses: {},
        };
        // Sort the CCs by their key before writing to the object
        const sortedCCs = [
            ...this.implementedCommandClasses.keys(),
        ].sort((a, b) => Math.sign(a - b));
        for (const cc of sortedCCs) {
            const serializedCC = {
                name: core_1.CommandClasses[cc],
                endpoints: {},
            };
            // We store the support and version information in this location rather than in the version CC
            // Therefore request the information from all endpoints
            for (const endpoint of this.getAllEndpoints()) {
                if (endpoint.implementedCommandClasses.has(cc)) {
                    serializedCC.endpoints[endpoint.index.toString()] = endpoint.implementedCommandClasses.get(cc);
                }
            }
            ret.commandClasses[shared_1.num2hex(cc)] = serializedCC;
        }
        return ret;
    }
    /**
     * @internal
     * Deserializes the information of this node from a cache.
     */
    async deserialize(obj) {
        if (obj.interviewStage in Types_1.InterviewStage) {
            this.interviewStage =
                typeof obj.interviewStage === "number"
                    ? obj.interviewStage
                    : Types_1.InterviewStage[obj.interviewStage];
        }
        if (typeguards_1.isObject(obj.deviceClass)) {
            const { basic, generic, specific } = obj.deviceClass;
            if (typeof basic === "number" &&
                typeof generic === "number" &&
                typeof specific === "number") {
                this._deviceClass = new DeviceClass_1.DeviceClass(this.driver.configManager, basic, generic, specific);
            }
        }
        // Parse single properties
        const tryParse = (key, type) => {
            if (typeof obj[key] === type)
                this[`_${key}`] = obj[key];
        };
        tryParse("isListening", "boolean");
        tryParse("isFrequentListening", "boolean");
        tryParse("isRouting", "boolean");
        tryParse("maxBaudRate", "number");
        // isSecure may be boolean or "unknown"
        tryParse("isSecure", "string");
        tryParse("isSecure", "boolean");
        tryParse("isBeaming", "boolean");
        tryParse("version", "number");
        if (typeguards_1.isArray(obj.neighbors)) {
            // parse only valid node IDs
            this._neighbors = obj.neighbors.filter((n) => typeof n === "number" && n > 0 && n <= core_1.MAX_NODES);
        }
        function enforceType(val, type) {
            return typeof val === type ? val : undefined;
        }
        // We need to cache the endpoint CC support until all CCs have been deserialized
        const endpointCCSupport = new Map();
        // Parse CommandClasses
        if (typeguards_1.isObject(obj.commandClasses)) {
            const ccDict = obj.commandClasses;
            for (const ccHex of Object.keys(ccDict)) {
                // First make sure this key describes a valid CC
                if (!/^0x[0-9a-fA-F]+$/.test(ccHex))
                    continue;
                const ccNum = parseInt(ccHex);
                if (!(ccNum in core_1.CommandClasses))
                    continue;
                // Parse the information we have
                const { values, metadata, 
                // Starting with v2.4.2, the CC versions are stored in the endpoints object
                endpoints, 
                // These are for compatibility with older versions
                isSupported, isControlled, version, } = ccDict[ccHex];
                if (typeguards_1.isObject(endpoints)) {
                    // New cache file with a dictionary of CC support information
                    const support = new Map();
                    for (const endpointIndex of Object.keys(endpoints)) {
                        // First make sure this key is a number
                        if (!/^\d+$/.test(endpointIndex))
                            continue;
                        const numEndpointIndex = parseInt(endpointIndex, 10);
                        // Verify the info object
                        const info = endpoints[endpointIndex];
                        info.isSupported = enforceType(info.isSupported, "boolean");
                        info.isControlled = enforceType(info.isControlled, "boolean");
                        info.version = enforceType(info.version, "number");
                        // Update the root endpoint immediately, save non-root endpoint information for later
                        if (numEndpointIndex === 0) {
                            this.addCC(ccNum, info);
                        }
                        else {
                            support.set(numEndpointIndex, info);
                        }
                    }
                    endpointCCSupport.set(ccNum, support);
                }
                else {
                    // Legacy cache with single properties for the root endpoint
                    this.addCC(ccNum, {
                        isSupported: enforceType(isSupported, "boolean"),
                        isControlled: enforceType(isControlled, "boolean"),
                        version: enforceType(version, "number"),
                    });
                }
                // In pre-3.0 cache files, the metadata and values array must be deserialized before creating endpoints
                // Post 3.0, the driver takes care of loading them before deserializing nodes
                // In order to understand pre-3.0 cache files, leave this deserialization code in
                // Metadata must be deserialized before values since that may be necessary to correctly translate value IDs
                if (typeguards_1.isArray(metadata) && metadata.length > 0) {
                    // If any exist, deserialize the metadata aswell
                    const ccInstance = this.createCCInstanceUnsafe(ccNum);
                    if (ccInstance) {
                        // In v2.0.0, propertyName was changed to property. The network caches might still reference the old property names
                        for (const m of metadata) {
                            if ("propertyName" in m) {
                                m.property = m.propertyName;
                                delete m.propertyName;
                            }
                        }
                        try {
                            ccInstance.deserializeMetadataFromCache(metadata);
                        }
                        catch (e) {
                            this.driver.controllerLog.logNode(this.id, {
                                message: `Error during deserialization of CC value metadata from cache:\n${e}`,
                                level: "error",
                            });
                        }
                    }
                }
                if (typeguards_1.isArray(values) && values.length > 0) {
                    // If any exist, deserialize the values aswell
                    const ccInstance = this.createCCInstanceUnsafe(ccNum);
                    if (ccInstance) {
                        // In v2.0.0, propertyName was changed to property. The network caches might still reference the old property names
                        for (const v of values) {
                            if ("propertyName" in v) {
                                v.property = v.propertyName;
                                delete v.propertyName;
                            }
                        }
                        try {
                            ccInstance.deserializeValuesFromCache(values);
                        }
                        catch (e) {
                            this.driver.controllerLog.logNode(this.id, {
                                message: `Error during deserialization of CC values from cache:\n${e}`,
                                level: "error",
                            });
                        }
                    }
                }
            }
        }
        // Now restore the CC versions for each non-root endpoint
        for (const [cc, support] of endpointCCSupport) {
            for (const [endpointIndex, info] of support) {
                const endpoint = this.getEndpoint(endpointIndex);
                if (!endpoint)
                    continue;
                endpoint.addCC(cc, info);
            }
        }
        // And restore the device config
        await this.loadDeviceConfig();
    }
    /** Returns whether the node is currently assumed awake */
    isAwake() {
        const isAsleep = this.supportsCC(core_1.CommandClasses["Wake Up"]) &&
            !WakeUpCC_1.WakeUpCC.isAwake(this);
        return !isAsleep;
    }
    /**
     * @internal
     * Sends the node a WakeUpCCNoMoreInformation so it can go back to sleep
     */
    async sendNoMoreInformation() {
        // Don't send the node back to sleep if it should be kept awake
        if (this.keepAwake)
            return false;
        // Avoid calling this method more than once
        if (this.isSendingNoMoreInformation)
            return false;
        this.isSendingNoMoreInformation = true;
        let msgSent = false;
        if (this.isAwake() && this.interviewStage === Types_1.InterviewStage.Complete) {
            this.driver.controllerLog.logNode(this.id, {
                message: "Sending node back to sleep...",
                direction: "outbound",
            });
            try {
                // it is important that we catch errors in this call
                // otherwise, this method will not work anymore because
                // isSendingNoMoreInformation is stuck on `true`
                await this.commandClasses["Wake Up"].sendNoMoreInformation();
                msgSent = true;
            }
            catch (_a) {
                /* ignore */
            }
            finally {
                this.markAsAsleep();
            }
        }
        this.isSendingNoMoreInformation = false;
        return msgSent;
    }
};
ZWaveNode = __decorate([
    shared_1.Mixin([events_1.EventEmitter])
], ZWaveNode);
exports.ZWaveNode = ZWaveNode;

//# sourceMappingURL=Node.js.map
