"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualEndpoint = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const arrays_1 = require("alcalzone-shared/arrays");
const API_1 = require("../commandclass/API");
const CommandClass_1 = require("../commandclass/CommandClass");
/**
 * Represents an endpoint of a virtual (broadcast, multicast) Z-Wave node.
 * This can either be the root device itself (index 0) or a more specific endpoint like a single plug.
 *
 * The endpoint's capabilities are determined by the capabilities of the individual nodes' endpoints.
 */
class VirtualEndpoint {
    constructor(
    /** The virtual node this endpoint belongs to (or undefined if it set later) */
    node, 
    /** The driver instance this endpoint belongs to */
    driver, 
    /** The index of this endpoint. 0 for the root device, 1+ otherwise */
    index) {
        this.driver = driver;
        this.index = index;
        this._commandClassAPIs = new Map();
        this._commandClassAPIsProxy = new Proxy(this._commandClassAPIs, {
            get: (target, ccNameOrId) => {
                // Avoid ultra-weird error messages during testing
                if (process.env.NODE_ENV === "test" &&
                    typeof ccNameOrId === "string" &&
                    (ccNameOrId === "$$typeof" ||
                        ccNameOrId === "constructor" ||
                        ccNameOrId.includes("@@__IMMUTABLE"))) {
                    return undefined;
                }
                if (typeof ccNameOrId === "symbol") {
                    // Allow access to the iterator symbol
                    if (ccNameOrId === Symbol.iterator) {
                        return this.commandClassesIterator;
                    }
                    else if (ccNameOrId === Symbol.toStringTag) {
                        return "[object Object]";
                    }
                    // ignore all other symbols
                    return undefined;
                }
                else {
                    // typeof ccNameOrId === "string"
                    let ccId;
                    // The command classes are exposed to library users by their name or the ID
                    if (/\d+/.test(ccNameOrId)) {
                        // Since this is a property accessor, ccNameOrID is passed as a string,
                        // even when it was a number (CommandClasses)
                        ccId = +ccNameOrId;
                    }
                    else {
                        // If a name was given, retrieve the corresponding ID
                        ccId = core_1.CommandClasses[ccNameOrId];
                        if (ccId == undefined) {
                            throw new core_1.ZWaveError(`Command Class ${ccNameOrId} is not implemented! If you are sure that the name/id is correct, consider opening an issue at https://github.com/AlCalzone/node-zwave-js`, core_1.ZWaveErrorCodes.CC_NotImplemented);
                        }
                    }
                    // When accessing a CC API for the first time, we need to create it
                    if (!target.has(ccId)) {
                        const api = this.createAPI(ccId);
                        target.set(ccId, api);
                    }
                    return target.get(ccId);
                }
            },
        });
        /**
         * Used to iterate over the commandClasses API without throwing errors by accessing unsupported CCs
         */
        this.commandClassesIterator = function* () {
            const allCCs = arrays_1.distinct(this._node.physicalNodes
                .map((n) => n.getEndpoint(this.index))
                .filter((e) => !!e)
                .map((e) => [...e.implementedCommandClasses.keys()])
                .reduce((acc, cur) => [...acc, ...cur], []));
            for (const cc of allCCs) {
                if (this.supportsCC(cc)) {
                    // When a CC is supported, it can still happen that the CC API
                    // cannot be created for virtual endpoints
                    const APIConstructor = CommandClass_1.getAPI(cc);
                    if (shared_1.staticExtends(APIConstructor, API_1.PhysicalCCAPI))
                        continue;
                    yield this.commandClasses[cc];
                }
            }
        }.bind(this);
        if (node)
            this._node = node;
    }
    get node() {
        return this._node;
    }
    /** @internal */
    setNode(node) {
        this._node = node;
    }
    get nodeId() {
        // Use the defined node ID if it exists
        if (this.node.id != undefined)
            return this.node.id;
        // Otherwise deduce it from the physical nodes
        const ret = this.node.physicalNodes.map((n) => n.id);
        if (ret.length === 1)
            return ret[0];
        return ret;
    }
    /** Tests if this endpoint supports the given CommandClass */
    supportsCC(cc) {
        // A virtual endpoints supports a CC if any of the physical endpoints it targets supports the CC non-securely
        // Security S0 does not support broadcast / multicast!
        return this.node.physicalNodes.some((n) => {
            const endpoint = n.getEndpoint(this.index);
            return (endpoint === null || endpoint === void 0 ? void 0 : endpoint.supportsCC(cc)) && !(endpoint === null || endpoint === void 0 ? void 0 : endpoint.isCCSecure(cc));
        });
    }
    /**
     * Retrieves the minimum non-zero version of the given CommandClass the physical endpoints implement
     * Returns 0 if the CC is not supported at all.
     */
    getCCVersion(cc) {
        const nonZeroVersions = this.node.physicalNodes
            .map((n) => { var _a; return (_a = n.getEndpoint(this.index)) === null || _a === void 0 ? void 0 : _a.getCCVersion(cc); })
            .filter((v) => v != undefined && v > 0);
        if (!nonZeroVersions.length)
            return 0;
        return Math.min(...nonZeroVersions);
    }
    /**
     * @internal
     * Creates an API instance for a given command class. Throws if no API is defined.
     * @param ccId The command class to create an API instance for
     */
    createAPI(ccId) {
        const APIConstructor = CommandClass_1.getAPI(ccId);
        const ccName = core_1.CommandClasses[ccId];
        if (APIConstructor == undefined) {
            throw new core_1.ZWaveError(`Command Class ${ccName} (${shared_1.num2hex(ccId)}) has no associated API!`, core_1.ZWaveErrorCodes.CC_NoAPI);
        }
        const apiInstance = new APIConstructor(this.driver, this);
        return new Proxy(apiInstance, {
            get: (target, property) => {
                // Forbid access to the API if it is not supported by the node
                if (property !== "ccId" &&
                    property !== "endpoint" &&
                    property !== "isSupported" &&
                    !target.isSupported()) {
                    const hasNodeId = typeof this.nodeId === "number";
                    throw new core_1.ZWaveError(`${hasNodeId ? "The" : "This"} virtual node${hasNodeId ? ` ${this.nodeId}` : ""}${this.index === 0 ? "" : ` (endpoint ${this.index})`} does not support the Command Class ${ccName}!`, core_1.ZWaveErrorCodes.CC_NotSupported);
                }
                return target[property];
            },
        });
    }
    /**
     * Provides access to simplified APIs that are taylored to specific CCs.
     * Make sure to check support of each API using `API.isSupported()` since
     * all other API calls will throw if the API is not supported
     */
    get commandClasses() {
        return this._commandClassAPIsProxy;
    }
    /**
     * @internal
     * DO NOT CALL THIS!
     */
    getNodeUnsafe() {
        throw new core_1.ZWaveError(`The node of a virtual endpoint cannot be accessed this way!`, core_1.ZWaveErrorCodes.CC_NoNodeID);
    }
}
exports.VirtualEndpoint = VirtualEndpoint;

//# sourceMappingURL=VirtualEndpoint.js.map
