"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PhysicalCCAPI = exports.CCAPI = exports.throwWrongValueType = exports.throwMissingPropertyKey = exports.throwUnsupportedPropertyKey = exports.throwUnsupportedProperty = exports.POLL_VALUE = exports.SET_VALUE = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const typeguards_1 = require("alcalzone-shared/typeguards");
const VirtualEndpoint_1 = require("../node/VirtualEndpoint");
const CommandClass_1 = require("./CommandClass");
/** Used to identify the method on the CC API class that handles setting values on nodes directly */
exports.SET_VALUE = Symbol.for("CCAPI_SET_VALUE");
/** Used to identify the method on the CC API class that handles polling values from nodes */
exports.POLL_VALUE = Symbol.for("CCAPI_POLL_VALUE");
// Since the setValue API is called from a point with very generic parameters,
// we must do narrowing inside the API calls. These three methods are for convenience
function throwUnsupportedProperty(cc, property) {
    throw new core_1.ZWaveError(`${core_1.CommandClasses[cc]}: "${property}" is not a supported property`, core_1.ZWaveErrorCodes.Argument_Invalid);
}
exports.throwUnsupportedProperty = throwUnsupportedProperty;
function throwUnsupportedPropertyKey(cc, property, propertyKey) {
    throw new core_1.ZWaveError(`${core_1.CommandClasses[cc]}: "${propertyKey}" is not a supported property key for property "${property}"`, core_1.ZWaveErrorCodes.Argument_Invalid);
}
exports.throwUnsupportedPropertyKey = throwUnsupportedPropertyKey;
function throwMissingPropertyKey(cc, property) {
    throw new core_1.ZWaveError(`${core_1.CommandClasses[cc]}: property "${property}" requires a property key, but none was given`, core_1.ZWaveErrorCodes.Argument_Invalid);
}
exports.throwMissingPropertyKey = throwMissingPropertyKey;
function throwWrongValueType(cc, property, expectedType, receivedType) {
    throw new core_1.ZWaveError(`${core_1.CommandClasses[cc]}: "${property}" must be of type "${expectedType}", received "${receivedType}"`, core_1.ZWaveErrorCodes.Argument_Invalid);
}
exports.throwWrongValueType = throwWrongValueType;
/**
 * The base class for all CC APIs exposed via `Node.commandClasses.<CCName>`
 * @publicAPI
 */
class CCAPI {
    constructor(driver, endpoint) {
        this.driver = driver;
        this.endpoint = endpoint;
        this.ccId = CommandClass_1.getCommandClass(this);
        this.scheduledPolls = new shared_1.ObjectKeyMap();
    }
    /**
     * Can be used on supported CC APIs to set a CC value by property name (and optionally the property key)
     */
    get setValue() {
        // wotan-disable-next-line no-restricted-property-access
        return this[exports.SET_VALUE];
    }
    /**
     * Can be used on supported CC APIs to poll a CC value by property name (and optionally the property key)
     */
    get pollValue() {
        var _a;
        // wotan-disable-next-line no-restricted-property-access
        const implementation = (_a = this[exports.POLL_VALUE]) === null || _a === void 0 ? void 0 : _a.bind(this);
        if (!implementation)
            return;
        // Polling manually should cancel scheduled polls to avoid polling too often
        // Therefore return a wrapper which takes care of that
        return (property) => {
            // Cancel any scheduled polls
            if (this.scheduledPolls.has(property)) {
                clearTimeout(this.scheduledPolls.get(property));
                this.scheduledPolls.delete(property);
            }
            // Call the implementation
            return implementation(property);
        };
    }
    /**
     * Schedules a value to be polled after a given time. Schedules are deduplicated on a per-property basis.
     * @returns `true` if the poll was scheduled, `false` otherwise
     */
    schedulePoll(property, timeoutMs = this.driver.options.timeouts.refreshValue) {
        if (this.scheduledPolls.has(property))
            return false;
        // wotan-disable-next-line no-restricted-property-access
        if (!this[exports.POLL_VALUE])
            return false;
        this.scheduledPolls.set(property, setTimeout(async () => {
            try {
                await this.pollValue(property);
            }
            catch (_a) {
                /* ignore */
            }
        }, timeoutMs).unref());
        return true;
    }
    /**
     * Retrieves the version of the given CommandClass this endpoint implements
     */
    get version() {
        return this.endpoint.getCCVersion(this.ccId);
    }
    /** Determines if this simplified API instance may be used. */
    isSupported() {
        return (
        // NoOperation is always supported
        this.ccId === core_1.CommandClasses["No Operation"] ||
            // Basic should always be supported. Since we are trying to hide it from library consumers
            // we cannot trust supportsCC to test it
            this.ccId === core_1.CommandClasses.Basic ||
            this.endpoint.supportsCC(this.ccId));
    }
    /**
     * Determine whether the linked node supports a specific command of this command class.
     * "unknown" means that the information has not been received yet
     */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    supportsCommand(command) {
        // This needs to be overwritten per command class. In the default implementation, we don't know anything!
        return core_1.unknownBoolean;
    }
    assertSupportsCommand(commandEnum, command) {
        if (this.supportsCommand(command) !== true) {
            const hasNodeId = typeof this.endpoint.nodeId === "number";
            throw new core_1.ZWaveError(`${hasNodeId
                ? "This virtual node"
                : `Node #${this.endpoint.nodeId}`}${this.endpoint.index > 0
                ? ` (Endpoint ${this.endpoint.index})`
                : ""} does not support the command ${shared_1.getEnumMemberName(commandEnum, command)}!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
    }
    assertPhysicalEndpoint(endpoint) {
        if (endpoint instanceof VirtualEndpoint_1.VirtualEndpoint) {
            throw new core_1.ZWaveError(`This method is not supported for virtual nodes!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
    }
    /** Returns the command options to use for sendCommand calls */
    get commandOptions() {
        // No default options
        return {};
    }
    /** Creates an instance of this API, scoped to use the given options */
    withOptions(options) {
        const mergedOptions = {
            ...this.commandOptions,
            ...options,
        };
        return new Proxy(this, {
            get: (target, property) => {
                if (property === "commandOptions") {
                    return mergedOptions;
                }
                else {
                    return target[property];
                }
            },
        });
    }
    isSinglecast() {
        return (typeof this.endpoint.nodeId === "number" &&
            this.endpoint.nodeId !== core_1.NODE_ID_BROADCAST);
    }
    isMulticast() {
        return (this.endpoint instanceof VirtualEndpoint_1.VirtualEndpoint &&
            typeguards_1.isArray(this.endpoint.nodeId));
    }
    isBroadcast() {
        return (this.endpoint instanceof VirtualEndpoint_1.VirtualEndpoint &&
            this.endpoint.nodeId === core_1.NODE_ID_BROADCAST);
    }
}
exports.CCAPI = CCAPI;
/** A CC API that is only available for physical endpoints */
class PhysicalCCAPI extends CCAPI {
    constructor(driver, endpoint) {
        super(driver, endpoint);
        this.assertPhysicalEndpoint(endpoint);
    }
}
exports.PhysicalCCAPI = PhysicalCCAPI;

//# sourceMappingURL=API.js.map
