import { ZWaveLogContainer } from "@zwave-js/core";
import { BasicDeviceClass, GenericDeviceClass, SpecificDeviceClass } from "./DeviceClasses";
import { DeviceConfig, DeviceConfigIndexEntry, FirmwareVersionRange } from "./Devices";
import { IndicatorProperty } from "./Indicators";
import { Meter, MeterScale } from "./Meters";
import { Notification } from "./Notifications";
import { Scale, ScaleGroup } from "./Scales";
import { SensorType } from "./SensorTypes";
export declare class ConfigManager {
    constructor(container?: ZWaveLogContainer);
    private logger;
    private indicators;
    private indicatorProperties;
    private manufacturers;
    private namedScales;
    private sensorTypes;
    private meters;
    private basicDeviceClasses;
    private genericDeviceClasses;
    private index;
    private notifications;
    loadManufacturers(): Promise<void>;
    saveManufacturers(): Promise<void>;
    /**
     * Looks up the name of the manufacturer with the given ID in the configuration DB
     * @param manufacturerId The manufacturer id to look up
     */
    lookupManufacturer(manufacturerId: number): string | undefined;
    /**
     * Add new manufacturers to configuration DB
     * @param manufacturerId The manufacturer id to look up
     * @param manufacturerName The manufacturer name
     */
    setManufacturer(manufacturerId: number, manufacturerName: string): void;
    loadIndicators(): Promise<void>;
    /**
     * Looks up the label for a given indicator id
     */
    lookupIndicator(indicatorId: number): string | undefined;
    /**
     * Looks up the property definition for a given indicator property id
     */
    lookupProperty(propertyId: number): IndicatorProperty | undefined;
    loadNamedScales(): Promise<void>;
    /**
     * Looks up all scales defined under a given name
     */
    lookupNamedScaleGroup(name: string): ScaleGroup | undefined;
    /** Looks up a scale definition for a given sensor type */
    lookupNamedScale(name: string, scale: number): Scale;
    loadSensorTypes(): Promise<void>;
    /**
     * Looks up the configuration for a given sensor type
     */
    lookupSensorType(sensorType: number): SensorType | undefined;
    /** Looks up a scale definition for a given sensor type */
    lookupSensorScale(sensorType: number, scale: number): Scale;
    getSensorTypeName(sensorType: number): string;
    loadMeters(): Promise<void>;
    /**
     * Looks up the notification configuration for a given notification type
     */
    lookupMeter(meterType: number): Meter | undefined;
    getMeterName(meterType: number): string;
    /** Looks up a scale definition for a given meter type */
    lookupMeterScale(type: number, scale: number): MeterScale;
    loadDeviceClasses(): Promise<void>;
    lookupBasicDeviceClass(basic: number): BasicDeviceClass;
    lookupGenericDeviceClass(generic: number): GenericDeviceClass;
    lookupSpecificDeviceClass(generic: number, specific: number): SpecificDeviceClass;
    loadDeviceIndex(): Promise<void>;
    /**
     * Looks up the definition of a given device in the configuration DB
     * @param manufacturerId The manufacturer id of the device
     * @param productType The product type of the device
     * @param productId The product id of the device
     * @param firmwareVersion If known, configuration for a specific firmware version can be loaded.
     * If this is `undefined` or not given, the first matching file with a defined firmware range will be returned.
     */
    lookupDevice(manufacturerId: number, productType: number, productId: number, firmwareVersion?: string): Promise<DeviceConfig | undefined>;
    /**
     * Adds a given device to the index
     * @param manufacturerId The manufacturer id of the device
     * @param productType The product type of the device
     * @param productId The product id of the device
     * @param filename The path to the json configuration of this device
     * @param firmwareVersionMin Min firmware version
     * @param firmwareVersionMax Max firmware version
     *
     */
    addDeviceToIndex(manufacturerId: number, productType: number, productId: number, filename: string, firmwareVersion?: FirmwareVersionRange | false): void;
    getIndex(): DeviceConfigIndexEntry[] | undefined;
    loadNotifications(): Promise<void>;
    /**
     * Looks up the notification configuration for a given notification type
     */
    lookupNotification(notificationType: number): Notification | undefined;
    /**
     * Looks up the notification configuration for a given notification type.
     * If the config has not been loaded yet, this returns undefined.
     */
    private lookupNotificationUnsafe;
    getNotificationName(notificationType: number): string;
}
//# sourceMappingURL=ConfigManager.d.ts.map