"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Duration = void 0;
const math_1 = require("alcalzone-shared/math");
const ZWaveError_1 = require("../error/ZWaveError");
/** Represents a duration that is used by some command classes */
class Duration {
    constructor(value, unit) {
        this.unit = unit;
        switch (unit) {
            case "minutes":
                // Don't allow 0 minutes as a duration
                if (value === 0)
                    this.unit = "seconds";
                break;
            case "unknown":
            case "default":
                value = 0;
                break;
        }
        this.value = value;
    }
    get value() {
        return this._value;
    }
    set value(v) {
        this._value = math_1.clamp(v, 0, 127);
    }
    /** Parses a duration as represented in Report commands */
    static parseReport(payload) {
        if (payload == undefined)
            return undefined;
        if (payload === 0xff)
            return undefined; // reserved value
        if (payload === 0xfe)
            return new Duration(0, "unknown");
        const isMinutes = !!(payload & 128);
        const value = (payload & 127) + (isMinutes ? 1 : 0); // minutes start at 1
        return new Duration(value, isMinutes ? "minutes" : "seconds");
    }
    /** Parses a duration as represented in Set commands */
    static parseSet(payload) {
        if (payload == undefined)
            return undefined;
        if (payload === 0xff)
            return new Duration(0, "default");
        const isMinutes = !!(payload & 128);
        const value = (payload & 127) + (isMinutes ? 1 : 0); // minutes start at 1
        return new Duration(value, isMinutes ? "minutes" : "seconds");
    }
    /** Serializes a duration for a Set command */
    serializeSet() {
        if (this.unit === "default")
            return 0xff;
        if (this.unit === "unknown")
            throw new ZWaveError_1.ZWaveError("Set commands don't support unknown durations", ZWaveError_1.ZWaveErrorCodes.CC_Invalid);
        const isMinutes = this.unit === "minutes";
        let payload = isMinutes ? 128 : 0;
        payload += (this._value - (isMinutes ? 1 : 0)) & 127;
        return payload;
    }
    /** Serializes a duration for a Report command */
    serializeReport() {
        if (this.unit === "unknown")
            return 0xfe;
        const isMinutes = this.unit === "minutes";
        let payload = isMinutes ? 128 : 0;
        payload += (this._value - (isMinutes ? 1 : 0)) & 127;
        return payload;
    }
    toJSON() {
        if (this.unit === "default" || this.unit === "unknown")
            return this.unit;
        return {
            value: this.value,
            unit: this.unit,
        };
    }
    toMilliseconds() {
        switch (this.unit) {
            case "minutes":
                return this._value * 60000;
            case "seconds":
                return this._value * 1000;
        }
        // The other values have no ms representation
    }
    toString() {
        switch (this.unit) {
            case "minutes":
                return `[Duration: ${this._value}${this.value === 1 ? "minute" : "minutes"}]`;
            case "seconds":
                return `[Duration: ${this._value}${this.value === 1 ? "second" : "seconds"}]`;
            default:
                return `[Duration: ${this.unit}]`;
        }
    }
}
exports.Duration = Duration;
//# sourceMappingURL=Duration.js.map