"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const core_1 = require("@zwave-js/core");
const VirtualEndpoint_1 = require("./VirtualEndpoint");
class VirtualNode extends VirtualEndpoint_1.VirtualEndpoint {
    constructor(id, driver, 
    /** The references to the physical node this virtual node abstracts */
    physicalNodes) {
        // Define this node's intrinsic endpoint as the root device (0)
        super(undefined, driver, 0);
        this.id = id;
        /** Cache for this node's endpoint instances */
        this._endpointInstances = new Map();
        // Set the reference to this and the physical nodes
        super.setNode(this);
        this.physicalNodes = [...physicalNodes].filter(
        // And avoid including the controller node in the support checks
        (n) => n.id !== driver.controller.ownNodeId);
    }
    /**
     * Updates a value for a given property of a given CommandClass.
     * This will communicate with the physical node(s) this virtual node represents!
     */
    async setValue(valueId, value) {
        // Try to retrieve the corresponding CC API
        try {
            // Access the CC API by name
            const endpointInstance = this.getEndpoint(valueId.endpoint || 0);
            if (!endpointInstance)
                return false;
            const api = endpointInstance.commandClasses[valueId.commandClass];
            // Check if the setValue method is implemented
            if (!api.setValue)
                return false;
            // And call it
            await api.setValue({
                property: valueId.property,
                propertyKey: valueId.propertyKey,
            }, value);
            return true;
        }
        catch (e) {
            // Define which errors during setValue are expected and won't crash
            // the driver:
            if (e instanceof core_1.ZWaveError) {
                let handled = false;
                let emitErrorEvent = false;
                switch (e.code) {
                    // This CC or API is not implemented
                    case core_1.ZWaveErrorCodes.CC_NotImplemented:
                    case core_1.ZWaveErrorCodes.CC_NoAPI:
                        handled = true;
                        break;
                    // A user tried to set an invalid value
                    case core_1.ZWaveErrorCodes.Argument_Invalid:
                        handled = true;
                        emitErrorEvent = true;
                        break;
                }
                if (emitErrorEvent)
                    this.driver.emit("error", e);
                if (handled)
                    return false;
            }
            throw e;
        }
    }
    getEndpoint(index) {
        var _a;
        if (index < 0)
            throw new core_1.ZWaveError("The endpoint index must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
        // Zero is the root endpoint - i.e. this node
        if (index === 0)
            return this;
        // Check if the Multi Channel CC interviews for all nodes are completed,
        // because we don't have all the information before that
        if (!this.isMultiChannelInterviewComplete) {
            this.driver.driverLog.print(`Virtual node ${(_a = this.id) !== null && _a !== void 0 ? _a : "??"}, Endpoint ${index}: Trying to access endpoint instance before the Multi Channel interview of all nodes was completed!`, "error");
            return undefined;
        }
        // Check if the requested endpoint exists on any physical node
        if (index > this.getEndpointCount())
            return undefined;
        // Create an endpoint instance if it does not exist
        if (!this._endpointInstances.has(index)) {
            this._endpointInstances.set(index, new VirtualEndpoint_1.VirtualEndpoint(this, this.driver, index));
        }
        return this._endpointInstances.get(index);
    }
    /** Returns the current endpoint count of this virtual node (the maximum in the list of physical nodes) */
    getEndpointCount() {
        let ret = 0;
        for (const node of this.physicalNodes) {
            const count = node.getEndpointCount();
            ret = Math.max(ret, count);
        }
        return ret;
    }
    get isMultiChannelInterviewComplete() {
        for (const node of this.physicalNodes) {
            // wotan-disable-next-line no-restricted-property-access
            if (!node["isMultiChannelInterviewComplete"])
                return false;
        }
        return true;
    }
}
exports.VirtualNode = VirtualNode;

//# sourceMappingURL=VirtualNode.js.map
