/// <reference types="node" />
import { API } from '@sentry/core';
import { DsnProtocol, Event, Response, SentryRequest, SentryRequestType, Session, SessionAggregates, Transport, TransportOptions } from '@sentry/types';
import { PromiseBuffer } from '@sentry/utils';
import * as http from 'http';
import * as https from 'https';
import * as url from 'url';
/**
 * Internal used interface for typescript.
 * @hidden
 */
export interface HTTPModule {
    /**
     * Request wrapper
     * @param options These are {@see TransportOptions}
     * @param callback Callback when request is finished
     */
    request(options: http.RequestOptions | https.RequestOptions | string | url.URL, callback?: (res: http.IncomingMessage) => void): http.ClientRequest;
}
/** Base Transport class implementation */
export declare abstract class BaseTransport implements Transport {
    options: TransportOptions;
    /** The Agent used for corresponding transport */
    module?: HTTPModule;
    /** The Agent used for corresponding transport */
    client?: http.Agent | https.Agent;
    /** API object */
    protected _api: API;
    /** A simple buffer holding all requests. */
    protected readonly _buffer: PromiseBuffer<Response>;
    /** Locks transport after receiving rate limits in a response */
    protected readonly _rateLimits: Record<string, Date>;
    /** Create instance and set this.dsn */
    constructor(options: TransportOptions);
    /**
     * @inheritDoc
     */
    sendEvent(_: Event): PromiseLike<Response>;
    /**
     * @inheritDoc
     */
    close(timeout?: number): PromiseLike<boolean>;
    /**
     * Extracts proxy settings from client options and env variables.
     *
     * Honors `no_proxy` env variable with the highest priority to allow for hosts exclusion.
     *
     * An order of priority for available protocols is:
     * `http`  => `options.httpProxy` | `process.env.http_proxy`
     * `https` => `options.httpsProxy` | `options.httpProxy` | `process.env.https_proxy` | `process.env.http_proxy`
     */
    protected _getProxy(protocol: DsnProtocol): string | undefined;
    /** Returns a build request option object used by request */
    protected _getRequestOptions(uri: url.URL): http.RequestOptions | https.RequestOptions;
    /**
     * Gets the time that given category is disabled until for rate limiting
     */
    protected _disabledUntil(requestType: SentryRequestType): Date;
    /**
     * Checks if a category is rate limited
     */
    protected _isRateLimited(requestType: SentryRequestType): boolean;
    /**
     * Sets internal _rateLimits from incoming headers. Returns true if headers contains a non-empty rate limiting header.
     */
    protected _handleRateLimit(headers: Record<string, string | null>): boolean;
    /** JSDoc */
    protected _send(sentryReq: SentryRequest, originalPayload?: Event | Session | SessionAggregates): Promise<Response>;
}
//# sourceMappingURL=base.d.ts.map