"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompatAddCC = exports.CompatConfig = void 0;
const typeguards_1 = require("alcalzone-shared/typeguards");
const utils_1 = require("./utils");
class CompatConfig {
    constructor(filename, definition) {
        this.valueIdRegex = /^\$value\$\[.+\]$/;
        if (definition.queryOnWakeup != undefined) {
            if (!typeguards_1.isArray(definition.queryOnWakeup) ||
                !definition.queryOnWakeup.every((cmd) => typeguards_1.isArray(cmd) &&
                    cmd.length >= 2 &&
                    typeof cmd[0] === "string" &&
                    typeof cmd[1] === "string" &&
                    cmd
                        .slice(2)
                        .every((arg) => typeof arg === "string" ||
                        typeof arg === "number" ||
                        typeof arg === "boolean"))) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option queryOnWakeup`);
            }
            // Parse "smart" values into partial Value IDs
            this.queryOnWakeup = definition.queryOnWakeup.map((cmd) => cmd.map((arg) => {
                if (typeof arg === "string" &&
                    this.valueIdRegex.test(arg)) {
                    const tuple = JSON.parse(arg.substr("$value$".length));
                    return {
                        property: tuple[0],
                        propertyKey: tuple[1],
                    };
                }
                return arg;
            }));
        }
        if (definition.disableBasicMapping != undefined) {
            if (definition.disableBasicMapping !== true) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option disableBasicMapping`);
            }
            this.disableBasicMapping = definition.disableBasicMapping;
        }
        if (definition.preserveRootApplicationCCValueIDs != undefined) {
            if (definition.preserveRootApplicationCCValueIDs !== true) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option preserveRootApplicationCCValueIDs`);
            }
            this.preserveRootApplicationCCValueIDs =
                definition.preserveRootApplicationCCValueIDs;
        }
        if (definition.skipConfigurationInfoQuery != undefined) {
            if (definition.skipConfigurationInfoQuery !== true) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option skipConfigurationInfoQuery`);
            }
            this.skipConfigurationInfoQuery =
                definition.skipConfigurationInfoQuery;
        }
        if (definition.treatBasicSetAsEvent != undefined) {
            if (definition.treatBasicSetAsEvent !== true) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option treatBasicSetAsEvent`);
            }
            this.treatBasicSetAsEvent = definition.treatBasicSetAsEvent;
        }
        if (definition.manualValueRefreshDelayMs != undefined) {
            if (typeof definition.manualValueRefreshDelayMs !== "number") {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option manualValueRefreshDelayMs must be a number!`);
            }
            if (definition.manualValueRefreshDelayMs % 1 !== 0 ||
                definition.manualValueRefreshDelayMs < 0) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option manualValueRefreshDelayMs must be a non-negative integer!`);
            }
            this.manualValueRefreshDelayMs =
                definition.manualValueRefreshDelayMs;
        }
        if (definition.overrideFloatEncoding != undefined) {
            if (!typeguards_1.isObject(definition.overrideFloatEncoding)) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option overrideFloatEncoding`);
            }
            this.overrideFloatEncoding = {};
            if ("precision" in definition.overrideFloatEncoding) {
                if (typeof definition.overrideFloatEncoding.precision !=
                    "number") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option overrideFloatEncoding.precision must be a number!`);
                }
                if (definition.overrideFloatEncoding.precision % 1 !== 0 ||
                    definition.overrideFloatEncoding.precision < 0) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option overrideFloatEncoding.precision must be a positive integer!`);
                }
                this.overrideFloatEncoding.precision =
                    definition.overrideFloatEncoding.precision;
            }
            if ("size" in definition.overrideFloatEncoding) {
                if (typeof definition.overrideFloatEncoding.size != "number") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option overrideFloatEncoding.size must be a number!`);
                }
                if (definition.overrideFloatEncoding.size % 1 !== 0 ||
                    definition.overrideFloatEncoding.size < 1 ||
                    definition.overrideFloatEncoding.size > 4) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
compat option overrideFloatEncoding.size must be an integer between 1 and 4!`);
                }
                this.overrideFloatEncoding.size =
                    definition.overrideFloatEncoding.size;
            }
            if (Object.keys(this.overrideFloatEncoding).length === 0) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option overrideFloatEncoding: size and/or precision must be specified!`);
            }
        }
        if (definition.commandClasses != undefined) {
            if (!typeguards_1.isObject(definition.commandClasses)) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option commandClasses`);
            }
            if (definition.commandClasses.add != undefined) {
                if (!typeguards_1.isObject(definition.commandClasses.add)) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option commandClasses.add`);
                }
                else if (!Object.keys(definition.commandClasses.add).every((k) => utils_1.hexKeyRegex2Digits.test(k))) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
All keys in compat option commandClasses.add must be 2-digit hex numbers!`);
                }
                else if (!Object.values(definition.commandClasses.add).every((v) => typeguards_1.isObject(v))) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
All values in compat option commandClasses.add must be objects`);
                }
                const addCCs = new Map();
                for (const [cc, info] of Object.entries(definition.commandClasses.add)) {
                    addCCs.set(parseInt(cc), new CompatAddCC(filename, info));
                }
                this.addCCs = addCCs;
            }
            if (definition.commandClasses.remove != undefined) {
                if (!typeguards_1.isObject(definition.commandClasses.remove)) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
error in compat option commandClasses.remove`);
                }
                else if (!Object.keys(definition.commandClasses.remove).every((k) => utils_1.hexKeyRegex2Digits.test(k))) {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
All keys in compat option commandClasses.remove must be 2-digit hex numbers!`);
                }
                const removeCCs = new Map();
                for (const [cc, info] of Object.entries(definition.commandClasses.remove)) {
                    if (typeguards_1.isObject(info) && "endpoints" in info) {
                        if (info.endpoints === "*" ||
                            (typeguards_1.isArray(info.endpoints) &&
                                info.endpoints.every((i) => typeof i === "number"))) {
                            removeCCs.set(parseInt(cc), info.endpoints);
                        }
                        else {
                            utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
Compat option commandClasses.remove has an invalid "endpoints" property. Only "*" and numeric arrays are allowed!`);
                        }
                    }
                    else {
                        utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
All values in compat option commandClasses.remove must be objects with an "endpoints" property!`);
                    }
                }
                this.removeCCs = removeCCs;
            }
        }
    }
}
exports.CompatConfig = CompatConfig;
class CompatAddCC {
    constructor(filename, definition) {
        const endpoints = new Map();
        const parseEndpointInfo = (endpoint, info) => {
            const parsed = {};
            if (info.isSupported != undefined) {
                if (typeof info.isSupported !== "boolean") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
Property isSupported in compat option commandClasses.add, endpoint ${endpoint} must be a boolean!`);
                }
                else {
                    parsed.isSupported = info.isSupported;
                }
            }
            if (info.isControlled != undefined) {
                if (typeof info.isControlled !== "boolean") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
Property isControlled in compat option commandClasses.add, endpoint ${endpoint} must be a boolean!`);
                }
                else {
                    parsed.isControlled = info.isControlled;
                }
            }
            if (info.secure != undefined) {
                if (typeof info.secure !== "boolean") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
Property secure in compat option commandClasses.add, endpoint ${endpoint} must be a boolean!`);
                }
                else {
                    parsed.secure = info.secure;
                }
            }
            if (info.version != undefined) {
                if (typeof info.version !== "number") {
                    utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
Property version in compat option commandClasses.add, endpoint ${endpoint} must be a number!`);
                }
                else {
                    parsed.version = info.version;
                }
            }
            endpoints.set(endpoint, parsed);
        };
        // Parse root endpoint info if given
        if (definition.isSupported != undefined ||
            definition.isControlled != undefined ||
            definition.version != undefined ||
            definition.secure != undefined) {
            // We have info for the root endpoint
            parseEndpointInfo(0, definition);
        }
        // Parse all other endpoints
        if (typeguards_1.isObject(definition.endpoints)) {
            if (!Object.keys(definition.endpoints).every((k) => /^\d+$/.test(k))) {
                utils_1.throwInvalidConfig("devices", `config/devices/${filename}:
invalid endpoint index in compat option commandClasses.add`);
            }
            else {
                for (const [ep, info] of Object.entries(definition.endpoints)) {
                    parseEndpointInfo(parseInt(ep), info);
                }
            }
        }
        this.endpoints = endpoints;
    }
}
exports.CompatAddCC = CompatAddCC;
//# sourceMappingURL=CompatConfig.js.map