"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpecificDeviceClass = exports.GenericDeviceClass = exports.getDefaultSpecificDeviceClass = exports.getDefaultGenericDeviceClass = exports.loadDeviceClassesInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const arrays_1 = require("alcalzone-shared/arrays");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "deviceClasses.json");
/** @internal */
async function loadDeviceClassesInternal() {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The device classes config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("device classes", `the dictionary is not an object`);
        }
        if (!typeguards_1.isObject(definition.basic)) {
            utils_1.throwInvalidConfig("device classes", `The "basic" property is not an object`);
        }
        if (!typeguards_1.isObject(definition.generic)) {
            utils_1.throwInvalidConfig("device classes", `The "generic" property is not an object`);
        }
        const basicDeviceClasses = new Map();
        for (const [key, basicClass] of objects_1.entries(definition.basic)) {
            if (!utils_1.hexKeyRegexNDigits.test(key)) {
                utils_1.throwInvalidConfig("device classes", `found non-hex key "${key}" in the basic device class definition`);
            }
            const keyNum = parseInt(key.slice(2), 16);
            basicDeviceClasses.set(keyNum, basicClass);
        }
        const genericDeviceClasses = new Map();
        for (const [key, genericDefinition] of objects_1.entries(definition.generic)) {
            if (!utils_1.hexKeyRegexNDigits.test(key)) {
                utils_1.throwInvalidConfig("device classes", `found non-hex key "${key}" in the generic device class definition`);
            }
            const keyNum = parseInt(key.slice(2), 16);
            genericDeviceClasses.set(keyNum, new GenericDeviceClass(keyNum, genericDefinition));
        }
        return { basicDeviceClasses, genericDeviceClasses };
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("device classes");
        }
    }
}
exports.loadDeviceClassesInternal = loadDeviceClassesInternal;
function getDefaultGenericDeviceClass(key) {
    return new GenericDeviceClass(key, {
        label: `UNKNOWN (${shared_1.num2hex(key)})`,
    });
}
exports.getDefaultGenericDeviceClass = getDefaultGenericDeviceClass;
function getDefaultSpecificDeviceClass(generic, key) {
    if (key === 0)
        return new SpecificDeviceClass(0x00, {
            label: "Unused",
        }, generic);
    return new SpecificDeviceClass(key, {
        label: `UNKNOWN (${shared_1.num2hex(key)})`,
    }, generic);
}
exports.getDefaultSpecificDeviceClass = getDefaultSpecificDeviceClass;
class GenericDeviceClass {
    constructor(key, definition) {
        this.key = key;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("device classes", `The label for generic device class ${shared_1.num2hex(key)} is not a string!`);
        }
        this.label = definition.label;
        if (definition.zwavePlusDeviceType != undefined) {
            if (typeof definition.zwavePlusDeviceType !== "string") {
                utils_1.throwInvalidConfig("device classes", `The zwavePlusDeviceType for generic device class ${shared_1.num2hex(key)} is not a string!`);
            }
            else {
                this.zwavePlusDeviceType = definition.zwavePlusDeviceType;
            }
        }
        if (definition.supportedCCs != undefined) {
            if (!typeguards_1.isArray(definition.supportedCCs) ||
                !definition.supportedCCs.every((cc) => typeof cc === "string")) {
                utils_1.throwInvalidConfig("device classes", `supportedCCs in device class ${this.label} (${shared_1.num2hex(this.key)}) is not a string array!`);
            }
            const supportedCCs = [];
            for (const ccName of definition.supportedCCs) {
                if (!(ccName in core_1.CommandClasses)) {
                    utils_1.throwInvalidConfig("device classes", `Found unknown CC "${ccName}" in supportedCCs of device class ${this.label} (${shared_1.num2hex(this.key)})!`);
                }
                supportedCCs.push(core_1.CommandClasses[ccName]);
            }
            this.supportedCCs = supportedCCs;
        }
        else {
            this.supportedCCs = [];
        }
        if (definition.controlledCCs != undefined) {
            if (!typeguards_1.isArray(definition.controlledCCs) ||
                !definition.controlledCCs.every((cc) => typeof cc === "string")) {
                utils_1.throwInvalidConfig("device classes", `controlledCCs in device class ${this.label} (${shared_1.num2hex(this.key)}) is not a string array!`);
            }
            const controlledCCs = [];
            for (const ccName of definition.controlledCCs) {
                if (!(ccName in core_1.CommandClasses)) {
                    utils_1.throwInvalidConfig("device classes", `Found unknown CC "${ccName}" in controlledCCs of device class ${this.label} (${shared_1.num2hex(this.key)})!`);
                }
                controlledCCs.push(core_1.CommandClasses[ccName]);
            }
            this.controlledCCs = controlledCCs;
        }
        else {
            this.controlledCCs = [];
        }
        const specific = new Map();
        if (typeguards_1.isObject(definition.specific)) {
            for (const [specificKey, specificDefinition] of objects_1.entries(definition.specific)) {
                if (!utils_1.hexKeyRegexNDigits.test(specificKey))
                    utils_1.throwInvalidConfig("device classes", `found non-hex key "${specificKey}" in device class ${this.label} (${shared_1.num2hex(this.key)})`);
                const specificKeyNum = parseInt(specificKey.slice(2), 16);
                specific.set(specificKeyNum, new SpecificDeviceClass(specificKeyNum, specificDefinition, this));
            }
        }
        this.specific = specific;
    }
}
exports.GenericDeviceClass = GenericDeviceClass;
class SpecificDeviceClass {
    constructor(key, definition, generic) {
        this.key = key;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("device classes", `The label for device class ${generic.label} -> ${shared_1.num2hex(key)} is not a string!`);
        }
        this.label = definition.label;
        if (definition.zwavePlusDeviceType != undefined) {
            if (typeof definition.zwavePlusDeviceType !== "string") {
                utils_1.throwInvalidConfig("device classes", `The zwavePlusDeviceType for device class ${generic.label} -> ${shared_1.num2hex(key)} is not a string!`);
            }
            else {
                this.zwavePlusDeviceType = definition.zwavePlusDeviceType;
            }
        }
        else if (generic.zwavePlusDeviceType != undefined) {
            this.zwavePlusDeviceType = generic.zwavePlusDeviceType;
        }
        if (definition.supportedCCs != undefined) {
            if (!typeguards_1.isArray(definition.supportedCCs) ||
                !definition.supportedCCs.every((cc) => typeof cc === "string")) {
                utils_1.throwInvalidConfig("device classes", `supportedCCs in device class ${generic.label} -> ${this.label} (${shared_1.num2hex(this.key)}) is not a string array!`);
            }
            const supportedCCs = [];
            for (const ccName of definition.supportedCCs) {
                if (!(ccName in core_1.CommandClasses)) {
                    utils_1.throwInvalidConfig("device classes", `Found unknown CC "${ccName}" in supportedCCs of device class ${generic.label} -> ${this.label} (${shared_1.num2hex(this.key)})!`);
                }
                supportedCCs.push(core_1.CommandClasses[ccName]);
            }
            this.supportedCCs = supportedCCs;
        }
        else {
            this.supportedCCs = [];
        }
        this.supportedCCs = arrays_1.distinct([
            ...generic.supportedCCs,
            ...this.supportedCCs,
        ]);
        if (definition.controlledCCs != undefined) {
            if (!typeguards_1.isArray(definition.controlledCCs) ||
                !definition.controlledCCs.every((cc) => typeof cc === "string")) {
                utils_1.throwInvalidConfig("device classes", `controlledCCs in device class ${generic.label} -> ${this.label} (${shared_1.num2hex(this.key)}) is not a string array!`);
            }
            const controlledCCs = [];
            for (const ccName of definition.controlledCCs) {
                if (!(ccName in core_1.CommandClasses)) {
                    utils_1.throwInvalidConfig("device classes", `Found unknown CC "${ccName}" in controlledCCs of device class ${generic.label} -> ${this.label} (${shared_1.num2hex(this.key)})!`);
                }
                controlledCCs.push(core_1.CommandClasses[ccName]);
            }
            this.controlledCCs = controlledCCs;
        }
        else {
            this.controlledCCs = [];
        }
        this.controlledCCs = arrays_1.distinct([
            ...generic.controlledCCs,
            ...this.controlledCCs,
        ]);
    }
}
exports.SpecificDeviceClass = SpecificDeviceClass;
//# sourceMappingURL=DeviceClasses.js.map