"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotificationParameterWithValue = exports.NotificationParameterWithCommandClass = exports.NotificationParameterWithDuration = exports.NotificationParameter = exports.NotificationEvent = exports.NotificationState = exports.NotificationVariable = exports.Notification = exports.loadNotificationsInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "notifications.json");
/** @internal */
async function loadNotificationsInternal() {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("notifications", "the database is not an object");
        }
        const notifications = new Map();
        for (const [id, ntfcnDefinition] of objects_1.entries(definition)) {
            if (!utils_1.hexKeyRegexNDigits.test(id)) {
                utils_1.throwInvalidConfig("notifications", `found non-hex key "${id}" at the root`);
            }
            const idNum = parseInt(id.slice(2), 16);
            notifications.set(idNum, new Notification(idNum, ntfcnDefinition));
        }
        return notifications;
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("notifications");
        }
    }
}
exports.loadNotificationsInternal = loadNotificationsInternal;
class Notification {
    constructor(id, definition) {
        this.id = id;
        this.name = definition.name;
        this.variables = typeguards_1.isArray(definition.variables)
            ? definition.variables.map((v) => new NotificationVariable(v))
            : [];
        const events = new Map();
        if (typeguards_1.isObject(definition.events)) {
            for (const [eventId, eventDefinition] of objects_1.entries(definition.events)) {
                if (!utils_1.hexKeyRegexNDigits.test(eventId)) {
                    utils_1.throwInvalidConfig("notifications", `found non-hex key "${eventId}" in notification ${shared_1.num2hex(id)}`);
                }
                const eventIdNum = parseInt(eventId.slice(2), 16);
                events.set(eventIdNum, new NotificationEvent(eventIdNum, eventDefinition));
            }
        }
        this.events = events;
    }
    lookupValue(value) {
        // Events are easier to look up, do that first
        if (this.events.has(value)) {
            const { id, ...event } = this.events.get(value);
            return {
                type: "event",
                ...event,
            };
        }
        // Then try to find a variable with a matching state
        const variable = this.variables.find((v) => v.states.has(value));
        if (variable) {
            const state = variable.states.get(value);
            return {
                type: "state",
                value,
                idle: variable.idle,
                label: state.label,
                description: state.description,
                variableName: variable.name,
                parameter: state.parameter,
            };
        }
    }
}
exports.Notification = Notification;
class NotificationVariable {
    constructor(definition) {
        this.name = definition.name;
        // By default all notification variables may return to idle
        // Otherwise it must be specified explicitly using `idle: false`
        this.idle = definition.idle !== false;
        if (!typeguards_1.isObject(definition.states)) {
            utils_1.throwInvalidConfig("notifications", `the variable definition for ${this.name} is not an object`);
        }
        const states = new Map();
        for (const [stateId, stateDefinition] of objects_1.entries(definition.states)) {
            if (!utils_1.hexKeyRegexNDigits.test(stateId)) {
                utils_1.throwInvalidConfig("notifications", `found non-hex key "${stateId}" in notification variable ${this.name}`);
            }
            const stateIdNum = parseInt(stateId.slice(2), 16);
            states.set(stateIdNum, new NotificationState(stateIdNum, stateDefinition));
        }
        this.states = states;
    }
}
exports.NotificationVariable = NotificationVariable;
class NotificationState {
    constructor(id, definition) {
        this.id = id;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("notifications", `The label of notification state ${shared_1.num2hex(id)} has a non-string label`);
        }
        this.label = definition.label;
        if (definition.description != undefined &&
            typeof definition.description !== "string") {
            utils_1.throwInvalidConfig("notifications", `The label of notification state ${shared_1.num2hex(id)} has a non-string description`);
        }
        this.description = definition.description;
        if (definition.params != undefined) {
            if (!typeguards_1.isObject(definition.params)) {
                utils_1.throwInvalidConfig("notifications", `The parameter definition of notification state ${shared_1.num2hex(id)} must be an object`);
            }
            else if (typeof definition.params.type !== "string") {
                utils_1.throwInvalidConfig("notifications", `The parameter type of notification state ${shared_1.num2hex(id)} must be a string`);
            }
            this.parameter = new NotificationParameter(definition.params);
        }
    }
}
exports.NotificationState = NotificationState;
class NotificationEvent {
    constructor(id, definition) {
        this.id = id;
        this.label = definition.label;
        this.description = definition.description;
        if (definition.params != undefined) {
            if (!typeguards_1.isObject(definition.params)) {
                utils_1.throwInvalidConfig("notifications", `The parameter definition of notification event ${shared_1.num2hex(id)} must be an object`);
            }
            else if (typeof definition.params.type !== "string") {
                utils_1.throwInvalidConfig("notifications", `The parameter type of notification event ${shared_1.num2hex(id)} must be a string`);
            }
            this.parameter = new NotificationParameter(definition.params);
        }
    }
}
exports.NotificationEvent = NotificationEvent;
class NotificationParameter {
    constructor(definition) {
        // Allow subclassing
        if (new.target !== NotificationParameter)
            return;
        // Return the correct subclass
        switch (definition.type) {
            case "duration":
                return new NotificationParameterWithDuration(definition);
            case "commandclass":
                return new NotificationParameterWithCommandClass(definition);
            case "value":
                return new NotificationParameterWithValue(definition);
            case "enum":
                // TODO
                break;
        }
    }
}
exports.NotificationParameter = NotificationParameter;
/** Marks a notification that contains a duration */
class NotificationParameterWithDuration {
    constructor(_definition) {
        // nothing to do
    }
}
exports.NotificationParameterWithDuration = NotificationParameterWithDuration;
/** Marks a notification that contains a CC */
class NotificationParameterWithCommandClass {
    constructor(_definition) {
        // nothing to do
    }
}
exports.NotificationParameterWithCommandClass = NotificationParameterWithCommandClass;
class NotificationParameterWithValue {
    constructor(definition) {
        if (typeof definition.name !== "string") {
            utils_1.throwInvalidConfig("notifications", `Missing property name definition for Notification parameter with type: "value"!`);
        }
        this.propertyName = definition.name;
    }
}
exports.NotificationParameterWithValue = NotificationParameterWithValue;
//# sourceMappingURL=Notifications.js.map