"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatDate = exports.timespan = exports.highResTimestamp = exports.getDSTInfo = exports.getDefaultDSTInfo = void 0;
const moment_1 = __importDefault(require("moment"));
const moment_timezone_1 = require("moment-timezone");
/**
 * Returns a fallback DSTInfo in case we cannot determine the correct one.
 * This fallback has no additional DST shift.
 * The dummy DST starts on March 31st and ends on October 31st, both times at 01:00 UTC.
 * @param defaultOffset - The offset to use for both standardOffset and dstOffset
 */
function getDefaultDSTInfo(defaultOffset) {
    const thisYear = new Date().getUTCFullYear();
    if (defaultOffset == undefined)
        defaultOffset = -new Date().getTimezoneOffset();
    return {
        startDate: new Date(Date.UTC(thisYear, 2, 31, 1)),
        endDate: new Date(Date.UTC(thisYear, 9, 31, 1)),
        standardOffset: defaultOffset,
        dstOffset: defaultOffset,
    };
}
exports.getDefaultDSTInfo = getDefaultDSTInfo;
/** Returns the current system's daylight savings time information if possible */
function getDSTInfo() {
    const thisYear = new Date().getUTCFullYear();
    // find out which timezone we're in
    const zoneName = moment_timezone_1.tz.guess();
    const zone = moment_timezone_1.tz.zone(zoneName);
    if (!zone)
        return;
    // moment-timezone stores the end dates of each timespan in zone.untils
    // iterate through them to find this year's dates
    const indizes = [];
    const dates = [];
    const offsets = [];
    for (let i = 0; i < zone.untils.length; i++) {
        const date = new Date(zone.untils[i]);
        if (date.getUTCFullYear() === thisYear) {
            indizes.push(i);
            dates.push(date);
            // Javascript has the offsets inverted, we use the normal interpretation
            offsets.push(-zone.offsets[i]);
        }
    }
    // We can only work with exactly two dates -> start and end of DST
    switch (indizes.length) {
        case 1:
            // if we have exactly 1 index, we use that offset information to construct the fallback info
            return getDefaultDSTInfo(offsets[0]);
        case 2:
            // if we have exactly 2 indizes, we know there's a start and end date
            break; // continue further down
        default:
            // otherwise we cannot construct dst info
            return undefined;
    }
    if (offsets[0] > offsets[1]) {
        // index 0 is end of DST, index 1 is start
        return {
            endDate: dates[0],
            startDate: dates[1],
            dstOffset: offsets[0],
            standardOffset: offsets[1],
        };
    }
    else {
        // index 0 is start of DST, index 1 is end
        return {
            startDate: dates[0],
            endDate: dates[1],
            dstOffset: offsets[1],
            standardOffset: offsets[0],
        };
    }
}
exports.getDSTInfo = getDSTInfo;
/** Returns a timestamp with nano-second precision */
function highResTimestamp() {
    const [s, ns] = process.hrtime();
    return s * 1e9 + ns;
}
exports.highResTimestamp = highResTimestamp;
exports.timespan = Object.freeze({
    seconds: (num) => num * 1000,
    minutes: (num) => exports.timespan.seconds(num * 60),
    hours: (num) => exports.timespan.minutes(num * 60),
    days: (num) => exports.timespan.hours(num * 24),
});
function formatDate(date, format) {
    return moment_1.default(date).format(format);
}
exports.formatDate = formatDate;
//# sourceMappingURL=date.js.map