"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateNode = void 0;
var utils_1 = require("./utils");
var types_1 = require("./types");
var utils_2 = require("./utils");
var State_1 = require("./State");
var actionTypes = require("./actionTypes");
var actions_1 = require("./actions");
var environment_1 = require("./environment");
var constants_1 = require("./constants");
var stateUtils_1 = require("./stateUtils");
var Actor_1 = require("./Actor");
var invokeUtils_1 = require("./invokeUtils");
var NULL_EVENT = '';
var STATE_IDENTIFIER = '#';
var WILDCARD = '*';
var EMPTY_OBJECT = {};
var isStateId = function (str) { return str[0] === STATE_IDENTIFIER; };
var createDefaultOptions = function () { return ({
    actions: {},
    guards: {},
    services: {},
    activities: {},
    delays: {}
}); };
var validateArrayifiedTransitions = function (stateNode, event, transitions) {
    var hasNonLastUnguardedTarget = transitions
        .slice(0, -1)
        .some(function (transition) {
        return !('cond' in transition) &&
            !('in' in transition) &&
            (utils_1.isString(transition.target) || utils_1.isMachine(transition.target));
    });
    var eventText = event === NULL_EVENT ? 'the transient event' : "event '" + event + "'";
    utils_1.warn(!hasNonLastUnguardedTarget, "One or more transitions for " + eventText + " on state '" + stateNode.id + "' are unreachable. " +
        "Make sure that the default transition is the last one defined.");
};
var StateNode = /** @class */ (function () {
    function StateNode(
    /**
     * The raw config used to create the machine.
     */
    config, options, 
    /**
     * The initial extended state
     */
    context // TODO: this is unsafe, but we're removing it in v5 anyway
    ) {
        var _this = this;
        if (context === void 0) { context = undefined; }
        var _a;
        this.config = config;
        this.context = context;
        /**
         * The order this state node appears. Corresponds to the implicit SCXML document order.
         */
        this.order = -1;
        this.__xstatenode = true;
        this.__cache = {
            events: undefined,
            relativeValue: new Map(),
            initialStateValue: undefined,
            initialState: undefined,
            on: undefined,
            transitions: undefined,
            candidates: {},
            delayedTransitions: undefined
        };
        this.idMap = {};
        this.tags = [];
        this.options = Object.assign(createDefaultOptions(), options);
        this.parent = this.options._parent;
        this.key =
            this.config.key || this.options._key || this.config.id || '(machine)';
        this.machine = this.parent ? this.parent.machine : this;
        this.path = this.parent ? this.parent.path.concat(this.key) : [];
        this.delimiter =
            this.config.delimiter ||
                (this.parent ? this.parent.delimiter : constants_1.STATE_DELIMITER);
        this.id =
            this.config.id || __spreadArray([this.machine.key], __read(this.path)).join(this.delimiter);
        this.version = this.parent
            ? this.parent.version
            : this.config.version;
        this.type =
            this.config.type ||
                (this.config.parallel
                    ? 'parallel'
                    : this.config.states && utils_1.keys(this.config.states).length
                        ? 'compound'
                        : this.config.history
                            ? 'history'
                            : 'atomic');
        this.schema = this.parent
            ? this.machine.schema
            : (_a = this.config.schema) !== null && _a !== void 0 ? _a : {};
        if (!environment_1.IS_PRODUCTION) {
            utils_1.warn(!('parallel' in this.config), "The \"parallel\" property is deprecated and will be removed in version 4.1. " + (this.config.parallel
                ? "Replace with `type: 'parallel'`"
                : "Use `type: '" + this.type + "'`") + " in the config for state node '" + this.id + "' instead.");
        }
        this.initial = this.config.initial;
        this.states = (this.config.states
            ? utils_1.mapValues(this.config.states, function (stateConfig, key) {
                var _a;
                var stateNode = new StateNode(stateConfig, {
                    _parent: _this,
                    _key: key
                });
                Object.assign(_this.idMap, __assign((_a = {}, _a[stateNode.id] = stateNode, _a), stateNode.idMap));
                return stateNode;
            })
            : EMPTY_OBJECT);
        // Document order
        var order = 0;
        function dfs(stateNode) {
            var e_1, _a;
            stateNode.order = order++;
            try {
                for (var _b = __values(stateUtils_1.getChildren(stateNode)), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var child = _c.value;
                    dfs(child);
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_1) throw e_1.error; }
            }
        }
        dfs(this);
        // History config
        this.history =
            this.config.history === true ? 'shallow' : this.config.history || false;
        this._transient =
            !!this.config.always ||
                (!this.config.on
                    ? false
                    : Array.isArray(this.config.on)
                        ? this.config.on.some(function (_a) {
                            var event = _a.event;
                            return event === NULL_EVENT;
                        })
                        : NULL_EVENT in this.config.on);
        this.strict = !!this.config.strict;
        // TODO: deprecate (entry)
        this.onEntry = utils_1.toArray(this.config.entry || this.config.onEntry).map(function (action) { return actions_1.toActionObject(action); });
        // TODO: deprecate (exit)
        this.onExit = utils_1.toArray(this.config.exit || this.config.onExit).map(function (action) { return actions_1.toActionObject(action); });
        this.meta = this.config.meta;
        this.doneData =
            this.type === 'final'
                ? this.config.data
                : undefined;
        this.invoke = utils_1.toArray(this.config.invoke).map(function (invokeConfig, i) {
            var _a, _b;
            if (utils_1.isMachine(invokeConfig)) {
                _this.machine.options.services = __assign((_a = {}, _a[invokeConfig.id] = invokeConfig, _a), _this.machine.options.services);
                return invokeUtils_1.toInvokeDefinition({
                    src: invokeConfig.id,
                    id: invokeConfig.id
                });
            }
            else if (utils_1.isString(invokeConfig.src)) {
                return invokeUtils_1.toInvokeDefinition(__assign(__assign({}, invokeConfig), { id: invokeConfig.id || invokeConfig.src, src: invokeConfig.src }));
            }
            else if (utils_1.isMachine(invokeConfig.src) || utils_1.isFunction(invokeConfig.src)) {
                var invokeSrc = _this.id + ":invocation[" + i + "]"; // TODO: util function
                _this.machine.options.services = __assign((_b = {}, _b[invokeSrc] = invokeConfig.src, _b), _this.machine.options.services);
                return invokeUtils_1.toInvokeDefinition(__assign(__assign({ id: invokeSrc }, invokeConfig), { src: invokeSrc }));
            }
            else {
                var invokeSource = invokeConfig.src;
                return invokeUtils_1.toInvokeDefinition(__assign(__assign({ id: invokeSource.type }, invokeConfig), { src: invokeSource }));
            }
        });
        this.activities = utils_1.toArray(this.config.activities)
            .concat(this.invoke)
            .map(function (activity) { return actions_1.toActivityDefinition(activity); });
        this.transition = this.transition.bind(this);
        this.tags = utils_1.toArray(this.config.tags);
        // TODO: this is the real fix for initialization once
        // state node getters are deprecated
        // if (!this.parent) {
        //   this._init();
        // }
    }
    StateNode.prototype._init = function () {
        if (this.__cache.transitions) {
            return;
        }
        stateUtils_1.getAllStateNodes(this).forEach(function (stateNode) { return stateNode.on; });
    };
    /**
     * Clones this state machine with custom options and context.
     *
     * @param options Options (actions, guards, activities, services) to recursively merge with the existing options.
     * @param context Custom context (will override predefined context)
     */
    StateNode.prototype.withConfig = function (options, context) {
        if (context === void 0) { context = this.context; }
        var _a = this.options, actions = _a.actions, activities = _a.activities, guards = _a.guards, services = _a.services, delays = _a.delays;
        return new StateNode(this.config, {
            actions: __assign(__assign({}, actions), options.actions),
            activities: __assign(__assign({}, activities), options.activities),
            guards: __assign(__assign({}, guards), options.guards),
            services: __assign(__assign({}, services), options.services),
            delays: __assign(__assign({}, delays), options.delays)
        }, context);
    };
    /**
     * Clones this state machine with custom context.
     *
     * @param context Custom context (will override predefined context, not recursive)
     */
    StateNode.prototype.withContext = function (context) {
        return new StateNode(this.config, this.options, context);
    };
    Object.defineProperty(StateNode.prototype, "definition", {
        /**
         * The well-structured state node definition.
         */
        get: function () {
            return {
                id: this.id,
                key: this.key,
                version: this.version,
                context: this.context,
                type: this.type,
                initial: this.initial,
                history: this.history,
                states: utils_1.mapValues(this.states, function (state) { return state.definition; }),
                on: this.on,
                transitions: this.transitions,
                entry: this.onEntry,
                exit: this.onExit,
                activities: this.activities || [],
                meta: this.meta,
                order: this.order || -1,
                data: this.doneData,
                invoke: this.invoke
            };
        },
        enumerable: false,
        configurable: true
    });
    StateNode.prototype.toJSON = function () {
        return this.definition;
    };
    Object.defineProperty(StateNode.prototype, "on", {
        /**
         * The mapping of events to transitions.
         */
        get: function () {
            if (this.__cache.on) {
                return this.__cache.on;
            }
            var transitions = this.transitions;
            return (this.__cache.on = transitions.reduce(function (map, transition) {
                map[transition.eventType] = map[transition.eventType] || [];
                map[transition.eventType].push(transition);
                return map;
            }, {}));
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StateNode.prototype, "after", {
        get: function () {
            return (this.__cache.delayedTransitions ||
                ((this.__cache.delayedTransitions = this.getDelayedTransitions()),
                    this.__cache.delayedTransitions));
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StateNode.prototype, "transitions", {
        /**
         * All the transitions that can be taken from this state node.
         */
        get: function () {
            return (this.__cache.transitions ||
                ((this.__cache.transitions = this.formatTransitions()),
                    this.__cache.transitions));
        },
        enumerable: false,
        configurable: true
    });
    StateNode.prototype.getCandidates = function (eventName) {
        if (this.__cache.candidates[eventName]) {
            return this.__cache.candidates[eventName];
        }
        var transient = eventName === NULL_EVENT;
        var candidates = this.transitions.filter(function (transition) {
            var sameEventType = transition.eventType === eventName;
            // null events should only match against eventless transitions
            return transient
                ? sameEventType
                : sameEventType || transition.eventType === WILDCARD;
        });
        this.__cache.candidates[eventName] = candidates;
        return candidates;
    };
    /**
     * All delayed transitions from the config.
     */
    StateNode.prototype.getDelayedTransitions = function () {
        var _this = this;
        var afterConfig = this.config.after;
        if (!afterConfig) {
            return [];
        }
        var mutateEntryExit = function (delay, i) {
            var delayRef = utils_1.isFunction(delay) ? _this.id + ":delay[" + i + "]" : delay;
            var eventType = actions_1.after(delayRef, _this.id);
            _this.onEntry.push(actions_1.send(eventType, { delay: delay }));
            _this.onExit.push(actions_1.cancel(eventType));
            return eventType;
        };
        var delayedTransitions = utils_1.isArray(afterConfig)
            ? afterConfig.map(function (transition, i) {
                var eventType = mutateEntryExit(transition.delay, i);
                return __assign(__assign({}, transition), { event: eventType });
            })
            : utils_1.flatten(utils_1.keys(afterConfig).map(function (delay, i) {
                var configTransition = afterConfig[delay];
                var resolvedTransition = utils_1.isString(configTransition)
                    ? { target: configTransition }
                    : configTransition;
                var resolvedDelay = !isNaN(+delay) ? +delay : delay;
                var eventType = mutateEntryExit(resolvedDelay, i);
                return utils_1.toArray(resolvedTransition).map(function (transition) { return (__assign(__assign({}, transition), { event: eventType, delay: resolvedDelay })); });
            }));
        return delayedTransitions.map(function (delayedTransition) {
            var delay = delayedTransition.delay;
            return __assign(__assign({}, _this.formatTransition(delayedTransition)), { delay: delay });
        });
    };
    /**
     * Returns the state nodes represented by the current state value.
     *
     * @param state The state value or State instance
     */
    StateNode.prototype.getStateNodes = function (state) {
        var _a;
        var _this = this;
        if (!state) {
            return [];
        }
        var stateValue = state instanceof State_1.State
            ? state.value
            : utils_1.toStateValue(state, this.delimiter);
        if (utils_1.isString(stateValue)) {
            var initialStateValue = this.getStateNode(stateValue).initial;
            return initialStateValue !== undefined
                ? this.getStateNodes((_a = {}, _a[stateValue] = initialStateValue, _a))
                : [this, this.states[stateValue]];
        }
        var subStateKeys = utils_1.keys(stateValue);
        var subStateNodes = subStateKeys.map(function (subStateKey) { return _this.getStateNode(subStateKey); });
        subStateNodes.push(this);
        return subStateNodes.concat(subStateKeys.reduce(function (allSubStateNodes, subStateKey) {
            var subStateNode = _this.getStateNode(subStateKey).getStateNodes(stateValue[subStateKey]);
            return allSubStateNodes.concat(subStateNode);
        }, []));
    };
    /**
     * Returns `true` if this state node explicitly handles the given event.
     *
     * @param event The event in question
     */
    StateNode.prototype.handles = function (event) {
        var eventType = utils_1.getEventType(event);
        return this.events.includes(eventType);
    };
    /**
     * Resolves the given `state` to a new `State` instance relative to this machine.
     *
     * This ensures that `.events` and `.nextEvents` represent the correct values.
     *
     * @param state The state to resolve
     */
    StateNode.prototype.resolveState = function (state) {
        var configuration = Array.from(stateUtils_1.getConfiguration([], this.getStateNodes(state.value)));
        return new State_1.State(__assign(__assign({}, state), { value: this.resolve(state.value), configuration: configuration, done: stateUtils_1.isInFinalState(configuration, this) }));
    };
    StateNode.prototype.transitionLeafNode = function (stateValue, state, _event) {
        var stateNode = this.getStateNode(stateValue);
        var next = stateNode.next(state, _event);
        if (!next || !next.transitions.length) {
            return this.next(state, _event);
        }
        return next;
    };
    StateNode.prototype.transitionCompoundNode = function (stateValue, state, _event) {
        var subStateKeys = utils_1.keys(stateValue);
        var stateNode = this.getStateNode(subStateKeys[0]);
        var next = stateNode._transition(stateValue[subStateKeys[0]], state, _event);
        if (!next || !next.transitions.length) {
            return this.next(state, _event);
        }
        return next;
    };
    StateNode.prototype.transitionParallelNode = function (stateValue, state, _event) {
        var e_2, _a;
        var transitionMap = {};
        try {
            for (var _b = __values(utils_1.keys(stateValue)), _c = _b.next(); !_c.done; _c = _b.next()) {
                var subStateKey = _c.value;
                var subStateValue = stateValue[subStateKey];
                if (!subStateValue) {
                    continue;
                }
                var subStateNode = this.getStateNode(subStateKey);
                var next = subStateNode._transition(subStateValue, state, _event);
                if (next) {
                    transitionMap[subStateKey] = next;
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_2) throw e_2.error; }
        }
        var stateTransitions = utils_1.keys(transitionMap).map(function (key) { return transitionMap[key]; });
        var enabledTransitions = utils_1.flatten(stateTransitions.map(function (st) { return st.transitions; }));
        var willTransition = stateTransitions.some(function (st) { return st.transitions.length > 0; });
        if (!willTransition) {
            return this.next(state, _event);
        }
        var entryNodes = utils_1.flatten(stateTransitions.map(function (t) { return t.entrySet; }));
        var configuration = utils_1.flatten(utils_1.keys(transitionMap).map(function (key) { return transitionMap[key].configuration; }));
        return {
            transitions: enabledTransitions,
            entrySet: entryNodes,
            exitSet: utils_1.flatten(stateTransitions.map(function (t) { return t.exitSet; })),
            configuration: configuration,
            source: state,
            actions: utils_1.flatten(utils_1.keys(transitionMap).map(function (key) {
                return transitionMap[key].actions;
            }))
        };
    };
    StateNode.prototype._transition = function (stateValue, state, _event) {
        // leaf node
        if (utils_1.isString(stateValue)) {
            return this.transitionLeafNode(stateValue, state, _event);
        }
        // hierarchical node
        if (utils_1.keys(stateValue).length === 1) {
            return this.transitionCompoundNode(stateValue, state, _event);
        }
        // orthogonal node
        return this.transitionParallelNode(stateValue, state, _event);
    };
    StateNode.prototype.next = function (state, _event) {
        var e_3, _a;
        var _this = this;
        var eventName = _event.name;
        var actions = [];
        var nextStateNodes = [];
        var selectedTransition;
        try {
            for (var _b = __values(this.getCandidates(eventName)), _c = _b.next(); !_c.done; _c = _b.next()) {
                var candidate = _c.value;
                var cond = candidate.cond, stateIn = candidate.in;
                var resolvedContext = state.context;
                var isInState = stateIn
                    ? utils_1.isString(stateIn) && isStateId(stateIn)
                        ? // Check if in state by ID
                            state.matches(utils_1.toStateValue(this.getStateNodeById(stateIn).path, this.delimiter))
                        : // Check if in state by relative grandparent
                            utils_2.matchesState(utils_1.toStateValue(stateIn, this.delimiter), utils_1.path(this.path.slice(0, -2))(state.value))
                    : true;
                var guardPassed = false;
                try {
                    guardPassed =
                        !cond ||
                            utils_1.evaluateGuard(this.machine, cond, resolvedContext, _event, state);
                }
                catch (err) {
                    throw new Error("Unable to evaluate guard '" + (cond.name || cond.type) + "' in transition for event '" + eventName + "' in state node '" + this.id + "':\n" + err.message);
                }
                if (guardPassed && isInState) {
                    if (candidate.target !== undefined) {
                        nextStateNodes = candidate.target;
                    }
                    actions.push.apply(actions, __spreadArray([], __read(candidate.actions)));
                    selectedTransition = candidate;
                    break;
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_3) throw e_3.error; }
        }
        if (!selectedTransition) {
            return undefined;
        }
        if (!nextStateNodes.length) {
            return {
                transitions: [selectedTransition],
                entrySet: [],
                exitSet: [],
                configuration: state.value ? [this] : [],
                source: state,
                actions: actions
            };
        }
        var allNextStateNodes = utils_1.flatten(nextStateNodes.map(function (stateNode) {
            return _this.getRelativeStateNodes(stateNode, state.historyValue);
        }));
        var isInternal = !!selectedTransition.internal;
        var reentryNodes = isInternal
            ? []
            : utils_1.flatten(allNextStateNodes.map(function (n) { return _this.nodesFromChild(n); }));
        return {
            transitions: [selectedTransition],
            entrySet: reentryNodes,
            exitSet: isInternal ? [] : [this],
            configuration: allNextStateNodes,
            source: state,
            actions: actions
        };
    };
    StateNode.prototype.nodesFromChild = function (childStateNode) {
        if (childStateNode.escapes(this)) {
            return [];
        }
        var nodes = [];
        var marker = childStateNode;
        while (marker && marker !== this) {
            nodes.push(marker);
            marker = marker.parent;
        }
        nodes.push(this); // inclusive
        return nodes;
    };
    /**
     * Whether the given state node "escapes" this state node. If the `stateNode` is equal to or the parent of
     * this state node, it does not escape.
     */
    StateNode.prototype.escapes = function (stateNode) {
        if (this === stateNode) {
            return false;
        }
        var parent = this.parent;
        while (parent) {
            if (parent === stateNode) {
                return false;
            }
            parent = parent.parent;
        }
        return true;
    };
    StateNode.prototype.getActions = function (transition, currentContext, _event, prevState) {
        var e_4, _a, e_5, _b;
        var prevConfig = stateUtils_1.getConfiguration([], prevState ? this.getStateNodes(prevState.value) : [this]);
        var resolvedConfig = transition.configuration.length
            ? stateUtils_1.getConfiguration(prevConfig, transition.configuration)
            : prevConfig;
        try {
            for (var resolvedConfig_1 = __values(resolvedConfig), resolvedConfig_1_1 = resolvedConfig_1.next(); !resolvedConfig_1_1.done; resolvedConfig_1_1 = resolvedConfig_1.next()) {
                var sn = resolvedConfig_1_1.value;
                if (!stateUtils_1.has(prevConfig, sn)) {
                    transition.entrySet.push(sn);
                }
            }
        }
        catch (e_4_1) { e_4 = { error: e_4_1 }; }
        finally {
            try {
                if (resolvedConfig_1_1 && !resolvedConfig_1_1.done && (_a = resolvedConfig_1.return)) _a.call(resolvedConfig_1);
            }
            finally { if (e_4) throw e_4.error; }
        }
        try {
            for (var prevConfig_1 = __values(prevConfig), prevConfig_1_1 = prevConfig_1.next(); !prevConfig_1_1.done; prevConfig_1_1 = prevConfig_1.next()) {
                var sn = prevConfig_1_1.value;
                if (!stateUtils_1.has(resolvedConfig, sn) || stateUtils_1.has(transition.exitSet, sn.parent)) {
                    transition.exitSet.push(sn);
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (prevConfig_1_1 && !prevConfig_1_1.done && (_b = prevConfig_1.return)) _b.call(prevConfig_1);
            }
            finally { if (e_5) throw e_5.error; }
        }
        if (!transition.source) {
            transition.exitSet = [];
            // Ensure that root StateNode (machine) is entered
            transition.entrySet.push(this);
        }
        var doneEvents = utils_1.flatten(transition.entrySet.map(function (sn) {
            var events = [];
            if (sn.type !== 'final') {
                return events;
            }
            var parent = sn.parent;
            if (!parent.parent) {
                return events;
            }
            events.push(actions_1.done(sn.id, sn.doneData), // TODO: deprecate - final states should not emit done events for their own state.
            actions_1.done(parent.id, sn.doneData
                ? utils_1.mapContext(sn.doneData, currentContext, _event)
                : undefined));
            var grandparent = parent.parent;
            if (grandparent.type === 'parallel') {
                if (stateUtils_1.getChildren(grandparent).every(function (parentNode) {
                    return stateUtils_1.isInFinalState(transition.configuration, parentNode);
                })) {
                    events.push(actions_1.done(grandparent.id));
                }
            }
            return events;
        }));
        transition.exitSet.sort(function (a, b) { return b.order - a.order; });
        transition.entrySet.sort(function (a, b) { return a.order - b.order; });
        var entryStates = new Set(transition.entrySet);
        var exitStates = new Set(transition.exitSet);
        var _c = __read([
            utils_1.flatten(Array.from(entryStates).map(function (stateNode) {
                return __spreadArray(__spreadArray([], __read(stateNode.activities.map(function (activity) { return actions_1.start(activity); }))), __read(stateNode.onEntry));
            })).concat(doneEvents.map(actions_1.raise)),
            utils_1.flatten(Array.from(exitStates).map(function (stateNode) { return __spreadArray(__spreadArray([], __read(stateNode.onExit)), __read(stateNode.activities.map(function (activity) { return actions_1.stop(activity); }))); }))
        ], 2), entryActions = _c[0], exitActions = _c[1];
        var actions = actions_1.toActionObjects(exitActions.concat(transition.actions).concat(entryActions), this.machine.options.actions);
        return actions;
    };
    /**
     * Determines the next state given the current `state` and sent `event`.
     *
     * @param state The current State instance or state value
     * @param event The event that was sent at the current state
     * @param context The current context (extended state) of the current state
     */
    StateNode.prototype.transition = function (state, event, context) {
        if (state === void 0) { state = this
            .initialState; }
        var _event = utils_1.toSCXMLEvent(event);
        var currentState;
        if (state instanceof State_1.State) {
            currentState =
                context === undefined
                    ? state
                    : this.resolveState(State_1.State.from(state, context));
        }
        else {
            var resolvedStateValue = utils_1.isString(state)
                ? this.resolve(utils_1.pathToStateValue(this.getResolvedPath(state)))
                : this.resolve(state);
            var resolvedContext = context ? context : this.machine.context;
            currentState = this.resolveState(State_1.State.from(resolvedStateValue, resolvedContext));
        }
        if (!environment_1.IS_PRODUCTION && _event.name === WILDCARD) {
            throw new Error("An event cannot have the wildcard type ('" + WILDCARD + "')");
        }
        if (this.strict) {
            if (!this.events.includes(_event.name) && !utils_1.isBuiltInEvent(_event.name)) {
                throw new Error("Machine '" + this.id + "' does not accept event '" + _event.name + "'");
            }
        }
        var stateTransition = this._transition(currentState.value, currentState, _event) || {
            transitions: [],
            configuration: [],
            entrySet: [],
            exitSet: [],
            source: currentState,
            actions: []
        };
        var prevConfig = stateUtils_1.getConfiguration([], this.getStateNodes(currentState.value));
        var resolvedConfig = stateTransition.configuration.length
            ? stateUtils_1.getConfiguration(prevConfig, stateTransition.configuration)
            : prevConfig;
        stateTransition.configuration = __spreadArray([], __read(resolvedConfig));
        return this.resolveTransition(stateTransition, currentState, _event);
    };
    StateNode.prototype.resolveRaisedTransition = function (state, _event, originalEvent) {
        var _a;
        var currentActions = state.actions;
        state = this.transition(state, _event);
        // Save original event to state
        // TODO: this should be the raised event! Delete in V5 (breaking)
        state._event = originalEvent;
        state.event = originalEvent.data;
        (_a = state.actions).unshift.apply(_a, __spreadArray([], __read(currentActions)));
        return state;
    };
    StateNode.prototype.resolveTransition = function (stateTransition, currentState, _event, context) {
        var e_6, _a;
        var _this = this;
        if (_event === void 0) { _event = actions_1.initEvent; }
        if (context === void 0) { context = this.machine.context; }
        var configuration = stateTransition.configuration;
        // Transition will "apply" if:
        // - this is the initial state (there is no current state)
        // - OR there are transitions
        var willTransition = !currentState || stateTransition.transitions.length > 0;
        var resolvedStateValue = willTransition
            ? stateUtils_1.getValue(this.machine, configuration)
            : undefined;
        var historyValue = currentState
            ? currentState.historyValue
                ? currentState.historyValue
                : stateTransition.source
                    ? this.machine.historyValue(currentState.value)
                    : undefined
            : undefined;
        var currentContext = currentState ? currentState.context : context;
        var actions = this.getActions(stateTransition, currentContext, _event, currentState);
        var activities = currentState ? __assign({}, currentState.activities) : {};
        try {
            for (var actions_2 = __values(actions), actions_2_1 = actions_2.next(); !actions_2_1.done; actions_2_1 = actions_2.next()) {
                var action = actions_2_1.value;
                if (action.type === actionTypes.start) {
                    activities[action.activity.id || action.activity.type] = action;
                }
                else if (action.type === actionTypes.stop) {
                    activities[action.activity.id || action.activity.type] = false;
                }
            }
        }
        catch (e_6_1) { e_6 = { error: e_6_1 }; }
        finally {
            try {
                if (actions_2_1 && !actions_2_1.done && (_a = actions_2.return)) _a.call(actions_2);
            }
            finally { if (e_6) throw e_6.error; }
        }
        var _b = __read(actions_1.resolveActions(this, currentState, currentContext, _event, actions), 2), resolvedActions = _b[0], updatedContext = _b[1];
        var _c = __read(utils_1.partition(resolvedActions, function (action) {
            return action.type === actionTypes.raise ||
                (action.type === actionTypes.send &&
                    action.to ===
                        types_1.SpecialTargets.Internal);
        }), 2), raisedEvents = _c[0], nonRaisedActions = _c[1];
        var invokeActions = resolvedActions.filter(function (action) {
            var _a;
            return (action.type === actionTypes.start &&
                ((_a = action.activity) === null || _a === void 0 ? void 0 : _a.type) ===
                    actionTypes.invoke);
        });
        var children = invokeActions.reduce(function (acc, action) {
            acc[action.activity.id] = Actor_1.createInvocableActor(action.activity, _this.machine, updatedContext, _event);
            return acc;
        }, currentState
            ? __assign({}, currentState.children) : {});
        var resolvedConfiguration = resolvedStateValue
            ? stateTransition.configuration
            : currentState
                ? currentState.configuration
                : [];
        var meta = resolvedConfiguration.reduce(function (acc, stateNode) {
            if (stateNode.meta !== undefined) {
                acc[stateNode.id] = stateNode.meta;
            }
            return acc;
        }, {});
        var isDone = stateUtils_1.isInFinalState(resolvedConfiguration, this);
        var nextState = new State_1.State({
            value: resolvedStateValue || currentState.value,
            context: updatedContext,
            _event: _event,
            // Persist _sessionid between states
            _sessionid: currentState ? currentState._sessionid : null,
            historyValue: resolvedStateValue
                ? historyValue
                    ? utils_1.updateHistoryValue(historyValue, resolvedStateValue)
                    : undefined
                : currentState
                    ? currentState.historyValue
                    : undefined,
            history: !resolvedStateValue || stateTransition.source
                ? currentState
                : undefined,
            actions: resolvedStateValue ? nonRaisedActions : [],
            activities: resolvedStateValue
                ? activities
                : currentState
                    ? currentState.activities
                    : {},
            meta: resolvedStateValue
                ? meta
                : currentState
                    ? currentState.meta
                    : undefined,
            events: [],
            configuration: resolvedConfiguration,
            transitions: stateTransition.transitions,
            children: children,
            done: isDone,
            tags: currentState === null || currentState === void 0 ? void 0 : currentState.tags
        });
        var didUpdateContext = currentContext !== updatedContext;
        nextState.changed = _event.name === actionTypes.update || didUpdateContext;
        // Dispose of penultimate histories to prevent memory leaks
        var history = nextState.history;
        if (history) {
            delete history.history;
        }
        if (!resolvedStateValue) {
            return nextState;
        }
        var maybeNextState = nextState;
        if (!isDone) {
            var isTransient = this._transient ||
                configuration.some(function (stateNode) {
                    return stateNode._transient;
                });
            if (isTransient) {
                maybeNextState = this.resolveRaisedTransition(maybeNextState, {
                    type: actionTypes.nullEvent
                }, _event);
            }
            while (raisedEvents.length) {
                var raisedEvent = raisedEvents.shift();
                maybeNextState = this.resolveRaisedTransition(maybeNextState, raisedEvent._event, _event);
            }
        }
        // Detect if state changed
        var changed = maybeNextState.changed ||
            (history
                ? !!maybeNextState.actions.length ||
                    didUpdateContext ||
                    typeof history.value !== typeof maybeNextState.value ||
                    !State_1.stateValuesEqual(maybeNextState.value, history.value)
                : undefined);
        maybeNextState.changed = changed;
        // Preserve original history after raised events
        maybeNextState.history = history;
        maybeNextState.tags = new Set(utils_1.flatten(maybeNextState.configuration.map(function (sn) { return sn.tags; })));
        return maybeNextState;
    };
    /**
     * Returns the child state node from its relative `stateKey`, or throws.
     */
    StateNode.prototype.getStateNode = function (stateKey) {
        if (isStateId(stateKey)) {
            return this.machine.getStateNodeById(stateKey);
        }
        if (!this.states) {
            throw new Error("Unable to retrieve child state '" + stateKey + "' from '" + this.id + "'; no child states exist.");
        }
        var result = this.states[stateKey];
        if (!result) {
            throw new Error("Child state '" + stateKey + "' does not exist on '" + this.id + "'");
        }
        return result;
    };
    /**
     * Returns the state node with the given `stateId`, or throws.
     *
     * @param stateId The state ID. The prefix "#" is removed.
     */
    StateNode.prototype.getStateNodeById = function (stateId) {
        var resolvedStateId = isStateId(stateId)
            ? stateId.slice(STATE_IDENTIFIER.length)
            : stateId;
        if (resolvedStateId === this.id) {
            return this;
        }
        var stateNode = this.machine.idMap[resolvedStateId];
        if (!stateNode) {
            throw new Error("Child state node '#" + resolvedStateId + "' does not exist on machine '" + this.id + "'");
        }
        return stateNode;
    };
    /**
     * Returns the relative state node from the given `statePath`, or throws.
     *
     * @param statePath The string or string array relative path to the state node.
     */
    StateNode.prototype.getStateNodeByPath = function (statePath) {
        if (typeof statePath === 'string' && isStateId(statePath)) {
            try {
                return this.getStateNodeById(statePath.slice(1));
            }
            catch (e) {
                // try individual paths
                // throw e;
            }
        }
        var arrayStatePath = utils_1.toStatePath(statePath, this.delimiter).slice();
        var currentStateNode = this;
        while (arrayStatePath.length) {
            var key = arrayStatePath.shift();
            if (!key.length) {
                break;
            }
            currentStateNode = currentStateNode.getStateNode(key);
        }
        return currentStateNode;
    };
    /**
     * Resolves a partial state value with its full representation in this machine.
     *
     * @param stateValue The partial state value to resolve.
     */
    StateNode.prototype.resolve = function (stateValue) {
        var _a;
        var _this = this;
        if (!stateValue) {
            return this.initialStateValue || EMPTY_OBJECT; // TODO: type-specific properties
        }
        switch (this.type) {
            case 'parallel':
                return utils_1.mapValues(this.initialStateValue, function (subStateValue, subStateKey) {
                    return subStateValue
                        ? _this.getStateNode(subStateKey).resolve(stateValue[subStateKey] || subStateValue)
                        : EMPTY_OBJECT;
                });
            case 'compound':
                if (utils_1.isString(stateValue)) {
                    var subStateNode = this.getStateNode(stateValue);
                    if (subStateNode.type === 'parallel' ||
                        subStateNode.type === 'compound') {
                        return _a = {}, _a[stateValue] = subStateNode.initialStateValue, _a;
                    }
                    return stateValue;
                }
                if (!utils_1.keys(stateValue).length) {
                    return this.initialStateValue || {};
                }
                return utils_1.mapValues(stateValue, function (subStateValue, subStateKey) {
                    return subStateValue
                        ? _this.getStateNode(subStateKey).resolve(subStateValue)
                        : EMPTY_OBJECT;
                });
            default:
                return stateValue || EMPTY_OBJECT;
        }
    };
    StateNode.prototype.getResolvedPath = function (stateIdentifier) {
        if (isStateId(stateIdentifier)) {
            var stateNode = this.machine.idMap[stateIdentifier.slice(STATE_IDENTIFIER.length)];
            if (!stateNode) {
                throw new Error("Unable to find state node '" + stateIdentifier + "'");
            }
            return stateNode.path;
        }
        return utils_1.toStatePath(stateIdentifier, this.delimiter);
    };
    Object.defineProperty(StateNode.prototype, "initialStateValue", {
        get: function () {
            var _a;
            if (this.__cache.initialStateValue) {
                return this.__cache.initialStateValue;
            }
            var initialStateValue;
            if (this.type === 'parallel') {
                initialStateValue = utils_1.mapFilterValues(this.states, function (state) { return state.initialStateValue || EMPTY_OBJECT; }, function (stateNode) { return !(stateNode.type === 'history'); });
            }
            else if (this.initial !== undefined) {
                if (!this.states[this.initial]) {
                    throw new Error("Initial state '" + this.initial + "' not found on '" + this.key + "'");
                }
                initialStateValue = (stateUtils_1.isLeafNode(this.states[this.initial])
                    ? this.initial
                    : (_a = {},
                        _a[this.initial] = this.states[this.initial].initialStateValue,
                        _a));
            }
            else {
                // The finite state value of a machine without child states is just an empty object
                initialStateValue = {};
            }
            this.__cache.initialStateValue = initialStateValue;
            return this.__cache.initialStateValue;
        },
        enumerable: false,
        configurable: true
    });
    StateNode.prototype.getInitialState = function (stateValue, context) {
        var configuration = this.getStateNodes(stateValue);
        return this.resolveTransition({
            configuration: configuration,
            entrySet: configuration,
            exitSet: [],
            transitions: [],
            source: undefined,
            actions: []
        }, undefined, undefined, context);
    };
    Object.defineProperty(StateNode.prototype, "initialState", {
        /**
         * The initial State instance, which includes all actions to be executed from
         * entering the initial state.
         */
        get: function () {
            this._init(); // TODO: this should be in the constructor (see note in constructor)
            var initialStateValue = this.initialStateValue;
            if (!initialStateValue) {
                throw new Error("Cannot retrieve initial state from simple state '" + this.id + "'.");
            }
            return this.getInitialState(initialStateValue);
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StateNode.prototype, "target", {
        /**
         * The target state value of the history state node, if it exists. This represents the
         * default state value to transition to if no history value exists yet.
         */
        get: function () {
            var target;
            if (this.type === 'history') {
                var historyConfig = this.config;
                if (utils_1.isString(historyConfig.target)) {
                    target = isStateId(historyConfig.target)
                        ? utils_1.pathToStateValue(this.machine
                            .getStateNodeById(historyConfig.target)
                            .path.slice(this.path.length - 1))
                        : historyConfig.target;
                }
                else {
                    target = historyConfig.target;
                }
            }
            return target;
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Returns the leaf nodes from a state path relative to this state node.
     *
     * @param relativeStateId The relative state path to retrieve the state nodes
     * @param history The previous state to retrieve history
     * @param resolve Whether state nodes should resolve to initial child state nodes
     */
    StateNode.prototype.getRelativeStateNodes = function (relativeStateId, historyValue, resolve) {
        if (resolve === void 0) { resolve = true; }
        return resolve
            ? relativeStateId.type === 'history'
                ? relativeStateId.resolveHistory(historyValue)
                : relativeStateId.initialStateNodes
            : [relativeStateId];
    };
    Object.defineProperty(StateNode.prototype, "initialStateNodes", {
        get: function () {
            var _this = this;
            if (stateUtils_1.isLeafNode(this)) {
                return [this];
            }
            // Case when state node is compound but no initial state is defined
            if (this.type === 'compound' && !this.initial) {
                if (!environment_1.IS_PRODUCTION) {
                    utils_1.warn(false, "Compound state node '" + this.id + "' has no initial state.");
                }
                return [this];
            }
            var initialStateNodePaths = utils_1.toStatePaths(this.initialStateValue);
            return utils_1.flatten(initialStateNodePaths.map(function (initialPath) {
                return _this.getFromRelativePath(initialPath);
            }));
        },
        enumerable: false,
        configurable: true
    });
    /**
     * Retrieves state nodes from a relative path to this state node.
     *
     * @param relativePath The relative path from this state node
     * @param historyValue
     */
    StateNode.prototype.getFromRelativePath = function (relativePath) {
        if (!relativePath.length) {
            return [this];
        }
        var _a = __read(relativePath), stateKey = _a[0], childStatePath = _a.slice(1);
        if (!this.states) {
            throw new Error("Cannot retrieve subPath '" + stateKey + "' from node with no states");
        }
        var childStateNode = this.getStateNode(stateKey);
        if (childStateNode.type === 'history') {
            return childStateNode.resolveHistory();
        }
        if (!this.states[stateKey]) {
            throw new Error("Child state '" + stateKey + "' does not exist on '" + this.id + "'");
        }
        return this.states[stateKey].getFromRelativePath(childStatePath);
    };
    StateNode.prototype.historyValue = function (relativeStateValue) {
        if (!utils_1.keys(this.states).length) {
            return undefined;
        }
        return {
            current: relativeStateValue || this.initialStateValue,
            states: utils_1.mapFilterValues(this.states, function (stateNode, key) {
                if (!relativeStateValue) {
                    return stateNode.historyValue();
                }
                var subStateValue = utils_1.isString(relativeStateValue)
                    ? undefined
                    : relativeStateValue[key];
                return stateNode.historyValue(subStateValue || stateNode.initialStateValue);
            }, function (stateNode) { return !stateNode.history; })
        };
    };
    /**
     * Resolves to the historical value(s) of the parent state node,
     * represented by state nodes.
     *
     * @param historyValue
     */
    StateNode.prototype.resolveHistory = function (historyValue) {
        var _this = this;
        if (this.type !== 'history') {
            return [this];
        }
        var parent = this.parent;
        if (!historyValue) {
            var historyTarget = this.target;
            return historyTarget
                ? utils_1.flatten(utils_1.toStatePaths(historyTarget).map(function (relativeChildPath) {
                    return parent.getFromRelativePath(relativeChildPath);
                }))
                : parent.initialStateNodes;
        }
        var subHistoryValue = utils_1.nestedPath(parent.path, 'states')(historyValue).current;
        if (utils_1.isString(subHistoryValue)) {
            return [parent.getStateNode(subHistoryValue)];
        }
        return utils_1.flatten(utils_1.toStatePaths(subHistoryValue).map(function (subStatePath) {
            return _this.history === 'deep'
                ? parent.getFromRelativePath(subStatePath)
                : [parent.states[subStatePath[0]]];
        }));
    };
    Object.defineProperty(StateNode.prototype, "stateIds", {
        /**
         * All the state node IDs of this state node and its descendant state nodes.
         */
        get: function () {
            var _this = this;
            var childStateIds = utils_1.flatten(utils_1.keys(this.states).map(function (stateKey) {
                return _this.states[stateKey].stateIds;
            }));
            return [this.id].concat(childStateIds);
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StateNode.prototype, "events", {
        /**
         * All the event types accepted by this state node and its descendants.
         */
        get: function () {
            var e_7, _a, e_8, _b;
            if (this.__cache.events) {
                return this.__cache.events;
            }
            var states = this.states;
            var events = new Set(this.ownEvents);
            if (states) {
                try {
                    for (var _c = __values(utils_1.keys(states)), _d = _c.next(); !_d.done; _d = _c.next()) {
                        var stateId = _d.value;
                        var state = states[stateId];
                        if (state.states) {
                            try {
                                for (var _e = (e_8 = void 0, __values(state.events)), _f = _e.next(); !_f.done; _f = _e.next()) {
                                    var event_1 = _f.value;
                                    events.add("" + event_1);
                                }
                            }
                            catch (e_8_1) { e_8 = { error: e_8_1 }; }
                            finally {
                                try {
                                    if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
                                }
                                finally { if (e_8) throw e_8.error; }
                            }
                        }
                    }
                }
                catch (e_7_1) { e_7 = { error: e_7_1 }; }
                finally {
                    try {
                        if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
                    }
                    finally { if (e_7) throw e_7.error; }
                }
            }
            return (this.__cache.events = Array.from(events));
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(StateNode.prototype, "ownEvents", {
        /**
         * All the events that have transitions directly from this state node.
         *
         * Excludes any inert events.
         */
        get: function () {
            var events = new Set(this.transitions
                .filter(function (transition) {
                return !(!transition.target &&
                    !transition.actions.length &&
                    transition.internal);
            })
                .map(function (transition) { return transition.eventType; }));
            return Array.from(events);
        },
        enumerable: false,
        configurable: true
    });
    StateNode.prototype.resolveTarget = function (_target) {
        var _this = this;
        if (_target === undefined) {
            // an undefined target signals that the state node should not transition from that state when receiving that event
            return undefined;
        }
        return _target.map(function (target) {
            if (!utils_1.isString(target)) {
                return target;
            }
            var isInternalTarget = target[0] === _this.delimiter;
            // If internal target is defined on machine,
            // do not include machine key on target
            if (isInternalTarget && !_this.parent) {
                return _this.getStateNodeByPath(target.slice(1));
            }
            var resolvedTarget = isInternalTarget ? _this.key + target : target;
            if (_this.parent) {
                try {
                    var targetStateNode = _this.parent.getStateNodeByPath(resolvedTarget);
                    return targetStateNode;
                }
                catch (err) {
                    throw new Error("Invalid transition definition for state node '" + _this.id + "':\n" + err.message);
                }
            }
            else {
                return _this.getStateNodeByPath(resolvedTarget);
            }
        });
    };
    StateNode.prototype.formatTransition = function (transitionConfig) {
        var _this = this;
        var normalizedTarget = utils_1.normalizeTarget(transitionConfig.target);
        var internal = 'internal' in transitionConfig
            ? transitionConfig.internal
            : normalizedTarget
                ? normalizedTarget.some(function (_target) { return utils_1.isString(_target) && _target[0] === _this.delimiter; })
                : true;
        var guards = this.machine.options.guards;
        var target = this.resolveTarget(normalizedTarget);
        var transition = __assign(__assign({}, transitionConfig), { actions: actions_1.toActionObjects(utils_1.toArray(transitionConfig.actions)), cond: utils_1.toGuard(transitionConfig.cond, guards), target: target, source: this, internal: internal, eventType: transitionConfig.event, toJSON: function () { return (__assign(__assign({}, transition), { target: transition.target
                    ? transition.target.map(function (t) { return "#" + t.id; })
                    : undefined, source: "#" + _this.id })); } });
        return transition;
    };
    StateNode.prototype.formatTransitions = function () {
        var e_9, _a;
        var _this = this;
        var onConfig;
        if (!this.config.on) {
            onConfig = [];
        }
        else if (Array.isArray(this.config.on)) {
            onConfig = this.config.on;
        }
        else {
            var _b = this.config.on, _c = WILDCARD, _d = _b[_c], wildcardConfigs = _d === void 0 ? [] : _d, strictTransitionConfigs_1 = __rest(_b, [typeof _c === "symbol" ? _c : _c + ""]);
            onConfig = utils_1.flatten(utils_1.keys(strictTransitionConfigs_1)
                .map(function (key) {
                if (!environment_1.IS_PRODUCTION && key === NULL_EVENT) {
                    utils_1.warn(false, "Empty string transition configs (e.g., `{ on: { '': ... }}`) for transient transitions are deprecated. Specify the transition in the `{ always: ... }` property instead. " +
                        ("Please check the `on` configuration for \"#" + _this.id + "\"."));
                }
                var transitionConfigArray = utils_1.toTransitionConfigArray(key, strictTransitionConfigs_1[key]);
                if (!environment_1.IS_PRODUCTION) {
                    validateArrayifiedTransitions(_this, key, transitionConfigArray);
                }
                return transitionConfigArray;
            })
                .concat(utils_1.toTransitionConfigArray(WILDCARD, wildcardConfigs)));
        }
        var eventlessConfig = this.config.always
            ? utils_1.toTransitionConfigArray('', this.config.always)
            : [];
        var doneConfig = this.config.onDone
            ? utils_1.toTransitionConfigArray(String(actions_1.done(this.id)), this.config.onDone)
            : [];
        if (!environment_1.IS_PRODUCTION) {
            utils_1.warn(!(this.config.onDone && !this.parent), "Root nodes cannot have an \".onDone\" transition. Please check the config of \"" + this.id + "\".");
        }
        var invokeConfig = utils_1.flatten(this.invoke.map(function (invokeDef) {
            var settleTransitions = [];
            if (invokeDef.onDone) {
                settleTransitions.push.apply(settleTransitions, __spreadArray([], __read(utils_1.toTransitionConfigArray(String(actions_1.doneInvoke(invokeDef.id)), invokeDef.onDone))));
            }
            if (invokeDef.onError) {
                settleTransitions.push.apply(settleTransitions, __spreadArray([], __read(utils_1.toTransitionConfigArray(String(actions_1.error(invokeDef.id)), invokeDef.onError))));
            }
            return settleTransitions;
        }));
        var delayedTransitions = this.after;
        var formattedTransitions = utils_1.flatten(__spreadArray(__spreadArray(__spreadArray(__spreadArray([], __read(doneConfig)), __read(invokeConfig)), __read(onConfig)), __read(eventlessConfig)).map(function (transitionConfig) {
            return utils_1.toArray(transitionConfig).map(function (transition) {
                return _this.formatTransition(transition);
            });
        }));
        try {
            for (var delayedTransitions_1 = __values(delayedTransitions), delayedTransitions_1_1 = delayedTransitions_1.next(); !delayedTransitions_1_1.done; delayedTransitions_1_1 = delayedTransitions_1.next()) {
                var delayedTransition = delayedTransitions_1_1.value;
                formattedTransitions.push(delayedTransition);
            }
        }
        catch (e_9_1) { e_9 = { error: e_9_1 }; }
        finally {
            try {
                if (delayedTransitions_1_1 && !delayedTransitions_1_1.done && (_a = delayedTransitions_1.return)) _a.call(delayedTransitions_1);
            }
            finally { if (e_9) throw e_9.error; }
        }
        return formattedTransitions;
    };
    return StateNode;
}());
exports.StateNode = StateNode;
