"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionCCZWaveSoftwareGet = exports.VersionCCZWaveSoftwareReport = exports.VersionCCCapabilitiesGet = exports.VersionCCCapabilitiesReport = exports.VersionCCCommandClassGet = exports.VersionCCCommandClassReport = exports.VersionCCGet = exports.VersionCCReport = exports.VersionCC = exports.VersionCCAPI = exports.VersionCommand = exports.getFirmwareVersionsValueId = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const ZWaveLibraryTypes_1 = require("../controller/ZWaveLibraryTypes");
const Constants_1 = require("../message/Constants");
const API_1 = require("./API");
const CommandClass_1 = require("./CommandClass");
function getFirmwareVersionsValueId() {
    return {
        commandClass: core_1.CommandClasses.Version,
        property: "firmwareVersions",
    };
}
exports.getFirmwareVersionsValueId = getFirmwareVersionsValueId;
var VersionCommand;
(function (VersionCommand) {
    VersionCommand[VersionCommand["Get"] = 17] = "Get";
    VersionCommand[VersionCommand["Report"] = 18] = "Report";
    VersionCommand[VersionCommand["CommandClassGet"] = 19] = "CommandClassGet";
    VersionCommand[VersionCommand["CommandClassReport"] = 20] = "CommandClassReport";
    VersionCommand[VersionCommand["CapabilitiesGet"] = 21] = "CapabilitiesGet";
    VersionCommand[VersionCommand["CapabilitiesReport"] = 22] = "CapabilitiesReport";
    VersionCommand[VersionCommand["ZWaveSoftwareGet"] = 23] = "ZWaveSoftwareGet";
    VersionCommand[VersionCommand["ZWaveSoftwareReport"] = 24] = "ZWaveSoftwareReport";
})(VersionCommand = exports.VersionCommand || (exports.VersionCommand = {}));
function parseVersion(buffer) {
    if (buffer[0] === 0 && buffer[1] === 0 && buffer[2] === 0)
        return "unused";
    return `${buffer[0]}.${buffer[1]}.${buffer[2]}`;
}
// @noSetValueAPI This CC is read-only
let VersionCCAPI = class VersionCCAPI extends API_1.PhysicalCCAPI {
    supportsCommand(cmd) {
        switch (cmd) {
            case VersionCommand.Get:
            case VersionCommand.CommandClassGet:
                return true; // This is mandatory
            case VersionCommand.CapabilitiesGet:
                // The API might have been created before the versions were determined,
                // so `this.version` may contains a wrong value
                return (this.driver.getSafeCCVersionForNode(this.ccId, this.endpoint.nodeId, this.endpoint.index) >= 3);
            case VersionCommand.ZWaveSoftwareGet: {
                const node = this.endpoint.getNodeUnsafe();
                let ret = node.getValue({
                    commandClass: CommandClass_1.getCommandClass(this),
                    endpoint: this.endpoint.index,
                    property: "supportsZWaveSoftwareGet",
                });
                if (ret == undefined)
                    ret = core_1.unknownBoolean;
                return ret;
            }
        }
        return super.supportsCommand(cmd);
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async get() {
        this.assertSupportsCommand(VersionCommand, VersionCommand.Get);
        const cc = new VersionCCGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, [
                "libraryType",
                "protocolVersion",
                "firmwareVersions",
                "hardwareVersion",
            ]);
        }
    }
    async getCCVersion(requestedCC) {
        this.assertSupportsCommand(VersionCommand, VersionCommand.CommandClassGet);
        const cc = new VersionCCCommandClassGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
            requestedCC,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        return response === null || response === void 0 ? void 0 : response.ccVersion;
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getCapabilities() {
        this.assertSupportsCommand(VersionCommand, VersionCommand.CapabilitiesGet);
        const cc = new VersionCCCapabilitiesGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, ["supportsZWaveSoftwareGet"]);
        }
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getZWaveSoftware() {
        this.assertSupportsCommand(VersionCommand, VersionCommand.ZWaveSoftwareGet);
        const cc = new VersionCCZWaveSoftwareGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            endpoint: this.endpoint.index,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, [
                "sdkVersion",
                "applicationFrameworkAPIVersion",
                "applicationFrameworkBuildNumber",
                "hostInterfaceVersion",
                "hostInterfaceBuildNumber",
                "zWaveProtocolVersion",
                "zWaveProtocolBuildNumber",
                "applicationVersion",
                "applicationBuildNumber",
            ]);
        }
    }
};
VersionCCAPI = __decorate([
    CommandClass_1.API(core_1.CommandClasses.Version)
], VersionCCAPI);
exports.VersionCCAPI = VersionCCAPI;
let VersionCC = class VersionCC extends CommandClass_1.CommandClass {
    determineRequiredCCInterviews() {
        // VersionCC must be the 2nd CC after ManufacturerSpecificCC
        return [core_1.CommandClasses["Manufacturer Specific"]];
    }
    skipEndpointInterview() {
        // SDS13782: In a Multi Channel device, the Version Command Class MUST be supported by the Root Device, while
        // the Version Command Class SHOULD NOT be supported by individual End Points.
        //
        // There may be cases where a given Command Class is not implemented by the Root Device of a Multi
        // Channel device. However, the Root Device MUST respond to Version requests for any Command Class
        // implemented by the Multi Channel device; also in cases where the actual Command Class is only
        // provided by an End Point.
        return true;
    }
    async interview(complete = true) {
        const node = this.getNode();
        const endpoint = this.getEndpoint();
        const api = endpoint.commandClasses.Version.withOptions({
            priority: Constants_1.MessagePriority.NodeQuery,
        });
        this.driver.controllerLog.logNode(node.id, {
            endpoint: this.endpointIndex,
            message: `${this.constructor.name}: doing a ${complete ? "complete" : "partial"} interview...`,
            direction: "none",
        });
        // Version information should not change (except for firmware updates)
        if (complete) {
            // Step 1: Query node versions
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: "querying node versions...",
                direction: "outbound",
            });
            const versionGetResponse = await api.get();
            if (versionGetResponse) {
                // prettier-ignore
                let logMessage = `received response for node versions:
  library type:      ${ZWaveLibraryTypes_1.ZWaveLibraryTypes[versionGetResponse.libraryType]} (${shared_1.num2hex(versionGetResponse.libraryType)})
  protocol version:  ${versionGetResponse.protocolVersion}
  firmware versions: ${versionGetResponse.firmwareVersions.join(", ")}`;
                if (versionGetResponse.hardwareVersion != undefined) {
                    logMessage += `\n  hardware version:  ${versionGetResponse.hardwareVersion}`;
                }
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: logMessage,
                    direction: "inbound",
                });
            }
            // Step 2: Query all CC versions
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: "querying CC versions...",
                direction: "outbound",
            });
            for (const [cc] of endpoint.implementedCommandClasses.entries()) {
                // only query the ones we support a version > 1 for
                const maxImplemented = CommandClass_1.getImplementedVersion(cc);
                if (maxImplemented <= 1) {
                    this.driver.controllerLog.logNode(node.id, `  skipping query for ${core_1.CommandClasses[cc]} (${shared_1.num2hex(cc)}) because max implemented version is ${maxImplemented}`);
                    continue;
                }
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: `  querying the CC version for ${core_1.getCCName(cc)}...`,
                    direction: "outbound",
                });
                // query the CC version
                const supportedVersion = await api.getCCVersion(cc);
                if (supportedVersion != undefined) {
                    // Remember which CC version this endpoint supports
                    let logMessage;
                    if (supportedVersion > 0) {
                        endpoint.addCC(cc, {
                            version: supportedVersion,
                        });
                        logMessage = `  supports CC ${core_1.CommandClasses[cc]} (${shared_1.num2hex(cc)}) in version ${supportedVersion}`;
                    }
                    else {
                        // We were lied to - the NIF said this CC is supported, now the node claims it isn't
                        endpoint.removeCC(cc);
                        logMessage = `  does NOT support CC ${core_1.CommandClasses[cc]} (${shared_1.num2hex(cc)})`;
                    }
                    this.driver.controllerLog.logNode(node.id, logMessage);
                }
                else {
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: `CC version query for ${core_1.getCCName(cc)} timed out - assuming the node supports version 1...`,
                        level: "warn",
                    });
                }
            }
            // Step 3: Query VersionCC capabilities
            if (
            // The CC was created before the versions were determined, so `this.version` contains a wrong value
            this.driver.getSafeCCVersionForNode(core_1.CommandClasses.Version, node.id, this.endpointIndex) >= 3) {
                // Step 3a: Support for SoftwareGet
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: "querying if Z-Wave Software Get is supported...",
                    direction: "outbound",
                });
                const capsResponse = await api.getCapabilities();
                if (capsResponse) {
                    const { supportsZWaveSoftwareGet } = capsResponse;
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: `Z-Wave Software Get is${supportsZWaveSoftwareGet ? "" : " not"} supported`,
                        direction: "inbound",
                    });
                    if (supportsZWaveSoftwareGet) {
                        // Step 3b: Query Z-Wave Software versions
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: "querying Z-Wave software versions...",
                            direction: "outbound",
                        });
                        await api.getZWaveSoftware();
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: "received Z-Wave software versions",
                            direction: "inbound",
                        });
                    }
                }
            }
        }
        // Remember that the interview is complete
        this.interviewComplete = true;
    }
};
VersionCC = __decorate([
    CommandClass_1.commandClass(core_1.CommandClasses.Version),
    CommandClass_1.implementedVersion(3)
], VersionCC);
exports.VersionCC = VersionCC;
let VersionCCReport = class VersionCCReport extends VersionCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 5);
        this._libraryType = this.payload[0];
        this._protocolVersion = `${this.payload[1]}.${this.payload[2]}`;
        this._firmwareVersions = [`${this.payload[3]}.${this.payload[4]}`];
        if (this.version >= 2 && this.payload.length >= 7) {
            this._hardwareVersion = this.payload[5];
            const additionalFirmwares = this.payload[6];
            core_1.validatePayload(this.payload.length >= 7 + 2 * additionalFirmwares);
            for (let i = 0; i < additionalFirmwares; i++) {
                this.firmwareVersions.push(`${this.payload[7 + 2 * i]}.${this.payload[7 + 2 * i + 1]}`);
            }
        }
        this.persistValues();
    }
    get libraryType() {
        return this._libraryType;
    }
    get protocolVersion() {
        return this._protocolVersion;
    }
    get firmwareVersions() {
        return this._firmwareVersions;
    }
    get hardwareVersion() {
        return this._hardwareVersion;
    }
    toLogEntry() {
        const message = {
            "library type": shared_1.getEnumMemberName(ZWaveLibraryTypes_1.ZWaveLibraryTypes, this._libraryType),
            "protocol version": this._protocolVersion,
            "firmware versions": this._firmwareVersions.join(", "),
        };
        if (this._hardwareVersion != undefined) {
            message["hardware version"] = this._hardwareVersion;
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
__decorate([
    CommandClass_1.ccValue(),
    CommandClass_1.ccValueMetadata({
        ...core_1.ValueMetadata.ReadOnly,
        label: "Library type",
    })
], VersionCCReport.prototype, "libraryType", null);
__decorate([
    CommandClass_1.ccValue(),
    CommandClass_1.ccValueMetadata({
        ...core_1.ValueMetadata.ReadOnly,
        label: "Z-Wave protocol version",
    })
], VersionCCReport.prototype, "protocolVersion", null);
__decorate([
    CommandClass_1.ccValue(),
    CommandClass_1.ccValueMetadata({
        ...core_1.ValueMetadata.ReadOnly,
        label: "Z-Wave chip firmware versions",
    })
], VersionCCReport.prototype, "firmwareVersions", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 2 }),
    CommandClass_1.ccValueMetadata({
        ...core_1.ValueMetadata.ReadOnly,
        label: "Z-Wave chip hardware version",
    })
], VersionCCReport.prototype, "hardwareVersion", null);
VersionCCReport = __decorate([
    CommandClass_1.CCCommand(VersionCommand.Report)
], VersionCCReport);
exports.VersionCCReport = VersionCCReport;
let VersionCCGet = class VersionCCGet extends VersionCC {
};
VersionCCGet = __decorate([
    CommandClass_1.CCCommand(VersionCommand.Get),
    CommandClass_1.expectedCCResponse(VersionCCReport)
], VersionCCGet);
exports.VersionCCGet = VersionCCGet;
let VersionCCCommandClassReport = class VersionCCCommandClassReport extends VersionCC {
    // @noCCValues see constructor comment
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 2);
        this._requestedCC = this.payload[0];
        this._ccVersion = this.payload[1];
        // No need to persist this, we're storing it manually
    }
    get ccVersion() {
        return this._ccVersion;
    }
    get requestedCC() {
        return this._requestedCC;
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                CC: core_1.getCCName(this.requestedCC),
                version: this._ccVersion,
            },
        };
    }
};
VersionCCCommandClassReport = __decorate([
    CommandClass_1.CCCommand(VersionCommand.CommandClassReport)
], VersionCCCommandClassReport);
exports.VersionCCCommandClassReport = VersionCCCommandClassReport;
function testResponseForVersionCommandClassGet(sent, received) {
    // We expect a Version CommandClass Report that matches the requested CC
    return sent.requestedCC === received.requestedCC;
}
let VersionCCCommandClassGet = class VersionCCCommandClassGet extends VersionCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.requestedCC = options.requestedCC;
        }
    }
    serialize() {
        this.payload = Buffer.from([this.requestedCC]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { CC: core_1.getCCName(this.requestedCC) },
        };
    }
};
VersionCCCommandClassGet = __decorate([
    CommandClass_1.CCCommand(VersionCommand.CommandClassGet),
    CommandClass_1.expectedCCResponse(VersionCCCommandClassReport, testResponseForVersionCommandClassGet)
], VersionCCCommandClassGet);
exports.VersionCCCommandClassGet = VersionCCCommandClassGet;
let VersionCCCapabilitiesReport = class VersionCCCapabilitiesReport extends VersionCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 1);
        const capabilities = this.payload[0];
        this._supportsZWaveSoftwareGet = !!(capabilities & 0b100);
        this.persistValues();
    }
    get supportsZWaveSoftwareGet() {
        return this._supportsZWaveSoftwareGet;
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "supports Z-Wave Software Get command": this
                    ._supportsZWaveSoftwareGet,
            },
        };
    }
};
__decorate([
    CommandClass_1.ccValue({
        minVersion: 3,
        internal: true,
    })
], VersionCCCapabilitiesReport.prototype, "supportsZWaveSoftwareGet", null);
VersionCCCapabilitiesReport = __decorate([
    CommandClass_1.CCCommand(VersionCommand.CapabilitiesReport)
], VersionCCCapabilitiesReport);
exports.VersionCCCapabilitiesReport = VersionCCCapabilitiesReport;
let VersionCCCapabilitiesGet = class VersionCCCapabilitiesGet extends VersionCC {
};
VersionCCCapabilitiesGet = __decorate([
    CommandClass_1.CCCommand(VersionCommand.CapabilitiesGet),
    CommandClass_1.expectedCCResponse(VersionCCCapabilitiesReport)
], VersionCCCapabilitiesGet);
exports.VersionCCCapabilitiesGet = VersionCCCapabilitiesGet;
let VersionCCZWaveSoftwareReport = class VersionCCZWaveSoftwareReport extends VersionCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 23);
        this._sdkVersion = parseVersion(this.payload);
        this._applicationFrameworkAPIVersion = parseVersion(this.payload.slice(3));
        if (this._applicationFrameworkAPIVersion !== "unused") {
            this._applicationFrameworkBuildNumber = this.payload.readUInt16BE(6);
        }
        else {
            this._applicationFrameworkBuildNumber = 0;
        }
        this._hostInterfaceVersion = parseVersion(this.payload.slice(8));
        if (this._hostInterfaceVersion !== "unused") {
            this._hostInterfaceBuildNumber = this.payload.readUInt16BE(11);
        }
        else {
            this._hostInterfaceBuildNumber = 0;
        }
        this._zWaveProtocolVersion = parseVersion(this.payload.slice(13));
        if (this._zWaveProtocolVersion !== "unused") {
            this._zWaveProtocolBuildNumber = this.payload.readUInt16BE(16);
        }
        else {
            this._zWaveProtocolBuildNumber = 0;
        }
        this._applicationVersion = parseVersion(this.payload.slice(18));
        if (this._applicationVersion !== "unused") {
            this._applicationBuildNumber = this.payload.readUInt16BE(21);
        }
        else {
            this._applicationBuildNumber = 0;
        }
        this.persistValues();
    }
    get sdkVersion() {
        return this._sdkVersion;
    }
    get applicationFrameworkAPIVersion() {
        return this._applicationFrameworkAPIVersion;
    }
    get applicationFrameworkBuildNumber() {
        return this._applicationFrameworkBuildNumber;
    }
    get hostInterfaceVersion() {
        return this._hostInterfaceVersion;
    }
    get hostInterfaceBuildNumber() {
        return this._hostInterfaceBuildNumber;
    }
    get zWaveProtocolVersion() {
        return this._zWaveProtocolVersion;
    }
    get zWaveProtocolBuildNumber() {
        return this._zWaveProtocolBuildNumber;
    }
    get applicationVersion() {
        return this._applicationVersion;
    }
    get applicationBuildNumber() {
        return this._applicationBuildNumber;
    }
    toLogEntry() {
        const message = {
            "SDK version": this._sdkVersion,
        };
        message["appl. framework API version"] = this._applicationFrameworkAPIVersion;
        if (this._applicationFrameworkAPIVersion !== "unused") {
            message["appl. framework build number"] = this._applicationFrameworkBuildNumber;
        }
        message["host interface version"] = this._hostInterfaceVersion;
        if (this._hostInterfaceVersion !== "unused") {
            message["host interface  build number"] = this._hostInterfaceBuildNumber;
        }
        message["Z-Wave protocol version"] = this._zWaveProtocolVersion;
        if (this._zWaveProtocolVersion !== "unused") {
            message["Z-Wave protocol build number"] = this._zWaveProtocolBuildNumber;
        }
        message["application version"] = this._applicationVersion;
        if (this._applicationVersion !== "unused") {
            message["application build number"] = this._applicationBuildNumber;
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "sdkVersion", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "applicationFrameworkAPIVersion", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "applicationFrameworkBuildNumber", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "hostInterfaceVersion", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "hostInterfaceBuildNumber", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "zWaveProtocolVersion", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "zWaveProtocolBuildNumber", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "applicationVersion", null);
__decorate([
    CommandClass_1.ccValue({ minVersion: 3 })
], VersionCCZWaveSoftwareReport.prototype, "applicationBuildNumber", null);
VersionCCZWaveSoftwareReport = __decorate([
    CommandClass_1.CCCommand(VersionCommand.ZWaveSoftwareReport)
], VersionCCZWaveSoftwareReport);
exports.VersionCCZWaveSoftwareReport = VersionCCZWaveSoftwareReport;
let VersionCCZWaveSoftwareGet = class VersionCCZWaveSoftwareGet extends VersionCC {
};
VersionCCZWaveSoftwareGet = __decorate([
    CommandClass_1.CCCommand(VersionCommand.ZWaveSoftwareGet),
    CommandClass_1.expectedCCResponse(VersionCCZWaveSoftwareReport)
], VersionCCZWaveSoftwareGet);
exports.VersionCCZWaveSoftwareGet = VersionCCZWaveSoftwareGet;

//# sourceMappingURL=VersionCC.js.map
