"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createCommandQueueMachine = void 0;
const sorted_list_1 = require("alcalzone-shared/sorted-list");
const xstate_1 = require("xstate");
const actions_1 = require("xstate/lib/actions");
const SendDataMessages_1 = require("../controller/SendDataMessages");
const SerialAPICommandMachine_1 = require("./SerialAPICommandMachine");
const StateMachineShared_1 = require("./StateMachineShared");
const setCurrentTransaction = xstate_1.assign((ctx) => ({
    ...ctx,
    currentTransaction: ctx.queue.shift(),
}));
const deleteCurrentTransaction = xstate_1.assign((ctx) => ({
    ...ctx,
    currentTransaction: undefined,
}));
const clearQueue = xstate_1.assign((ctx) => {
    const queue = ctx.queue;
    queue.clear();
    return { ...ctx, queue };
});
const notifyResult = actions_1.sendParent((ctx, evt) => ({
    ...evt.data,
    type: evt.data.type === "success" ? "command_success" : "command_failure",
    transaction: ctx.currentTransaction,
}));
const notifyError = actions_1.sendParent((ctx, evt) => ({
    type: "command_error",
    error: evt.data,
    transaction: ctx.currentTransaction,
}));
function createCommandQueueMachine(implementations, params) {
    return xstate_1.Machine({
        id: "CommandQueue",
        initial: "idle",
        context: {
            queue: new sorted_list_1.SortedList(),
            // currentTransaction: undefined,
        },
        on: {
            add: {
                actions: [
                    xstate_1.assign({
                        queue: (ctx, evt) => {
                            ctx.queue.add(evt.transaction);
                            return ctx.queue;
                        },
                    }),
                    actions_1.raise("trigger"),
                ],
            },
            // By default, return all messages as unsolicited. The only exception is an active serial API machine
            message: { actions: StateMachineShared_1.respondUnsolicited },
        },
        states: {
            idle: {
                entry: deleteCurrentTransaction,
                on: {
                    trigger: "execute",
                },
                always: {
                    target: "execute",
                    cond: "queueNotEmpty",
                },
            },
            execute: {
                entry: setCurrentTransaction,
                on: {
                    // Now the message event gets auto-forwarded to the serial API machine
                    message: undefined,
                    // Clear the queue and abort ongoing send data commands when a reset event is received
                    reset: [
                        {
                            cond: "currentTransactionIsSendData",
                            actions: clearQueue,
                            target: "abortSendData",
                        },
                        { actions: clearQueue, target: "executeDone" },
                    ],
                },
                invoke: {
                    id: "execute",
                    src: "executeSerialAPICommand",
                    autoForward: true,
                    onDone: [
                        // On success, forward the response to our parent machine
                        {
                            cond: "executeSuccessful",
                            actions: notifyResult,
                            target: "executeDone",
                        },
                        // On failure, abort timed out send attempts
                        {
                            cond: "isSendDataWithCallbackTimeout",
                            target: "abortSendData",
                            actions: notifyResult,
                        },
                        // And just notify the parent about other failures
                        {
                            target: "executeDone",
                            actions: notifyResult,
                        },
                    ],
                    onError: {
                        target: "executeDone",
                        actions: notifyError,
                    },
                },
            },
            abortSendData: {
                invoke: {
                    id: "executeSendDataAbort",
                    src: "executeSendDataAbort",
                    autoForward: true,
                    onDone: "executeDone",
                },
            },
            executeDone: {
                always: {
                    target: "idle",
                    actions: [
                        // Delete the current transaction after we're done
                        deleteCurrentTransaction,
                    ],
                },
            },
        },
    }, {
        services: {
            executeSerialAPICommand: (ctx) => {
                // If there is an error while creating the command machine (e.g. during message serialization)
                // wrap it in a rejected promise, so xstate can handle it
                try {
                    return SerialAPICommandMachine_1.createSerialAPICommandMachine(ctx.currentTransaction.message, implementations, params);
                }
                catch (e) {
                    implementations.log(`Unexpected error during SerialAPI command: ${e}`, "error");
                    return Promise.reject(e);
                }
            },
            executeSendDataAbort: (_) => SerialAPICommandMachine_1.createSerialAPICommandMachine(implementations.createSendDataAbort(), implementations, params),
        },
        guards: {
            executeSuccessful: (_, evt) => { var _a; return ((_a = evt.data) === null || _a === void 0 ? void 0 : _a.type) === "success"; },
            queueNotEmpty: (ctx) => ctx.queue.length > 0,
            currentTransactionIsSendData: (ctx) => {
                var _a;
                const msg = (_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message;
                return (msg instanceof SendDataMessages_1.SendDataRequest ||
                    msg instanceof SendDataMessages_1.SendDataMulticastRequest);
            },
            isSendDataWithCallbackTimeout: (ctx, evt) => {
                var _a, _b, _c;
                const msg = (_a = ctx.currentTransaction) === null || _a === void 0 ? void 0 : _a.message;
                return ((msg instanceof SendDataMessages_1.SendDataRequest ||
                    msg instanceof SendDataMessages_1.SendDataMulticastRequest) &&
                    ((_b = evt.data) === null || _b === void 0 ? void 0 : _b.type) === "failure" &&
                    ((_c = evt.data) === null || _c === void 0 ? void 0 : _c.reason) === "callback timeout");
            },
        },
        delays: {},
    });
}
exports.createCommandQueueMachine = createCommandQueueMachine;

//# sourceMappingURL=CommandQueueMachine.js.map
