/// <reference types="node" />
import { MessageOrCCLogEntry } from "@zwave-js/core";
import type { JSONObject } from "@zwave-js/shared";
import type { Driver } from "../driver/Driver";
import type { ZWaveNode } from "../node/Node";
import { FunctionType, MessagePriority, MessageType } from "./Constants";
declare type Constructable<T extends Message> = new (driver: Driver, options?: MessageOptions) => T;
export interface MessageDeserializationOptions {
    data: Buffer;
}
/**
 * Tests whether the given message constructor options contain a buffer for deserialization
 */
export declare function gotDeserializationOptions(options: Record<any, any> | undefined): options is MessageDeserializationOptions;
export interface MessageBaseOptions {
    callbackId?: number;
}
interface MessageCreationOptions extends MessageBaseOptions {
    type?: MessageType;
    functionType?: FunctionType;
    expectedResponse?: FunctionType | typeof Message | ResponsePredicate;
    expectedCallback?: FunctionType | typeof Message | ResponsePredicate;
    payload?: Buffer;
}
export declare type MessageOptions = MessageCreationOptions | MessageDeserializationOptions;
/**
 * Represents a Z-Wave message for communication with the serial interface
 */
export declare class Message {
    protected driver: Driver;
    constructor(driver: Driver, options?: MessageOptions);
    type: MessageType;
    functionType: FunctionType;
    expectedResponse: FunctionType | typeof Message | ResponsePredicate | undefined;
    expectedCallback: FunctionType | typeof Message | ResponsePredicate | undefined;
    payload: Buffer;
    private _callbackId;
    /**
     * Used to map requests to responses.
     *
     * WARNING: Accessing this property will generate a new callback ID if this message had none.
     * If you want to compare the callback ID, use `hasCallbackId()` beforehand to check if the callback ID is already defined.
     */
    get callbackId(): number;
    set callbackId(v: number);
    /**
     * Tests whether this message's callback ID is defined
     */
    hasCallbackId(): boolean;
    /**
     * Tests whether this message needs a callback ID to match its response
     */
    needsCallbackId(): boolean;
    /** Returns the callback timeout for this message in case the default settings do not apply. */
    getCallbackTimeout(): number | undefined;
    /** Serializes this message into a Buffer */
    serialize(): Buffer;
    /** Returns the number of bytes the first message in the buffer occupies */
    static getMessageLength(data: Buffer): number;
    /**
     * Checks if there's enough data in the buffer to deserialize
     */
    static isComplete(data?: Buffer): boolean;
    /**
     * Retrieves the correct constructor for the next message in the given Buffer.
     * It is assumed that the buffer has been checked beforehand
     */
    static getConstructor(data: Buffer): Constructable<Message>;
    /** Creates an instance of the message that is serialized in the given buffer */
    static from(driver: Driver, data: Buffer): Message;
    /** Returns the slice of data which represents the message payload */
    static extractPayload(data: Buffer): Buffer;
    /** Generates a representation of this Message for the log */
    toLogEntry(): MessageOrCCLogEntry;
    /** Generates the JSON representation of this Message */
    toJSON(): JSONObject;
    private toJSONInternal;
    protected toJSONInherited(props: JSONObject): JSONObject;
    private testMessage;
    /** Tests whether this message expects a response from the controller */
    expectsResponse(): boolean;
    /** Tests whether this message expects a callback from the controller */
    expectsCallback(): boolean;
    /** Checks if a message is an expected response for this message */
    isExpectedResponse(msg: Message): boolean;
    /** Checks if a message is an expected callback for this message */
    isExpectedCallback(msg: Message): boolean;
    /** Finds the ID of the target or source node in a message, if it contains that information */
    getNodeId(): number | undefined;
    /**
     * Returns the node this message is linked to or undefined
     */
    getNodeUnsafe(): ZWaveNode | undefined;
}
export declare type ResponseRole = "unexpected" | "confirmation" | "final" | "fatal_controller" | "fatal_node";
/**
 * A predicate function to test if a received message matches to the sent message
 */
export declare type ResponsePredicate<TSent extends Message = Message> = (sentMessage: TSent, receivedMessage: Message) => boolean;
/**
 * Defines the message and function type associated with a Z-Wave message
 */
export declare function messageTypes(messageType: MessageType, functionType: FunctionType): ClassDecorator;
/**
 * Retrieves the message type defined for a Z-Wave message class
 */
export declare function getMessageType<T extends Message>(messageClass: T): MessageType | undefined;
/**
 * Retrieves the message type defined for a Z-Wave message class
 */
export declare function getMessageTypeStatic<T extends Constructable<Message>>(classConstructor: T): MessageType | undefined;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
export declare function getFunctionType<T extends Message>(messageClass: T): FunctionType | undefined;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
export declare function getFunctionTypeStatic<T extends Constructable<Message>>(classConstructor: T): FunctionType | undefined;
/**
 * Defines the expected response function type or message class for a Z-Wave message
 */
export declare function expectedResponse(typeOrPredicate: FunctionType | typeof Message | ResponsePredicate): ClassDecorator;
/**
 * Retrieves the expected response function type or message class defined for a Z-Wave message class
 */
export declare function getExpectedResponse<T extends Message>(messageClass: T): FunctionType | typeof Message | ResponsePredicate | undefined;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
export declare function getExpectedResponseStatic<T extends Constructable<Message>>(classConstructor: T): FunctionType | typeof Message | ResponsePredicate | undefined;
/**
 * Defines the expected callback function type or message class for a Z-Wave message
 */
export declare function expectedCallback<TSent extends Message>(typeOrPredicate: FunctionType | typeof Message | ResponsePredicate<TSent>): ClassDecorator;
/**
 * Retrieves the expected callback function type or message class defined for a Z-Wave message class
 */
export declare function getExpectedCallback<T extends Message>(messageClass: T): FunctionType | typeof Message | ResponsePredicate | undefined;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
export declare function getExpectedCallbackStatic<T extends Constructable<Message>>(classConstructor: T): FunctionType | typeof Message | ResponsePredicate | undefined;
/**
 * Defines the default priority associated with a Z-Wave message
 */
export declare function priority(prio: MessagePriority): ClassDecorator;
/**
 * Retrieves the default priority defined for a Z-Wave message class
 */
export declare function getDefaultPriority<T extends Message>(messageClass: T): MessagePriority | undefined;
/**
 * Retrieves the default priority defined for a Z-Wave message class
 */
export declare function getDefaultPriorityStatic<T extends Constructable<Message>>(classConstructor: T): MessagePriority | undefined;
export {};
