/// <reference types="node" />
import type { DeviceConfig } from "@zwave-js/config";
import { CommandClasses, Maybe, ValueDB, ValueID, ValueMetadata } from "@zwave-js/core";
import { ZWavePlusNodeType, ZWavePlusRoleType } from "../commandclass/ZWavePlusCC";
import { Baudrate } from "../controller/GetNodeProtocolInfoMessages";
import type { Driver } from "../driver/Driver";
import { DeviceClass } from "./DeviceClass";
import { Endpoint } from "./Endpoint";
import type { TranslatedValueID, ZWaveNodeEventCallbacks, ZWaveNodeEvents } from "./Types";
import { InterviewStage, NodeStatus } from "./Types";
export interface ZWaveNode {
    on<TEvent extends ZWaveNodeEvents>(event: TEvent, callback: ZWaveNodeEventCallbacks[TEvent]): this;
    once<TEvent extends ZWaveNodeEvents>(event: TEvent, callback: ZWaveNodeEventCallbacks[TEvent]): this;
    removeListener<TEvent extends ZWaveNodeEvents>(event: TEvent, callback: ZWaveNodeEventCallbacks[TEvent]): this;
    off<TEvent extends ZWaveNodeEvents>(event: TEvent, callback: ZWaveNodeEventCallbacks[TEvent]): this;
    removeAllListeners(event?: ZWaveNodeEvents): this;
    emit<TEvent extends ZWaveNodeEvents>(event: TEvent, ...args: Parameters<ZWaveNodeEventCallbacks[TEvent]>): boolean;
}
/**
 * A ZWaveNode represents a node in a Z-Wave network. It is also an instance
 * of its root endpoint (index 0)
 */
export declare class ZWaveNode extends Endpoint {
    readonly id: number;
    constructor(id: number, driver: Driver, deviceClass?: DeviceClass, supportedCCs?: CommandClasses[], controlledCCs?: CommandClasses[], valueDB?: ValueDB);
    /**
     * Cleans up all resources used by this node
     */
    destroy(): void;
    /**
     * Enhances a value id so it can be consumed better by applications
     */
    private translateValueID;
    /**
     * Enhances the raw event args of the ValueDB so it can be consumed better by applications
     */
    private translateValueEvent;
    private statusMachine;
    private _status;
    private onStatusChange;
    /**
     * Which status the node is believed to be in
     */
    get status(): NodeStatus;
    private readyMachine;
    private _ready;
    private onReadyChange;
    /**
     * Whether the node is ready to be used
     */
    get ready(): boolean;
    private _deviceClass;
    get deviceClass(): DeviceClass | undefined;
    private _isListening;
    get isListening(): boolean | undefined;
    private _isFrequentListening;
    get isFrequentListening(): boolean | undefined;
    get canSleep(): boolean | undefined;
    private _isRouting;
    get isRouting(): boolean | undefined;
    private _maxBaudRate;
    get maxBaudRate(): Baudrate | undefined;
    private _isSecure;
    get isSecure(): Maybe<boolean> | undefined;
    set isSecure(value: Maybe<boolean> | undefined);
    private _version;
    /** The Z-Wave protocol version this node implements */
    get version(): number | undefined;
    private _isBeaming;
    get isBeaming(): boolean | undefined;
    private _isRoutingSlave;
    get isRoutingSlave(): boolean | undefined;
    private _isController;
    get isController(): boolean | undefined;
    get manufacturerId(): number | undefined;
    get productId(): number | undefined;
    get productType(): number | undefined;
    get firmwareVersion(): string | undefined;
    get zwavePlusVersion(): number | undefined;
    get nodeType(): ZWavePlusNodeType | undefined;
    get roleType(): ZWavePlusRoleType | undefined;
    /**
     * The user-defined name of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the name locally. To permanently change the name of the node, use
     * the `commandClasses` API.
     */
    get name(): string | undefined;
    set name(value: string | undefined);
    /**
     * The user-defined location of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the location locally. To permanently change the location of the node, use
     * the `commandClasses` API.
     */
    get location(): string | undefined;
    set location(value: string | undefined);
    private _deviceConfig;
    /**
     * Contains additional information about this node, loaded from a config file
     */
    get deviceConfig(): DeviceConfig | undefined;
    get label(): string | undefined;
    private _neighbors;
    /** The IDs of all direct neighbors of this node */
    get neighbors(): readonly number[];
    private _valueDB;
    /**
     * Retrieves a stored value for a given value id.
     * This does not request an updated value from the node!
     */
    getValue<T = unknown>(valueId: ValueID): T | undefined;
    /**
     * Retrieves metadata for a given value id.
     * This can be used to enhance the user interface of an application
     */
    getValueMetadata(valueId: ValueID): ValueMetadata;
    /** Returns a list of all value names that are defined on all endpoints of this node */
    getDefinedValueIDs(): TranslatedValueID[];
    private shouldHideValueID;
    /**
     * Removes all Value IDs from an array that belong to a root endpoint and have a corresponding
     * Value ID on a non-root endpoint
     */
    private filterRootApplicationCCValueIDs;
    /**
     * Updates a value for a given property of a given CommandClass on the node.
     * This will communicate with the node!
     */
    setValue(valueId: ValueID, value: unknown): Promise<boolean>;
    /**
     * Requests a value for a given property of a given CommandClass by polling the node.
     * **Warning:** Some value IDs share a command, so make sure not to blindly call this for every property
     */
    pollValue<T extends unknown = unknown>(valueId: ValueID): Promise<T | undefined>;
    get endpointCountIsDynamic(): boolean | undefined;
    get endpointsHaveIdenticalCapabilities(): boolean | undefined;
    get individualEndpointCount(): number | undefined;
    get aggregatedEndpointCount(): number | undefined;
    private getEndpointCCs;
    /** Returns the current endpoint count of this node */
    getEndpointCount(): number;
    /** Whether the Multi Channel CC has been interviewed and all endpoint information is known */
    private get isMultiChannelInterviewComplete();
    /** Cache for this node's endpoint instances */
    private _endpointInstances;
    /**
     * Returns an endpoint of this node with the given index. 0 returns the node itself.
     */
    getEndpoint(index: 0): Endpoint;
    getEndpoint(index: number): Endpoint | undefined;
    /** Returns a list of all endpoints of this node, including the root endpoint (index 0) */
    getAllEndpoints(): Endpoint[];
    /**
     * This tells us which interview stage was last completed
     */
    interviewStage: InterviewStage;
    private _interviewAttempts;
    /** How many attempts to interview this node have already been made */
    get interviewAttempts(): number;
    private _hasEmittedNoNetworkKeyError;
    /** Utility function to check if this node is the controller */
    isControllerNode(): boolean;
    /**
     * Resets all information about this node and forces a fresh interview.
     *
     * WARNING: Take care NOT to call this method when the node is already being interviewed.
     * Otherwise the node information may become inconsistent.
     */
    refreshInfo(): Promise<void>;
    /** Updates this node's interview stage and saves to cache when appropriate */
    private setInterviewStage;
    /** Step #1 of the node interview */
    protected queryProtocolInfo(): Promise<void>;
    /** Node interview: pings the node to see if it responds */
    ping(): Promise<boolean>;
    /**
     * Step #5 of the node interview
     * Request node info
     */
    protected queryNodeInfo(): Promise<void>;
    /**
     * Loads the device configuration for this node from a config file
     */
    protected loadDeviceConfig(): Promise<void>;
    /** Step #? of the node interview */
    protected interviewCCs(): Promise<boolean>;
    /** Returns whether a manual refresh of non-static values is likely necessary for this node */
    requiresManualValueRefresh(): boolean;
    /**
     * Schedules the regular refreshes of some CC values
     */
    private scheduleManualValueRefreshesForListeningNodes;
    private manualRefreshTimers;
    /**
     * Is used to schedule a manual value refresh for nodes that don't send unsolicited commands
     */
    private scheduleManualValueRefresh;
    private cancelManualValueRefresh;
    /**
     * Refreshes all non-static values of a single CC from this node.
     * WARNING: It is not recommended to await this method!
     */
    private refreshCCValues;
    /**
     * Refreshes all non-static values from this node.
     * WARNING: It is not recommended to await this method!
     */
    refreshValues(): Promise<void>;
    /** Overwrites the reported configuration with information from a config file */
    protected overwriteConfig(): Promise<void>;
    /** Queries a node for its neighbor nodes during the node interview */
    protected queryNeighbors(): Promise<void>;
    private hasLoggedNoNetworkKey;
    /**
     * Is called when a nonce report is received that does not belong to any transaction.
     * The received nonce reports are stored as "free" nonces
     */
    private handleSecurityNonceReport;
    private handleHail;
    /** Stores information about a currently held down key */
    private centralSceneKeyHeldDownContext;
    private lastCentralSceneNotificationSequenceNumber;
    /** Handles the receipt of a Central Scene notifification */
    private handleCentralSceneNotification;
    /** The timestamp of the last received wakeup notification */
    private lastWakeUp;
    /** Handles the receipt of a Wake Up notification */
    private handleWakeUpNotification;
    private compatDoWakeupQueries;
    /** Handles the receipt of a BasicCC Set or Report */
    private handleBasicCommand;
    /**
     * Allows automatically resetting notification values to idle if the node does not do it itself
     */
    private notificationIdleTimeouts;
    /** Schedules a notification value to be reset */
    private scheduleNotificationIdleReset;
    /** Removes a scheduled notification reset */
    private clearNotificationIdleReset;
    /**
     * Handles the receipt of a Notification Report
     */
    private handleNotificationReport;
    private handleKnownNotification;
    private handleClockReport;
    private _firmwareUpdateStatus;
    /**
     * Starts an OTA firmware update process for this node.
     *
     * **WARNING: Use at your own risk! We don't take any responsibility if your devices don't work after an update.**
     *
     * @param data The firmware image
     * @param target The firmware target (i.e. chip) to upgrade. 0 updates the Z-Wave chip, >=1 updates others if they exist
     */
    beginFirmwareUpdate(data: Buffer, target?: number): Promise<void>;
    /**
     * Aborts an active firmware update process
     */
    abortFirmwareUpdate(): Promise<void>;
    private sendCorruptedFirmwareUpdateReport;
    private handleFirmwareUpdateGet;
    private timeoutFirmwareUpdate;
    private handleFirmwareUpdateStatusReport;
    private finishFirmwareUpdate;
    /**
     * Whether the node should be kept awake when there are no pending messages.
     */
    keepAwake: boolean;
    private isSendingNoMoreInformation;
}
