/*! @sentry/integrations 6.19.7 (5b3a175) | https://github.com/getsentry/sentry-javascript */
(function (__window) {
var exports = {};

/**
 * NOTE: In order to avoid circular dependencies, if you add a function to this module and it needs to print something,
 * you must either a) use `console.log` rather than the logger, or b) put your function elsewhere.
 */
const fallbackGlobalObject = {};
/**
 * Safely get global scope object
 *
 * @returns Global scope object
 */
function getGlobalObject() {
    return (typeof window !== 'undefined' // eslint-disable-line no-restricted-globals
            ? window // eslint-disable-line no-restricted-globals
            : typeof self !== 'undefined'
                ? self
                : fallbackGlobalObject);
}
/**
 * Returns a global singleton contained in the global `__SENTRY__` object.
 *
 * If the singleton doesn't already exist in `__SENTRY__`, it will be created using the given factory
 * function and added to the `__SENTRY__` object.
 *
 * @param name name of the global singleton on __SENTRY__
 * @param creator creator Factory function to create the singleton if it doesn't already exist on `__SENTRY__`
 * @param obj (Optional) The global object on which to look for `__SENTRY__`, if not `getGlobalObject`'s return value
 * @returns the singleton
 */
function getGlobalSingleton(name, creator, obj) {
    const global = (obj || getGlobalObject());
    const __SENTRY__ = (global.__SENTRY__ = global.__SENTRY__ || {});
    const singleton = __SENTRY__[name] || (__SENTRY__[name] = creator());
    return singleton;
}

/*
 * This file defines flags and constants that can be modified during compile time in order to facilitate tree shaking
 * for users.
 *
 * Debug flags need to be declared in each package individually and must not be imported across package boundaries,
 * because some build tools have trouble tree-shaking imported guards.
 *
 * As a convention, we define debug flags in a `flags.ts` file in the root of a package's `src` folder.
 *
 * Debug flag files will contain "magic strings" like `true` that may get replaced with actual values during
 * our, or the user's build process. Take care when introducing new flags - they must not throw if they are not
 * replaced.
 */
/** Flag that is true for debug builds, false otherwise. */
const IS_DEBUG_BUILD$1 = true;

// TODO: Implement different loggers for different environments
const global = getGlobalObject();
/** Prefix for logging strings */
const PREFIX = 'Sentry Logger ';
const CONSOLE_LEVELS = ['debug', 'info', 'warn', 'error', 'log', 'assert'];
/**
 * Temporarily disable sentry console instrumentations.
 *
 * @param callback The function to run against the original `console` messages
 * @returns The results of the callback
 */
function consoleSandbox(callback) {
    const global = getGlobalObject();
    if (!('console' in global)) {
        return callback();
    }
    const originalConsole = global.console;
    const wrappedLevels = {};
    // Restore all wrapped console methods
    CONSOLE_LEVELS.forEach(level => {
        // TODO(v7): Remove this check as it's only needed for Node 6
        const originalWrappedFunc = originalConsole[level] && originalConsole[level].__sentry_original__;
        if (level in global.console && originalWrappedFunc) {
            wrappedLevels[level] = originalConsole[level];
            originalConsole[level] = originalWrappedFunc;
        }
    });
    try {
        return callback();
    }
    finally {
        // Revert restoration to wrapped state
        Object.keys(wrappedLevels).forEach(level => {
            originalConsole[level] = wrappedLevels[level];
        });
    }
}
function makeLogger() {
    let enabled = false;
    const logger = {
        enable: () => {
            enabled = true;
        },
        disable: () => {
            enabled = false;
        },
    };
    if (IS_DEBUG_BUILD$1) {
        CONSOLE_LEVELS.forEach(name => {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            logger[name] = (...args) => {
                if (enabled) {
                    consoleSandbox(() => {
                        global.console[name](`${PREFIX}[${name}]:`, ...args);
                    });
                }
            };
        });
    }
    else {
        CONSOLE_LEVELS.forEach(name => {
            logger[name] = () => undefined;
        });
    }
    return logger;
}
// Ensure we only have a single logger instance, even if multiple versions of @sentry/utils are being used
let logger;
if (IS_DEBUG_BUILD$1) {
    logger = getGlobalSingleton('logger', makeLogger);
}
else {
    logger = makeLogger();
}

/*
 * This file defines flags and constants that can be modified during compile time in order to facilitate tree shaking
 * for users.
 *
 * Debug flags need to be declared in each package individually and must not be imported across package boundaries,
 * because some build tools have trouble tree-shaking imported guards.
 *
 * As a convention, we define debug flags in a `flags.ts` file in the root of a package's `src` folder.
 *
 * Debug flag files will contain "magic strings" like `true` that may get replaced with actual values during
 * our, or the user's build process. Take care when introducing new flags - they must not throw if they are not
 * replaced.
 */
/** Flag that is true for debug builds, false otherwise. */
const IS_DEBUG_BUILD = true;

// See https://github.com/angular/angular.js/blob/v1.4.7/src/minErr.js
const angularPattern = /^\[((?:[$a-zA-Z0-9]+:)?(?:[$a-zA-Z0-9]+))\] (.*?)\n?(\S+)$/;
/**
 * AngularJS integration
 *
 * Provides an $exceptionHandler for AngularJS
 */
class Angular {
    /**
     * @inheritDoc
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    constructor(options = {}) {
        /**
         * @inheritDoc
         */
        this.name = Angular.id;
        IS_DEBUG_BUILD && logger.log('You are still using the Angular integration, consider moving to @sentry/angular');
        // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/no-unsafe-member-access
        this._angular = options.angular || getGlobalObject().angular;
        if (!this._angular) {
            IS_DEBUG_BUILD && logger.error('AngularIntegration is missing an Angular instance');
            return;
        }
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        this._module = this._angular.module(Angular.moduleName, []);
    }
    /**
     * @inheritDoc
     */
    setupOnce(_, getCurrentHub) {
        if (!this._module) {
            return;
        }
        this._getCurrentHub = getCurrentHub;
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        this._module.config([
            '$provide',
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            ($provide) => {
                // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
                $provide.decorator('$exceptionHandler', ['$delegate', this._$exceptionHandlerDecorator.bind(this)]);
            },
        ]);
    }
    /**
     * Angular's exceptionHandler for Sentry integration
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    _$exceptionHandlerDecorator($delegate) {
        return (exception, cause) => {
            const hub = this._getCurrentHub && this._getCurrentHub();
            if (hub && hub.getIntegration(Angular)) {
                hub.withScope(scope => {
                    if (cause) {
                        scope.setExtra('cause', cause);
                    }
                    scope.addEventProcessor((event) => {
                        const ex = event.exception && event.exception.values && event.exception.values[0];
                        if (ex) {
                            const matches = angularPattern.exec(ex.value || '');
                            if (matches) {
                                // This type now becomes something like: $rootScope:inprog
                                ex.type = matches[1];
                                ex.value = matches[2];
                                event.message = `${ex.type}: ${ex.value}`;
                                // auto set a new tag specifically for the angular error url
                                event.extra = Object.assign(Object.assign({}, event.extra), { angularDocs: matches[3].substr(0, 250) });
                            }
                        }
                        return event;
                    });
                    hub.captureException(exception);
                });
            }
            $delegate(exception, cause);
        };
    }
}
/**
 * @inheritDoc
 */
Angular.id = 'AngularJS';
/**
 * moduleName used in Angular's DI resolution algorithm
 */
Angular.moduleName = 'ngSentry';

exports.Angular = Angular;


  // Add this module's exports to the global `Sentry.Integrations`
  __window.Sentry = __window.Sentry || {};
  __window.Sentry.Integrations = __window.Sentry.Integrations || {};
  for (var key in exports) {
    if (Object.prototype.hasOwnProperty.call(exports, key)) {
      __window.Sentry.Integrations[key] = exports[key];
    }
  }
}(window));
//# sourceMappingURL=angular.es6.js.map
