"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigManager = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const fs_extra_1 = require("fs-extra");
const path_1 = __importDefault(require("path"));
const DeviceClasses_1 = require("./DeviceClasses");
const Devices_1 = require("./Devices");
const Indicators_1 = require("./Indicators");
const Logger_1 = require("./Logger");
const Manufacturers_1 = require("./Manufacturers");
const Meters_1 = require("./Meters");
const Notifications_1 = require("./Notifications");
const Scales_1 = require("./Scales");
const SensorTypes_1 = require("./SensorTypes");
const utils_1 = require("./utils");
class ConfigManager {
    constructor(container) {
        // Make it easier to use this in tests and scripts
        if (!container) {
            container = new core_1.ZWaveLogContainer({ enabled: false });
        }
        this.logger = new Logger_1.ConfigLogger(container);
    }
    async loadManufacturers() {
        try {
            this.manufacturers = await Manufacturers_1.loadManufacturersInternal();
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load manufacturers config: ${e.message}`, "error");
                }
                if (!this.manufacturers)
                    this.manufacturers = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    async saveManufacturers() {
        if (!this.manufacturers) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        await Manufacturers_1.saveManufacturersInternal(this.manufacturers);
    }
    /**
     * Looks up the name of the manufacturer with the given ID in the configuration DB
     * @param manufacturerId The manufacturer id to look up
     */
    lookupManufacturer(manufacturerId) {
        if (!this.manufacturers) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.manufacturers.get(manufacturerId);
    }
    /**
     * Add new manufacturers to configuration DB
     * @param manufacturerId The manufacturer id to look up
     * @param manufacturerName The manufacturer name
     */
    setManufacturer(manufacturerId, manufacturerName) {
        if (!this.manufacturers) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        this.manufacturers.set(manufacturerId, manufacturerName);
    }
    async loadIndicators() {
        try {
            const config = await Indicators_1.loadIndicatorsInternal();
            this.indicators = config.indicators;
            this.indicatorProperties = config.properties;
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load indicators config: ${e.message}`, "error");
                }
                if (!this.indicators)
                    this.indicators = new Map();
                if (!this.indicatorProperties)
                    this.indicatorProperties = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    /**
     * Looks up the label for a given indicator id
     */
    lookupIndicator(indicatorId) {
        if (!this.indicators) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.indicators.get(indicatorId);
    }
    /**
     * Looks up the property definition for a given indicator property id
     */
    lookupProperty(propertyId) {
        if (!this.indicatorProperties) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.indicatorProperties.get(propertyId);
    }
    async loadNamedScales() {
        try {
            this.namedScales = await Scales_1.loadNamedScalesInternal();
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load scales config: ${e.message}`, "error");
                }
                if (!this.namedScales)
                    this.namedScales = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    /**
     * Looks up all scales defined under a given name
     */
    lookupNamedScaleGroup(name) {
        if (!this.namedScales) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.namedScales.get(name);
    }
    /** Looks up a scale definition for a given sensor type */
    lookupNamedScale(name, scale) {
        var _a;
        const group = this.lookupNamedScaleGroup(name);
        return (_a = group === null || group === void 0 ? void 0 : group.get(scale)) !== null && _a !== void 0 ? _a : Scales_1.getDefaultScale(scale);
    }
    async loadSensorTypes() {
        try {
            this.sensorTypes = await SensorTypes_1.loadSensorTypesInternal(this);
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load sensor types config: ${e.message}`, "error");
                }
                if (!this.sensorTypes)
                    this.sensorTypes = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    /**
     * Looks up the configuration for a given sensor type
     */
    lookupSensorType(sensorType) {
        if (!this.sensorTypes) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.sensorTypes.get(sensorType);
    }
    /** Looks up a scale definition for a given sensor type */
    lookupSensorScale(sensorType, scale) {
        var _a;
        const sensor = this.lookupSensorType(sensorType);
        return (_a = sensor === null || sensor === void 0 ? void 0 : sensor.scales.get(scale)) !== null && _a !== void 0 ? _a : Scales_1.getDefaultScale(scale);
    }
    getSensorTypeName(sensorType) {
        const sensor = this.lookupSensorType(sensorType);
        if (sensor)
            return sensor.label;
        return `UNKNOWN (${shared_1.num2hex(sensorType)})`;
    }
    async loadMeters() {
        try {
            this.meters = await Meters_1.loadMetersInternal();
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not meters config: ${e.message}`, "error");
                }
                if (!this.meters)
                    this.meters = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    /**
     * Looks up the notification configuration for a given notification type
     */
    lookupMeter(meterType) {
        if (!this.meters) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.meters.get(meterType);
    }
    getMeterName(meterType) {
        var _a;
        const meter = this.lookupMeter(meterType);
        return (_a = meter === null || meter === void 0 ? void 0 : meter.name) !== null && _a !== void 0 ? _a : `UNKNOWN (${shared_1.num2hex(meterType)})`;
    }
    /** Looks up a scale definition for a given meter type */
    lookupMeterScale(type, scale) {
        var _a;
        const meter = this.lookupMeter(type);
        return (_a = meter === null || meter === void 0 ? void 0 : meter.scales.get(scale)) !== null && _a !== void 0 ? _a : Meters_1.getDefaultMeterScale(scale);
    }
    async loadDeviceClasses() {
        try {
            const config = await DeviceClasses_1.loadDeviceClassesInternal();
            this.basicDeviceClasses = config.basicDeviceClasses;
            this.genericDeviceClasses = config.genericDeviceClasses;
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load scales config: ${e.message}`, "error");
                }
                if (!this.basicDeviceClasses)
                    this.basicDeviceClasses = new Map();
                if (!this.genericDeviceClasses)
                    this.genericDeviceClasses = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    lookupBasicDeviceClass(basic) {
        var _a;
        if (!this.basicDeviceClasses) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return {
            key: basic,
            label: (_a = this.basicDeviceClasses.get(basic)) !== null && _a !== void 0 ? _a : `UNKNOWN (${shared_1.num2hex(basic)})`,
        };
    }
    lookupGenericDeviceClass(generic) {
        var _a;
        if (!this.genericDeviceClasses) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return ((_a = this.genericDeviceClasses.get(generic)) !== null && _a !== void 0 ? _a : DeviceClasses_1.getDefaultGenericDeviceClass(generic));
    }
    lookupSpecificDeviceClass(generic, specific) {
        var _a;
        const genericClass = this.lookupGenericDeviceClass(generic);
        return ((_a = genericClass.specific.get(specific)) !== null && _a !== void 0 ? _a : DeviceClasses_1.getDefaultSpecificDeviceClass(genericClass, specific));
    }
    async loadDeviceIndex() {
        try {
            this.index = await Devices_1.loadDeviceIndexInternal(this.logger);
        }
        catch (e) {
            // If the index file is missing or invalid, don't try to find it again
            if ((!(e instanceof core_1.ZWaveError) && e instanceof Error) ||
                (e instanceof core_1.ZWaveError &&
                    e.code === core_1.ZWaveErrorCodes.Config_Invalid)) {
                // Fall back to no index on production systems
                if (!this.index)
                    this.index = [];
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load or regenerate device config index: ${e.message}`, "error");
                    // and don't throw
                    return;
                }
                // But fail hard in tests
                throw e;
            }
        }
    }
    getIndex() {
        return this.index;
    }
    async loadFulltextDeviceIndex() {
        this.fulltextIndex = await Devices_1.loadFulltextDeviceIndexInternal(this.logger);
    }
    getFulltextIndex() {
        return this.fulltextIndex;
    }
    /**
     * Looks up the definition of a given device in the configuration DB
     * @param manufacturerId The manufacturer id of the device
     * @param productType The product type of the device
     * @param productId The product id of the device
     * @param firmwareVersion If known, configuration for a specific firmware version can be loaded.
     * If this is `undefined` or not given, the first matching file with a defined firmware range will be returned.
     */
    async lookupDevice(manufacturerId, productType, productId, firmwareVersion) {
        // Load/regenerate the index if necessary
        if (!this.index)
            await this.loadDeviceIndex();
        // Look up the device in the index
        const indexEntry = this.index.find(utils_1.getDeviceEntryPredicate(manufacturerId, productType, productId, firmwareVersion));
        if (indexEntry) {
            const filePath = path_1.default.join(utils_1.configDir, "devices", indexEntry.filename);
            if (!(await fs_extra_1.pathExists(filePath)))
                return;
            try {
                return await Devices_1.DeviceConfig.from(filePath, {
                    // Specify how the device was identified in order to evaluate conditions
                    deviceId: {
                        manufacturerId,
                        productType,
                        productId,
                        firmwareVersion,
                    },
                });
            }
            catch (e) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Error loading device config ${filePath}`, "error");
                }
            }
        }
    }
    async loadNotifications() {
        try {
            this.notifications = await Notifications_1.loadNotificationsInternal();
        }
        catch (e) {
            // If the config file is missing or invalid, don't try to find it again
            if (e instanceof core_1.ZWaveError &&
                e.code === core_1.ZWaveErrorCodes.Config_Invalid) {
                if (process.env.NODE_ENV !== "test") {
                    this.logger.print(`Could not load notifications config: ${e.message}`, "error");
                }
                this.notifications = new Map();
            }
            else {
                // This is an unexpected error
                throw e;
            }
        }
    }
    /**
     * Looks up the notification configuration for a given notification type
     */
    lookupNotification(notificationType) {
        if (!this.notifications) {
            throw new core_1.ZWaveError("The config has not been loaded yet!", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this.notifications.get(notificationType);
    }
    /**
     * Looks up the notification configuration for a given notification type.
     * If the config has not been loaded yet, this returns undefined.
     */
    lookupNotificationUnsafe(notificationType) {
        var _a;
        return (_a = this.notifications) === null || _a === void 0 ? void 0 : _a.get(notificationType);
    }
    getNotificationName(notificationType) {
        var _a, _b;
        return ((_b = (_a = this.lookupNotificationUnsafe(notificationType)) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : `Unknown (${shared_1.num2hex(notificationType)})`);
    }
}
exports.ConfigManager = ConfigManager;
//# sourceMappingURL=ConfigManager.js.map