"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SensorType = exports.loadSensorTypesInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const Scales_1 = require("./Scales");
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "sensorTypes.json");
/** @internal */
async function loadSensorTypesInternal(manager) {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The sensor types config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("sensor types", `the dictionary is not an object`);
        }
        const sensorTypes = new Map();
        for (const [key, sensorDefinition] of objects_1.entries(definition)) {
            if (!utils_1.hexKeyRegexNDigits.test(key)) {
                utils_1.throwInvalidConfig("sensor types", `found non-hex key "${key}" at the root`);
            }
            const keyNum = parseInt(key.slice(2), 16);
            sensorTypes.set(keyNum, new SensorType(manager, keyNum, sensorDefinition));
        }
        return sensorTypes;
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("sensor types");
        }
    }
}
exports.loadSensorTypesInternal = loadSensorTypesInternal;
const namedScalesMarker = "$SCALES:";
class SensorType {
    constructor(manager, key, definition) {
        this.key = key;
        if (typeof definition.label !== "string")
            utils_1.throwInvalidConfig("sensor types", `label for ${shared_1.num2hex(key)} is not a string`);
        this.label = definition.label;
        if (typeof definition.scales === "string" &&
            definition.scales.startsWith(namedScalesMarker)) {
            // This is referencing a named scale
            const scaleName = definition.scales.substr(namedScalesMarker.length);
            const scales = manager.lookupNamedScaleGroup(scaleName);
            if (!scales) {
                throw new core_1.ZWaveError(`Sensor type ${shared_1.num2hex(key)} is referencing non-existing named scale "${scaleName}"!`, core_1.ZWaveErrorCodes.Config_Invalid);
            }
            this.scales = scales;
        }
        else {
            // This is an inline scale definition
            const scales = new Map();
            if (!typeguards_1.isObject(definition.scales))
                utils_1.throwInvalidConfig("sensor types", `scale definition for ${shared_1.num2hex(key)} is not an object`);
            for (const [scaleKey, scaleDefinition] of objects_1.entries(definition.scales)) {
                if (!utils_1.hexKeyRegexNDigits.test(scaleKey))
                    utils_1.throwInvalidConfig("sensor types", `found non-hex key "${scaleKey}" in sensor type ${shared_1.num2hex(key)}`);
                const scaleKeyNum = parseInt(scaleKey.slice(2), 16);
                scales.set(scaleKeyNum, new Scales_1.Scale(scaleKeyNum, scaleDefinition));
            }
            this.scales = scales;
        }
    }
}
exports.SensorType = SensorType;
//# sourceMappingURL=SensorTypes.js.map