/// <reference types="node" />
import { CacheMetadata, CacheValue, CommandClasses, MessageOrCCLogEntry, ValueDB, ValueID, ValueMetadata } from "@zwave-js/core";
import { JSONObject } from "@zwave-js/shared";
import type { Driver } from "../driver/Driver";
import type { Endpoint } from "../node/Endpoint";
import type { ZWaveNode } from "../node/Node";
import type { VirtualEndpoint } from "../node/VirtualEndpoint";
import { CCAPI } from "./API";
import { EncapsulatingCommandClass } from "./EncapsulatingCommandClass";
export declare type MulticastDestination = [number, number, ...number[]];
export declare type CommandClassDeserializationOptions = {
    data: Buffer;
} & ({
    fromEncapsulation?: false;
    nodeId: number;
} | {
    fromEncapsulation: true;
    encapCC: CommandClass;
});
export declare function gotDeserializationOptions(options: CommandClassOptions): options is CommandClassDeserializationOptions;
export interface CCCommandOptions {
    nodeId: number | MulticastDestination;
    endpoint?: number;
    supervised?: boolean;
}
interface CommandClassCreationOptions extends CCCommandOptions {
    ccId?: number;
    ccCommand?: number;
    payload?: Buffer;
}
export declare type CommandClassOptions = CommandClassCreationOptions | CommandClassDeserializationOptions;
export declare class CommandClass {
    constructor(driver: Driver, options: CommandClassOptions);
    protected driver: Driver;
    /** This CC's identifier */
    ccId: CommandClasses;
    ccCommand?: number;
    /** The ID of the target node(s) */
    nodeId: number | MulticastDestination;
    payload: Buffer;
    /** The version of the command class used */
    version: number;
    /** Which endpoint of the node this CC belongs to. 0 for the root device. */
    endpointIndex: number;
    /** Contains a reference to the encapsulating CC if this CC is encapsulated */
    encapsulatingCC?: EncapsulatingCommandClass;
    /** Returns true if this CC is an extended CC (0xF100..0xFFFF) */
    isExtended(): boolean;
    /** Whether the interview for this CC was previously completed */
    get interviewComplete(): boolean;
    set interviewComplete(value: boolean);
    /** Can be used by endpoints to test if the root device was already interviewed */
    get rootDeviceInterviewComplete(): boolean;
    /**
     * Deserializes a CC from a buffer that contains a serialized CC
     */
    private deserialize;
    /**
     * Serializes this CommandClass to be embedded in a message payload or another CC
     */
    serialize(): Buffer;
    /** Extracts the CC id from a buffer that contains a serialized CC */
    static getCommandClass(data: Buffer): CommandClasses;
    /** Extracts the CC command from a buffer that contains a serialized CC  */
    static getCCCommand(data: Buffer): number | undefined;
    /**
     * Retrieves the correct constructor for the CommandClass in the given Buffer.
     * It is assumed that the buffer only contains the serialized CC. This throws if the CC is not implemented.
     */
    static getConstructor(ccData: Buffer): Constructable<CommandClass>;
    /**
     * Creates an instance of the CC that is serialized in the given buffer
     */
    static from(driver: Driver, options: CommandClassDeserializationOptions): CommandClass;
    /** Generates a representation of this CC for the log */
    toLogEntry(): MessageOrCCLogEntry;
    /** Generates the JSON representation of this CC */
    toJSON(): JSONObject;
    private toJSONInternal;
    protected toJSONInherited(props: JSONObject): JSONObject;
    /**
     * Performs the interview procedure for this CC according to SDS14223
     * @param complete Whether a complete interview should be performed or only the necessary steps on startup
     */
    interview(complete?: boolean): Promise<void>;
    /** Determines which CC interviews must be performed before this CC can be interviewed */
    determineRequiredCCInterviews(): readonly CommandClasses[];
    /**
     * Whether the endpoint interview may be skipped by a CC. Can be overwritten by a subclass.
     */
    skipEndpointInterview(): boolean;
    /**
     * Maps a BasicCC value to a more specific CC implementation. Returns true if the value was mapped, false otherwise.
     * @param value The value of the received BasicCC
     */
    setMappedBasicValue(value: number): boolean;
    isSinglecast(): this is SinglecastCC<this>;
    isMulticast(): this is MulticastCC<this>;
    /**
     * Returns the node this CC is linked to. Throws if the controller is not yet ready.
     */
    getNode(): ZWaveNode | undefined;
    getEndpoint(): Endpoint | undefined;
    /** Returns the value DB for this CC's node */
    protected getValueDB(): ValueDB;
    /** Which variables should be persisted when requested */
    private _registeredCCValues;
    /**
     * Creates a value that will be stored in the valueDB alongside with the ones marked with `@ccValue()`
     * @param property The property the value belongs to
     * @param internal Whether the value should be exposed to library users
     */
    registerValue(property: string | number, internal?: boolean): void;
    /** Returns a list of all value names that are defined for this CommandClass */
    getDefinedValueIDs(): ValueID[];
    /** Determines if the given value is an internal value */
    isInternalValue(property: keyof this): boolean;
    /** Determines if the given value should always be persisted */
    shouldValueAlwaysBeCreated(property: keyof this): boolean;
    /** Determines if the given value should be persisted or represents an event */
    isStatefulValue(property: keyof this): boolean;
    /** Persists all values on the given node into the value. Returns true if the process succeeded, false otherwise */
    persistValues(valueNames?: (keyof this)[]): boolean;
    /** Serializes all values to be stored in the cache */
    serializeValuesForCache(): CacheValue[];
    /** Serializes metadata to be stored in the cache */
    serializeMetadataForCache(): CacheMetadata[];
    /** Deserializes values from the cache */
    deserializeValuesFromCache(values: CacheValue[]): void;
    /** Deserializes value metadata from the cache */
    deserializeMetadataFromCache(allMetadata: CacheMetadata[]): void;
    /**
     * When a CC supports to be split into multiple partial CCs, this can be used to identify the
     * session the partial CCs belong to.
     * If a CC expects `mergePartialCCs` to be always called, you should return an empty object here.
     */
    getPartialCCSessionId(): Record<string, any> | undefined;
    /** When a CC supports to be split into multiple partial CCs, this indicates that the last report hasn't been received yet */
    expectMoreMessages(): boolean;
    /** Include previously received partial responses into a final CC */
    mergePartialCCs(partials: CommandClass[]): void;
    /** Tests whether this CC expects at least one command in return */
    expectsCCResponse(): boolean;
    isExpectedCCResponse(received: CommandClass): boolean;
    /**
     * Translates a property identifier into a speaking name for use in an external API
     * @param property The property identifier that should be translated
     * @param _propertyKey The (optional) property key the translated name may depend on
     */
    translateProperty(property: string | number, _propertyKey?: string | number): string;
    /**
     * Translates a property key into a speaking name for use in an external API
     * @param property The property the key in question belongs to
     * @param propertyKey The property key for which the speaking name should be retrieved
     */
    translatePropertyKey(property: string | number, propertyKey: string | number): string | undefined;
    /** Whether this CC needs to exchange one or more messages before it can be sent */
    requiresPreTransmitHandshake(): boolean;
    /**
     * Perform a handshake before the actual message will be transmitted.
     */
    preTransmitHandshake(): Promise<void>;
    /** Returns the number of bytes that are added to the payload by this CC */
    protected computeEncapsulationOverhead(): number;
    /** Computes the maximum net payload size that can be transmitted inside this CC */
    getMaxPayloadLength(baseLength: number): number;
    /** Checks whether this CC is encapsulated with one that has the given CC id and (optionally) CC Command */
    isEncapsulatedWith(ccId: CommandClasses, ccCommand?: number): boolean;
}
export declare type SinglecastCC<T extends CommandClass = CommandClass> = T & {
    nodeId: number;
};
export declare type MulticastCC<T extends CommandClass = CommandClass> = T & {
    nodeId: MulticastDestination;
};
export declare type Constructable<T extends CommandClass> = typeof CommandClass & {
    new (driver: Driver, options: any): T;
};
declare type APIConstructor = new (driver: Driver, endpoint: Endpoint | VirtualEndpoint) => CCAPI;
declare type TypedClassDecorator<TTarget extends Object> = <T extends TTarget, TConstructor extends new (...args: any[]) => T>(apiClass: TConstructor) => TConstructor | void;
/**
 * May be used to define different expected CC responses depending on the sent CC
 */
export declare type DynamicCCResponse<TSent extends CommandClass, TReceived extends CommandClass = CommandClass> = (sentCC: TSent) => Constructable<TReceived> | Constructable<TReceived>[] | undefined;
export declare type CCResponseRole = boolean | "checkEncapsulated";
/**
 * A predicate function to test if a received CC matches the sent CC
 */
export declare type CCResponsePredicate<TSent extends CommandClass, TReceived extends CommandClass = CommandClass> = (sentCommand: TSent, receivedCommand: TReceived) => CCResponseRole;
/**
 * Defines the command class associated with a Z-Wave message
 */
export declare function commandClass(cc: CommandClasses): TypedClassDecorator<CommandClass>;
/**
 * Retrieves the command class defined for a Z-Wave message class
 */
export declare function getCommandClass<T extends CommandClass | CCAPI>(cc: T): CommandClasses;
/**
 * Retrieves the function type defined for a Z-Wave message class
 */
export declare function getCommandClassStatic<T extends Constructable<CommandClass>>(classConstructor: T): CommandClasses;
/**
 * Looks up the command class constructor for a given command class type and function type
 */
export declare function getCCConstructor(cc: CommandClasses): Constructable<CommandClass> | undefined;
/**
 * Defines the implemented version of a Z-Wave command class
 */
export declare function implementedVersion(version: number): TypedClassDecorator<CommandClass>;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
export declare function getImplementedVersion<T extends CommandClass>(cc: T | CommandClasses): number;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
export declare function getImplementedVersionStatic<T extends Constructable<CommandClass>>(classConstructor: T): number;
/**
 * Defines the CC command a subclass of a CC implements
 */
export declare function CCCommand(command: number): TypedClassDecorator<CommandClass>;
/**
 * Defines the expected response associated with a Z-Wave message
 */
export declare function expectedCCResponse<TSent extends CommandClass, TReceived extends CommandClass>(cc: Constructable<TReceived> | DynamicCCResponse<TSent, TReceived>, predicate?: CCResponsePredicate<TSent, TReceived>): TypedClassDecorator<CommandClass>;
/**
 * Retrieves the expected response (static or dynamic) defined for a Z-Wave message class
 */
export declare function getExpectedCCResponse<T extends CommandClass>(ccClass: T): typeof CommandClass | DynamicCCResponse<T> | undefined;
/**
 * Retrieves the CC response predicate defined for a Z-Wave message class
 */
export declare function getCCResponsePredicate<T extends CommandClass>(ccClass: T): CCResponsePredicate<T> | undefined;
export interface CCValueOptions {
    /**
     * Whether the decorated CC value is internal. Internal values are not
     * exposed to the user.
     */
    internal?: boolean;
    /**
     * The minimum CC version required for this value to exist.
     */
    minVersion?: number;
    /**
     * Whether this value should always be created/persisted, even if it is undefined. Default: false
     */
    forceCreation?: boolean;
    /**
     * Whether this value represents a state (`true`) or a notification/event (`false`). Default: `true`
     */
    stateful?: boolean;
}
/**
 * Marks the decorated property as a value of the Command Class. This allows saving it on the node with persistValues()
 * @param internal Whether the value should be exposed to library users
 */
export declare function ccValue(options?: CCValueOptions): PropertyDecorator;
/**
 * Marks the decorated property as the key of a Command Class's key value pair,
 * which can later be saved with persistValues()
 * @param internal Whether the key value pair should be exposed to library users
 */
export declare function ccKeyValuePair(options?: CCValueOptions): PropertyDecorator;
/**
 * Defines additional metadata for the given CC value
 */
export declare function ccValueMetadata(meta: ValueMetadata): PropertyDecorator;
/**
 * Retrieves defined metadata for the given CC value. If none is found, the default settings are returned.
 */
export declare function getCCValueMetadata(cc: CommandClasses, property: string | number): ValueMetadata;
/**
 * Defines the simplified API associated with a Z-Wave command class
 */
export declare function API(cc: CommandClasses): TypedClassDecorator<CCAPI>;
/**
 * Retrieves the implemented version defined for a Z-Wave command class
 */
export declare function getAPI(cc: CommandClasses): APIConstructor | undefined;
export {};
