"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationCCDefaultReset = exports.ConfigurationCCPropertiesGet = exports.ConfigurationCCPropertiesReport = exports.ConfigurationCCInfoGet = exports.ConfigurationCCInfoReport = exports.ConfigurationCCNameGet = exports.ConfigurationCCNameReport = exports.ConfigurationCCBulkGet = exports.ConfigurationCCBulkReport = exports.ConfigurationCCBulkSet = exports.ConfigurationCCSet = exports.ConfigurationCCGet = exports.ConfigurationCCReport = exports.ConfigurationCC = exports.ConfigurationCCAPI = exports.getParamInformationValueID = exports.ConfigurationCCError = exports.ValueFormat = exports.ConfigurationCommand = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const arrays_1 = require("alcalzone-shared/arrays");
const objects_1 = require("alcalzone-shared/objects");
const strings_1 = require("alcalzone-shared/strings");
const Constants_1 = require("../message/Constants");
const API_1 = require("./API");
const CommandClass_1 = require("./CommandClass");
var ConfigurationCommand;
(function (ConfigurationCommand) {
    ConfigurationCommand[ConfigurationCommand["Set"] = 4] = "Set";
    ConfigurationCommand[ConfigurationCommand["Get"] = 5] = "Get";
    ConfigurationCommand[ConfigurationCommand["Report"] = 6] = "Report";
    ConfigurationCommand[ConfigurationCommand["BulkSet"] = 7] = "BulkSet";
    ConfigurationCommand[ConfigurationCommand["BulkGet"] = 8] = "BulkGet";
    ConfigurationCommand[ConfigurationCommand["BulkReport"] = 9] = "BulkReport";
    ConfigurationCommand[ConfigurationCommand["NameGet"] = 10] = "NameGet";
    ConfigurationCommand[ConfigurationCommand["NameReport"] = 11] = "NameReport";
    ConfigurationCommand[ConfigurationCommand["InfoGet"] = 12] = "InfoGet";
    ConfigurationCommand[ConfigurationCommand["InfoReport"] = 13] = "InfoReport";
    ConfigurationCommand[ConfigurationCommand["PropertiesGet"] = 14] = "PropertiesGet";
    ConfigurationCommand[ConfigurationCommand["PropertiesReport"] = 15] = "PropertiesReport";
    ConfigurationCommand[ConfigurationCommand["DefaultReset"] = 1] = "DefaultReset";
})(ConfigurationCommand = exports.ConfigurationCommand || (exports.ConfigurationCommand = {}));
var ValueFormat;
(function (ValueFormat) {
    ValueFormat[ValueFormat["SignedInteger"] = 0] = "SignedInteger";
    ValueFormat[ValueFormat["UnsignedInteger"] = 1] = "UnsignedInteger";
    ValueFormat[ValueFormat["Enumerated"] = 2] = "Enumerated";
    ValueFormat[ValueFormat["BitField"] = 3] = "BitField";
})(ValueFormat = exports.ValueFormat || (exports.ValueFormat = {}));
function configValueToString(value) {
    if (typeof value === "number")
        return value.toString();
    else
        return [...value].join(", ");
}
class ConfigurationCCError extends core_1.ZWaveError {
    constructor(message, code, argument) {
        super(message, code);
        this.message = message;
        this.code = code;
        this.argument = argument;
        // We need to set the prototype explicitly
        Object.setPrototypeOf(this, ConfigurationCCError.prototype);
    }
}
exports.ConfigurationCCError = ConfigurationCCError;
function getParamInformationValueID(parameter, bitMask) {
    return {
        commandClass: core_1.CommandClasses.Configuration,
        property: parameter,
        propertyKey: bitMask,
    };
}
exports.getParamInformationValueID = getParamInformationValueID;
const isParamInfoFromConfigValueId = {
    commandClass: core_1.CommandClasses.Configuration,
    property: "isParamInformationFromConfig",
};
let ConfigurationCCAPI = class ConfigurationCCAPI extends API_1.PhysicalCCAPI {
    constructor() {
        super(...arguments);
        this[_a] = async ({ property, propertyKey }, value) => {
            // Config parameters are addressed with numeric properties/keys
            if (typeof property !== "number") {
                API_1.throwUnsupportedProperty(this.ccId, property);
            }
            if (propertyKey != undefined && typeof propertyKey !== "number") {
                API_1.throwUnsupportedPropertyKey(this.ccId, property, propertyKey);
            }
            if (typeof value !== "number") {
                API_1.throwWrongValueType(this.ccId, property, "number", typeof value);
            }
            const ccInstance = this.endpoint.createCCInstance(ConfigurationCC);
            let valueSize = ccInstance.getParamInformation(property).valueSize;
            const valueFormat = ccInstance.getParamInformation(property).format ||
                ValueFormat.SignedInteger;
            let targetValue;
            if (propertyKey) {
                // This is a partial value, we need to update some bits only
                // Find out the correct value size
                if (!valueSize) {
                    valueSize = ccInstance.getParamInformation(property, propertyKey).valueSize;
                }
                // Add the target value to the remaining partial values
                targetValue = ccInstance.composePartialParamValue(property, propertyKey, value);
            }
            else {
                targetValue = value;
            }
            if (!valueSize) {
                // If there's no value size configured, figure out a matching value size
                valueSize = core_1.getMinIntegerSize(targetValue, valueFormat === ValueFormat.SignedInteger);
                // Throw if the value is too large or too small
                if (!valueSize) {
                    throw new core_1.ZWaveError(`The value ${targetValue} is not valid for configuration parameters!`, core_1.ZWaveErrorCodes.Argument_Invalid);
                }
            }
            // Make sure that the given value fits into the value size
            if (!isSafeValue(targetValue, valueSize, valueFormat)) {
                // If there is a value size configured, check that the given value is compatible
                throwInvalidValueError(value, property, valueSize, valueFormat);
            }
            await this.set(property, targetValue, valueSize);
            // Verify the current value after a delay
            this.schedulePoll({ property, propertyKey }, 1000);
        };
        this[_b] = async ({ property, propertyKey, }) => {
            // Config parameters are addressed with numeric properties/keys
            if (typeof property !== "number") {
                API_1.throwUnsupportedProperty(this.ccId, property);
            }
            if (propertyKey != undefined && typeof propertyKey !== "number") {
                API_1.throwUnsupportedPropertyKey(this.ccId, property, propertyKey);
            }
            return this.get(property, { valueBitMask: propertyKey });
        };
    }
    supportsCommand(cmd) {
        switch (cmd) {
            case ConfigurationCommand.Get:
            case ConfigurationCommand.Set:
                return true; // This is mandatory
            case ConfigurationCommand.BulkGet:
            case ConfigurationCommand.BulkSet:
                return this.version >= 2;
            case ConfigurationCommand.NameGet:
            case ConfigurationCommand.InfoGet:
            case ConfigurationCommand.PropertiesGet:
                return this.version >= 3;
            case ConfigurationCommand.DefaultReset:
                return this.version >= 4;
        }
        return super.supportsCommand(cmd);
    }
    /**
     * Requests the current value of a given config parameter from the device.
     * This may timeout and return `undefined` if the node does not respond.
     * If the node replied with a different parameter number, a `ConfigurationCCError`
     * is thrown with the `argument` property set to the reported parameter number.
     */
    async get(parameter, options) {
        this.assertSupportsCommand(ConfigurationCommand, ConfigurationCommand.Get);
        const { valueBitMask, allowUnexpectedResponse } = options !== null && options !== void 0 ? options : {};
        const cc = new ConfigurationCCGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
            allowUnexpectedResponse,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (!response)
            return;
        // Nodes may respond with a different parameter, e.g. if we
        // requested a non-existing one
        if (response.parameter === parameter) {
            if (!valueBitMask)
                return response.value;
            // If a partial parameter was requested, extract that value
            return ((response.value & valueBitMask) >>>
                core_1.getMinimumShiftForBitMask(valueBitMask));
        }
        this.driver.controllerLog.logNode(this.endpoint.nodeId, {
            message: `Received unexpected ConfigurationReport (param = ${response.parameter}, value = ${response.value.toString()})`,
            direction: "inbound",
            level: "error",
        });
        throw new ConfigurationCCError(`The first existing parameter on this node is ${response.parameter}`, core_1.ZWaveErrorCodes.ConfigurationCC_FirstParameterNumber, response.parameter);
    }
    /**
     * Sets a new value for a given config parameter of the device.
     */
    async set(parameter, value, valueSize) {
        this.assertSupportsCommand(ConfigurationCommand, ConfigurationCommand.Set);
        const cc = new ConfigurationCCSet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
            value,
            valueSize,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    /**
     * Resets a configuration parameter to its default value.
     *
     * WARNING: This will throw on legacy devices (ConfigurationCC v3 and below)
     */
    async reset(parameter) {
        this.assertSupportsCommand(ConfigurationCommand, ConfigurationCommand.Set);
        const cc = new ConfigurationCCSet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
            resetToDefault: true,
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    /** Resets all configuration parameters to their default value */
    async resetAll() {
        this.assertSupportsCommand(ConfigurationCommand, ConfigurationCommand.DefaultReset);
        const cc = new ConfigurationCCDefaultReset(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
        });
        await this.driver.sendCommand(cc, this.commandOptions);
    }
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    async getProperties(parameter) {
        const cc = new ConfigurationCCPropertiesGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        if (response) {
            return shared_1.pick(response, [
                "valueSize",
                "valueFormat",
                "minValue",
                "maxValue",
                "defaultValue",
                "nextParameter",
                "altersCapabilities",
                "isReadonly",
                "isAdvanced",
                "noBulkSupport",
            ]);
        }
    }
    /** Requests the name of a configuration parameter from the node */
    async getName(parameter) {
        const cc = new ConfigurationCCNameGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        return response === null || response === void 0 ? void 0 : response.name;
    }
    /** Requests usage info for a configuration parameter from the node */
    async getInfo(parameter) {
        const cc = new ConfigurationCCInfoGet(this.driver, {
            nodeId: this.endpoint.nodeId,
            // Don't set an endpoint here, Configuration is device specific, not endpoint specific
            parameter,
        });
        const response = await this.driver.sendCommand(cc, this.commandOptions);
        return response === null || response === void 0 ? void 0 : response.info;
    }
    /**
     * This scans the node for the existing parameters. Found parameters will be reported
     * through the `value added` and `value updated` events.
     *
     * WARNING: This method throws for newer devices.
     *
     * WARNING: On nodes implementing V1 and V2, this process may take
     * **up to an hour**, depending on the configured timeout.
     *
     * WARNING: On nodes implementing V2, all parameters after 255 will be ignored.
     */
    async scanParametersLegacy() {
        if (this.version >= 3) {
            throw new core_1.ZWaveError("Use ConfigurationCC.interview instead of scanning parameters for versions 3 and above.", core_1.ZWaveErrorCodes.ConfigurationCC_NoLegacyScanOnNewDevices);
        }
        // TODO: Reduce the priority of the messages
        this.driver.controllerLog.logNode(this.endpoint.nodeId, `Scanning available parameters...`);
        const ccInstance = this.endpoint.createCCInstance(ConfigurationCC);
        for (let param = 1; param <= 255; param++) {
            // Check if the parameter is readable
            let originalValue;
            this.driver.controllerLog.logNode(this.endpoint.nodeId, {
                message: `  trying param ${param}...`,
                direction: "outbound",
            });
            try {
                originalValue = await this.get(param, {
                    // When requesting a non-existing parameter, a node SHOULD respond with the
                    // first available parameter. We use this for the first param only,
                    // because delayed responses might otherwise confuse the interview process
                    allowUnexpectedResponse: param === 1,
                });
                if (originalValue != undefined) {
                    const logMessage = `  Param ${param}:
    readable  = true
    valueSize = ${ccInstance.getParamInformation(param).valueSize}
    value     = ${originalValue.toString()}`;
                    this.driver.controllerLog.logNode(this.endpoint.nodeId, {
                        message: logMessage,
                        direction: "inbound",
                    });
                }
            }
            catch (e) {
                if (e instanceof ConfigurationCCError &&
                    e.code ===
                        core_1.ZWaveErrorCodes.ConfigurationCC_FirstParameterNumber) {
                    // Continue iterating with the next param
                    if (e.argument - 1 > param)
                        param = e.argument - 1;
                    continue;
                }
                throw e;
            }
        }
    }
};
_a = API_1.SET_VALUE, _b = API_1.POLL_VALUE;
ConfigurationCCAPI = __decorate([
    CommandClass_1.API(core_1.CommandClasses.Configuration)
], ConfigurationCCAPI);
exports.ConfigurationCCAPI = ConfigurationCCAPI;
let ConfigurationCC = class ConfigurationCC extends CommandClass_1.CommandClass {
    constructor(driver, options) {
        super(driver, options);
        this.registerValue("isParamInformationFromConfig", true);
    }
    async interview(complete = true) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        const node = this.getNode();
        const config = (_c = node.deviceConfig) === null || _c === void 0 ? void 0 : _c.paramInformation;
        if (config) {
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: `${this.constructor.name}: Loading configuration parameters from device config`,
                direction: "none",
            });
            this.deserializeParamInformationFromConfig(config);
        }
        const endpoint = this.getEndpoint();
        const api = endpoint.commandClasses.Configuration.withOptions({
            priority: Constants_1.MessagePriority.NodeQuery,
        });
        if (this.version < 3) {
            const paramInfo = (_d = node.deviceConfig) === null || _d === void 0 ? void 0 : _d.paramInformation;
            if (paramInfo === null || paramInfo === void 0 ? void 0 : paramInfo.size) {
                // Query all values
                // Because partial params share the same parameter number,
                // we need to remember which ones we have already queried.
                const alreadyQueried = new Set();
                for (const param of paramInfo.keys()) {
                    // No need to query writeonly params
                    if (paramInfo.get(param).writeOnly)
                        continue;
                    // Don't double-query params
                    if (alreadyQueried.has(param.parameter))
                        continue;
                    alreadyQueried.add(param.parameter);
                    // Query the current value
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: `querying parameter #${param.parameter} value...`,
                        direction: "outbound",
                    });
                    // ... at least try to
                    const paramValue = await api.get(param.parameter);
                    if (typeof paramValue === "number") {
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: `parameter #${param.parameter} has value: ${paramValue}`,
                            direction: "inbound",
                        });
                    }
                    else if (!paramValue) {
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: `received no value for parameter #${param.parameter}`,
                            direction: "inbound",
                            level: "warn",
                        });
                    }
                }
            }
            else {
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: `${this.constructor.name}: skipping interview because CC version is < 3 and there is no config file`,
                    direction: "none",
                });
            }
        }
        else {
            // Version >= 3
            this.driver.controllerLog.logNode(node.id, {
                endpoint: this.endpointIndex,
                message: `${this.constructor.name}: doing a ${complete ? "complete" : "partial"} interview...`,
                direction: "none",
            });
            if (complete) {
                // Only scan all parameters during complete interviews
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: "finding first configuration parameter...",
                    direction: "outbound",
                });
                const param0props = await api.getProperties(0);
                let param;
                if (param0props) {
                    param = param0props.nextParameter;
                    if (param === 0) {
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: `didn't report any config params, trying #1 just to be sure...`,
                            direction: "inbound",
                        });
                        param = 1;
                    }
                }
                else {
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: "Finding first configuration parameter timed out, skipping interview...",
                        level: "warn",
                    });
                    return;
                }
                while (param > 0) {
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: `querying parameter #${param} information...`,
                        direction: "outbound",
                    });
                    // Query properties and the next param
                    const props = await api.getProperties(param);
                    if (!props) {
                        this.driver.controllerLog.logNode(node.id, {
                            endpoint: this.endpointIndex,
                            message: `Querying parameter #${param} information timed out, skipping interview...`,
                            level: "warn",
                        });
                        return;
                    }
                    const { nextParameter, ...properties } = props;
                    let logMessage;
                    if (properties.valueSize === 0) {
                        logMessage = `Parameter #${param} is unsupported. Next parameter: ${nextParameter}`;
                    }
                    else {
                        // Query name and info only if the parameter is supported
                        const name = (_e = (await api.getName(param))) !== null && _e !== void 0 ? _e : "(unknown)";
                        // Skip the info query for bugged devices
                        if (!((_g = (_f = node.deviceConfig) === null || _f === void 0 ? void 0 : _f.compat) === null || _g === void 0 ? void 0 : _g.skipConfigurationInfoQuery)) {
                            await api.getInfo(param);
                        }
                        logMessage = `received information for parameter #${param}:
parameter name:      ${name}
value format:        ${shared_1.getEnumMemberName(ValueFormat, properties.valueFormat)}
value size:          ${properties.valueSize} bytes
min value:           ${(_j = (_h = properties.minValue) === null || _h === void 0 ? void 0 : _h.toString()) !== null && _j !== void 0 ? _j : "undefined"}
max value:           ${(_l = (_k = properties.maxValue) === null || _k === void 0 ? void 0 : _k.toString()) !== null && _l !== void 0 ? _l : "undefined"}
default value:       ${(_o = (_m = properties.defaultValue) === null || _m === void 0 ? void 0 : _m.toString()) !== null && _o !== void 0 ? _o : "undefined"}
is read-only:        ${!!properties.isReadonly}
is advanced (UI):    ${!!properties.isAdvanced}
has bulk support:    ${!properties.noBulkSupport}
alters capabilities: ${!!properties.altersCapabilities}`;
                    }
                    this.driver.controllerLog.logNode(node.id, {
                        endpoint: this.endpointIndex,
                        message: logMessage,
                        direction: "inbound",
                    });
                    // Some devices report their parameter 1 instead of 0 as the next one
                    // when reaching the end. To avoid infinite loops, stop scanning
                    // once the next parameter is lower than the current one
                    if (nextParameter > param) {
                        param = nextParameter;
                    }
                    else {
                        break;
                    }
                }
            }
            // Query the values of supported parameters during every interview
            const parameters = arrays_1.distinct(this.getDefinedValueIDs()
                .map((v) => v.property)
                .filter((p) => typeof p === "number"));
            for (const param of parameters) {
                this.driver.controllerLog.logNode(node.id, {
                    endpoint: this.endpointIndex,
                    message: `querying parameter #${param} value...`,
                    direction: "outbound",
                });
                await api.get(param);
            }
        }
        // Remember that the interview is complete
        this.interviewComplete = true;
    }
    /**
     * Whether this node's param information was loaded from a config file.
     * If this is true, we don't trust what the node reports
     */
    get isParamInformationFromConfig() {
        return (this.getValueDB().getValue(isParamInfoFromConfigValueId) === true);
    }
    /**
     * @internal
     * Stores config parameter metadata for this CC's node
     */
    extendParamInformation(parameter, valueBitMask, info) {
        // Don't trust param information that a node reports if we have already loaded it from a config file
        if (this.isParamInformationFromConfig)
            return;
        const valueDB = this.getValueDB();
        const valueId = getParamInformationValueID(parameter, valueBitMask);
        // Retrieve the base metadata
        const metadata = this.getParamInformation(parameter, valueBitMask);
        // Override it with new data
        Object.assign(metadata, info);
        // And store it back
        valueDB.setMetadata(valueId, metadata);
    }
    /**
     * @internal
     * Returns stored config parameter metadata for this CC's node
     */
    getParamInformation(parameter, valueBitMask) {
        var _c;
        const valueDB = this.getValueDB();
        const valueId = getParamInformationValueID(parameter, valueBitMask);
        return ((_c = valueDB.getMetadata(valueId)) !== null && _c !== void 0 ? _c : {
            ...core_1.ValueMetadata.Any,
        });
    }
    /**
     * Returns stored config parameter metadata for all partial config params addressed with the given parameter number
     */
    getPartialParamInfos(parameter) {
        const valueDB = this.getValueDB();
        return valueDB.findMetadata((id) => id.commandClass === this.ccId &&
            id.property === parameter &&
            id.propertyKey != undefined);
    }
    /**
     * Returns stored config parameter metadata for all partial config params addressed with the given parameter number
     */
    composePartialParamValue(parameter, valueBitMask, maskedValue) {
        const valueDB = this.getValueDB();
        // Add the other values
        const otherValues = valueDB
            .findValues((id) => id.commandClass === this.ccId &&
            id.property === parameter &&
            id.propertyKey != undefined)
            .map(({ propertyKey, value }) => propertyKey === valueBitMask
            ? 0
            : value <<
                core_1.getMinimumShiftForBitMask(propertyKey))
            .reduce((prev, cur) => prev | cur, 0);
        return ((otherValues & ~valueBitMask) |
            (maskedValue << core_1.getMinimumShiftForBitMask(valueBitMask)));
    }
    serializeValuesForCache() {
        // Leave out the paramInformation if we have loaded it from a config file
        let values = super.serializeValuesForCache();
        values = values.filter((v) => v.property !== "isParamInformationFromConfig");
        return values;
    }
    serializeMetadataForCache() {
        // Leave out the param metadata if we have loaded it from a config file
        let metadata = super.serializeMetadataForCache();
        if (this.isParamInformationFromConfig) {
            metadata = metadata.filter((m) => typeof m.property === "number");
        }
        return metadata;
    }
    /** Deserializes the config parameter info from a config file */
    deserializeParamInformationFromConfig(config) {
        const valueDB = this.getValueDB();
        // Clear old param information
        for (const meta of valueDB.getAllMetadata(this.ccId)) {
            if (typeof meta.property === "number") {
                // this is a param information, delete it
                valueDB.setMetadata(meta, undefined, 
                // Don't emit the added/updated events, as this will spam applications with untranslated events
                { noEvent: true });
            }
        }
        // Allow overwriting the param info (mark it as unloaded)
        valueDB.setValue(isParamInfoFromConfigValueId, false);
        for (const [param, info] of config.entries()) {
            // We need to make the config information compatible with the
            // format that ConfigurationCC reports
            const paramInfo = {
                // TODO: Make this smarter! (0...1 ==> boolean)
                type: "number",
                valueSize: info.valueSize,
                min: info.minValue,
                max: info.maxValue,
                default: info.defaultValue,
                unit: info.unit,
                format: info.unsigned
                    ? ValueFormat.UnsignedInteger
                    : ValueFormat.SignedInteger,
                readable: !info.writeOnly,
                writeable: !info.readOnly,
                allowManualEntry: info.allowManualEntry,
                states: info.options.length > 0
                    ? objects_1.composeObject(info.options.map(({ label, value }) => [
                        value.toString(),
                        label,
                    ]))
                    : undefined,
                label: info.label,
                description: info.description,
                isFromConfig: true,
            };
            this.extendParamInformation(param.parameter, param.valueBitMask, paramInfo);
        }
        // Remember that we loaded the param information from a config file
        valueDB.setValue(isParamInfoFromConfigValueId, true);
    }
    translatePropertyKey(property, propertyKey) {
        if (typeof property === "number" &&
            (propertyKey == undefined || typeof propertyKey === "number")) {
            // This CC names all configuration parameters differently,
            // so no name for the property key is required
            return undefined;
        }
        return super.translateProperty(property, propertyKey);
    }
    translateProperty(property, propertyKey) {
        // Try to retrieve the configured param label
        if (typeof property === "number" &&
            (propertyKey == undefined || typeof propertyKey === "number")) {
            const paramInfo = this.getParamInformation(property, propertyKey);
            if (paramInfo.label)
                return paramInfo.label;
            // fall back to paramXYZ[_key] if none is defined
            let ret = `param${strings_1.padStart(property.toString(), 3, "0")}`;
            if (propertyKey != undefined) {
                ret += "_" + propertyKey.toString();
            }
            return ret;
        }
        return super.translateProperty(property, propertyKey);
    }
};
ConfigurationCC = __decorate([
    CommandClass_1.commandClass(core_1.CommandClasses.Configuration),
    CommandClass_1.implementedVersion(4)
], ConfigurationCC);
exports.ConfigurationCC = ConfigurationCC;
let ConfigurationCCReport = class ConfigurationCCReport extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        // All fields must be present
        core_1.validatePayload(this.payload.length > 2);
        this._parameter = this.payload[0];
        this._valueSize = this.payload[1] & 0b111;
        // Ensure we received a valid report
        core_1.validatePayload(this._valueSize >= 1, this._valueSize <= 4, this.payload.length >= 2 + this._valueSize);
        const oldParamInformation = this.getParamInformation(this._parameter);
        this._value = parseValue(this.payload.slice(2), this._valueSize, 
        // In Config CC v1/v2, this must be SignedInteger
        // As those nodes don't communicate any parameter information
        // we fall back to that default value anyways
        oldParamInformation.format || ValueFormat.SignedInteger);
        // Store the parameter size and value
        this.extendParamInformation(this._parameter, undefined, {
            valueSize: this._valueSize,
            type: oldParamInformation.format === ValueFormat.BitField
                ? "number[]"
                : "number",
        });
        if (this.version < 3 &&
            !this.isParamInformationFromConfig &&
            oldParamInformation.min == undefined &&
            oldParamInformation.max == undefined) {
            const isSigned = oldParamInformation.format == undefined ||
                oldParamInformation.format === ValueFormat.SignedInteger;
            this.extendParamInformation(this._parameter, undefined, core_1.getIntegerLimits(this._valueSize, isSigned));
        }
        // And store the value itself
        // If we have partial config params defined, we need to split the value
        const partialParams = this.getPartialParamInfos(this._parameter);
        if (partialParams.length > 0) {
            for (const param of partialParams) {
                if (typeof param.propertyKey === "number") {
                    this.getValueDB().setValue({
                        commandClass: this.ccId,
                        property: this._parameter,
                        propertyKey: param.propertyKey,
                    }, (this._value & param.propertyKey) >>>
                        core_1.getMinimumShiftForBitMask(param.propertyKey));
                }
            }
        }
        else {
            // This is a single param
            this.getValueDB().setValue({
                commandClass: this.ccId,
                property: this._parameter,
            }, this._value);
        }
    }
    get parameter() {
        return this._parameter;
    }
    get value() {
        return this._value;
    }
    get valueSize() {
        return this._valueSize;
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "parameter #": this.parameter,
                "value size": this.valueSize,
                value: configValueToString(this.value),
            },
        };
    }
};
ConfigurationCCReport = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.Report)
], ConfigurationCCReport);
exports.ConfigurationCCReport = ConfigurationCCReport;
function testResponseForConfigurationGet(sent, received) {
    // We expect a Configuration Report that matches the requested parameter
    return (sent.parameter === received.parameter || sent.allowUnexpectedResponse);
}
let ConfigurationCCGet = class ConfigurationCCGet extends ConfigurationCC {
    constructor(driver, options) {
        var _c;
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.parameter = options.parameter;
            this.allowUnexpectedResponse =
                (_c = options.allowUnexpectedResponse) !== null && _c !== void 0 ? _c : false;
        }
    }
    serialize() {
        this.payload = Buffer.from([this.parameter]);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "parameter #": this.parameter },
        };
    }
};
ConfigurationCCGet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.Get),
    CommandClass_1.expectedCCResponse(ConfigurationCCReport, testResponseForConfigurationGet)
], ConfigurationCCGet);
exports.ConfigurationCCGet = ConfigurationCCGet;
let ConfigurationCCSet = class ConfigurationCCSet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.parameter = options.parameter;
            // According to SDS14223 this flag SHOULD NOT be set
            // Because we don't want to test the behavior, we enforce that it MUST not be set
            // on legacy nodes
            if (options.resetToDefault && this.version <= 3) {
                throw new core_1.ZWaveError(`The resetToDefault flag MUST not be used on nodes implementing ConfigurationCC V3 or less!`, core_1.ZWaveErrorCodes.ConfigurationCC_NoResetToDefaultOnLegacyDevices);
            }
            this.resetToDefault = !!options.resetToDefault;
            if (!options.resetToDefault) {
                // TODO: Default to the stored value size
                this.valueSize = options.valueSize;
                this.value = options.value;
            }
        }
    }
    serialize() {
        const valueSize = this.resetToDefault ? 1 : this.valueSize;
        const payloadLength = 2 + valueSize;
        this.payload = Buffer.alloc(payloadLength, 0);
        this.payload[0] = this.parameter;
        this.payload[1] =
            (this.resetToDefault ? 128 : 0) | (valueSize & 0b111);
        if (!this.resetToDefault) {
            const valueFormat = this.getParamInformation(this.parameter).format ||
                ValueFormat.SignedInteger;
            // Make sure that the given value fits into the value size
            if (typeof this.value === "number" &&
                !isSafeValue(this.value, valueSize, valueFormat)) {
                // If there is a value size configured, check that the given value is compatible
                throwInvalidValueError(this.value, this.parameter, valueSize, valueFormat);
            }
            try {
                serializeValue(this.payload, 2, valueSize, valueFormat, this.value);
            }
            catch (e) {
                tryCatchOutOfBoundsError(e, this.value, this.parameter, valueSize, valueFormat);
            }
        }
        return super.serialize();
    }
    toLogEntry() {
        const message = {
            "parameter #": this.parameter,
            "reset to default": this.resetToDefault,
        };
        if (this.valueSize != undefined) {
            message["value size"] = this.valueSize;
        }
        if (this.value != undefined) {
            message.value = configValueToString(this.value);
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
ConfigurationCCSet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.Set)
], ConfigurationCCSet);
exports.ConfigurationCCSet = ConfigurationCCSet;
function getResponseForBulkSet(cc) {
    return cc.handshake ? ConfigurationCCBulkReport : undefined;
}
let ConfigurationCCBulkSet = class ConfigurationCCBulkSet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this._parameters = options.parameters;
            if (this._parameters.length < 1) {
                throw new core_1.ZWaveError(`In a ConfigurationCC.BulkSet, parameters must be a non-empty array`, core_1.ZWaveErrorCodes.CC_Invalid);
            }
            else if (!core_1.isConsecutiveArray(this._parameters)) {
                throw new core_1.ZWaveError(`A ConfigurationCC.BulkSet can only be used for consecutive parameters`, core_1.ZWaveErrorCodes.CC_Invalid);
            }
            this._handshake = !!options.handshake;
            this._resetToDefault = !!options.resetToDefault;
            if (!!options.resetToDefault) {
                this._valueSize = 1;
                this._values = this._parameters.map(() => 0);
            }
            else {
                this._valueSize = options.valueSize;
                this._values = options.values;
            }
        }
    }
    get parameters() {
        return this._parameters;
    }
    get resetToDefault() {
        return this._resetToDefault;
    }
    get valueSize() {
        return this._valueSize;
    }
    get values() {
        return this._values;
    }
    get handshake() {
        return this._handshake;
    }
    serialize() {
        const valueSize = this._resetToDefault ? 1 : this.valueSize;
        const payloadLength = 4 + valueSize * this.parameters.length;
        this.payload = Buffer.alloc(payloadLength, 0);
        this.payload.writeUInt16BE(this.parameters[0], 0);
        this.payload[2] = this.parameters.length;
        this.payload[3] =
            (this._resetToDefault ? 128 : 0) |
                (this.handshake ? 64 : 0) |
                (valueSize & 0b111);
        if (!this._resetToDefault) {
            for (let i = 0; i < this.parameters.length; i++) {
                const value = this._values[i];
                const param = this._parameters[i];
                const valueFormat = this.getParamInformation(param).format ||
                    ValueFormat.SignedInteger;
                // Make sure that the given value fits into the value size
                if (!isSafeValue(value, valueSize, valueFormat)) {
                    // If there is a value size configured, check that the given value is compatible
                    throwInvalidValueError(value, param, valueSize, valueFormat);
                }
                try {
                    serializeValue(this.payload, 4 + i * valueSize, valueSize, valueFormat, value);
                }
                catch (e) {
                    tryCatchOutOfBoundsError(e, value, param, valueSize, valueFormat);
                }
            }
        }
        return super.serialize();
    }
    toLogEntry() {
        const message = {
            handshake: this.handshake,
            "reset to default": this.resetToDefault,
            "value size": this.valueSize,
        };
        if (this._values.length > 0) {
            message.values = this._values
                .map((value, i) => `\n· #${this._parameters[i]}: ${configValueToString(value)}`)
                .join("");
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
ConfigurationCCBulkSet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.BulkSet),
    CommandClass_1.expectedCCResponse(getResponseForBulkSet)
], ConfigurationCCBulkSet);
exports.ConfigurationCCBulkSet = ConfigurationCCBulkSet;
let ConfigurationCCBulkReport = class ConfigurationCCBulkReport extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        this._values = new Map();
        // Ensure we received enough bytes for the preamble
        core_1.validatePayload(this.payload.length >= 5);
        const firstParameter = this.payload.readUInt16BE(0);
        const numParams = this.payload[2];
        this._reportsToFollow = this.payload[3];
        this._defaultValues = !!(this.payload[4] & 128);
        this._isHandshakeResponse = !!(this.payload[4] & 64);
        this._valueSize = this.payload[4] & 0b111;
        // Ensure the payload is long enough for all reported values
        core_1.validatePayload(this.payload.length >= 5 + numParams * this._valueSize);
        for (let i = 0; i < numParams; i++) {
            const param = firstParameter + i;
            this._values.set(param, parseValue(this.payload.slice(5 + i * this.valueSize), this.valueSize, this.getParamInformation(param).format ||
                ValueFormat.SignedInteger));
        }
        // Store every received parameter
        for (const [parameter, value] of this._values.entries()) {
            this.getValueDB().setValue({
                commandClass: this.ccId,
                property: parameter,
            }, value);
        }
    }
    get reportsToFollow() {
        return this._reportsToFollow;
    }
    getPartialCCSessionId() {
        // We don't expect the driver to merge CCs but we want to wait until all reports have been received
        return {};
    }
    expectMoreMessages() {
        return this._reportsToFollow > 0;
    }
    get defaultValues() {
        return this._defaultValues;
    }
    get isHandshakeResponse() {
        return this._isHandshakeResponse;
    }
    get valueSize() {
        return this._valueSize;
    }
    get values() {
        return this._values;
    }
    toLogEntry() {
        const message = {
            "handshake response": this._isHandshakeResponse,
            "default values": this._defaultValues,
            "value size": this._valueSize,
            "reports to follow": this.reportsToFollow,
        };
        if (this._values.size > 0) {
            message.values = [...this._values]
                .map(([param, value]) => `
· #${param}: ${configValueToString(value)}`)
                .join("");
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
ConfigurationCCBulkReport = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.BulkReport)
], ConfigurationCCBulkReport);
exports.ConfigurationCCBulkReport = ConfigurationCCBulkReport;
let ConfigurationCCBulkGet = class ConfigurationCCBulkGet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this._parameters = options.parameters.sort();
            if (!core_1.isConsecutiveArray(this.parameters)) {
                throw new core_1.ZWaveError(`A ConfigurationCC.BulkGet can only be used for consecutive parameters`, core_1.ZWaveErrorCodes.CC_Invalid);
            }
        }
    }
    get parameters() {
        return this._parameters;
    }
    serialize() {
        this.payload = Buffer.allocUnsafe(3);
        this.payload.writeUInt16BE(this.parameters[0], 0);
        this.payload[2] = this.parameters.length;
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { parameters: this.parameters.join(", ") },
        };
    }
};
ConfigurationCCBulkGet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.BulkGet),
    CommandClass_1.expectedCCResponse(ConfigurationCCBulkReport)
], ConfigurationCCBulkGet);
exports.ConfigurationCCBulkGet = ConfigurationCCBulkGet;
let ConfigurationCCNameReport = class ConfigurationCCNameReport extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        // Parameter and # of reports must be present
        core_1.validatePayload(this.payload.length >= 3);
        this._parameter = this.payload.readUInt16BE(0);
        this._reportsToFollow = this.payload[2];
        if (this._reportsToFollow > 0) {
            // If more reports follow, the info must at least be one byte
            core_1.validatePayload(this.payload.length >= 4);
        }
        this._name = this.payload.slice(3).toString("utf8");
    }
    get parameter() {
        return this._parameter;
    }
    get name() {
        return this._name;
    }
    get reportsToFollow() {
        return this._reportsToFollow;
    }
    getPartialCCSessionId() {
        // Distinguish sessions by the parameter number
        return { parameter: this._parameter };
    }
    expectMoreMessages() {
        return this._reportsToFollow > 0;
    }
    mergePartialCCs(partials) {
        // Concat the name
        this._name = [...partials, this]
            .map((report) => report._name)
            .reduce((prev, cur) => prev + cur, "");
        this.extendParamInformation(this.parameter, undefined, {
            name: this.name,
        });
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "parameter #": this.parameter,
                name: this.name,
                "reports to follow": this.reportsToFollow,
            },
        };
    }
};
ConfigurationCCNameReport = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.NameReport)
], ConfigurationCCNameReport);
exports.ConfigurationCCNameReport = ConfigurationCCNameReport;
let ConfigurationCCNameGet = class ConfigurationCCNameGet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.parameter = options.parameter;
        }
    }
    serialize() {
        this.payload = Buffer.allocUnsafe(2);
        this.payload.writeUInt16BE(this.parameter, 0);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "parameter #": this.parameter },
        };
    }
};
ConfigurationCCNameGet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.NameGet),
    CommandClass_1.expectedCCResponse(ConfigurationCCNameReport)
], ConfigurationCCNameGet);
exports.ConfigurationCCNameGet = ConfigurationCCNameGet;
let ConfigurationCCInfoReport = class ConfigurationCCInfoReport extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        // Parameter and # of reports must be present
        core_1.validatePayload(this.payload.length >= 3);
        this._parameter = this.payload.readUInt16BE(0);
        this._reportsToFollow = this.payload[2];
        if (this._reportsToFollow > 0) {
            // If more reports follow, the info must at least be one byte
            core_1.validatePayload(this.payload.length >= 4);
        }
        this._info = this.payload.slice(3).toString("utf8");
    }
    get parameter() {
        return this._parameter;
    }
    get info() {
        return this._info;
    }
    get reportsToFollow() {
        return this._reportsToFollow;
    }
    getPartialCCSessionId() {
        // Distinguish sessions by the parameter number
        return { parameter: this._parameter };
    }
    expectMoreMessages() {
        return this._reportsToFollow > 0;
    }
    mergePartialCCs(partials) {
        // Concat the info
        this._info = [...partials, this]
            .map((report) => report._info)
            .reduce((prev, cur) => prev + cur, "");
        this.extendParamInformation(this._parameter, undefined, {
            info: this._info,
        });
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: {
                "parameter #": this.parameter,
                info: this.info,
                "reports to follow": this.reportsToFollow,
            },
        };
    }
};
ConfigurationCCInfoReport = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.InfoReport)
], ConfigurationCCInfoReport);
exports.ConfigurationCCInfoReport = ConfigurationCCInfoReport;
let ConfigurationCCInfoGet = class ConfigurationCCInfoGet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.parameter = options.parameter;
        }
    }
    serialize() {
        this.payload = Buffer.allocUnsafe(2);
        this.payload.writeUInt16BE(this.parameter, 0);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "parameter #": this.parameter },
        };
    }
};
ConfigurationCCInfoGet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.InfoGet),
    CommandClass_1.expectedCCResponse(ConfigurationCCInfoReport)
], ConfigurationCCInfoGet);
exports.ConfigurationCCInfoGet = ConfigurationCCInfoGet;
let ConfigurationCCPropertiesReport = class ConfigurationCCPropertiesReport extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        core_1.validatePayload(this.payload.length >= 3);
        this._parameter = this.payload.readUInt16BE(0);
        this._valueFormat = (this.payload[2] & 0b111000) >>> 3;
        this._valueSize = this.payload[2] & 0b111;
        // GH#1309 Some devices don't tell us the first parameter if we query #0
        // Instead, they contain 0x000000
        if (this._valueSize === 0 && this.payload.length < 5) {
            this._nextParameter = 0;
            return;
        }
        // Ensure the payload contains the two bytes for next parameter
        const nextParameterOffset = 3 + 3 * this._valueSize;
        core_1.validatePayload(this.payload.length >= nextParameterOffset + 2);
        if (this.valueSize > 0) {
            if (this._valueFormat !== ValueFormat.BitField) {
                this._minValue = parseValue(this.payload.slice(3), this._valueSize, this._valueFormat);
            }
            this._maxValue = parseValue(this.payload.slice(3 + this._valueSize), this._valueSize, this._valueFormat);
            this._defaultValue = parseValue(this.payload.slice(3 + 2 * this._valueSize), this._valueSize, this._valueFormat);
        }
        if (this.version < 4) {
            // Read the last 2 bytes to work around nodes not omitting min/max value when their size is 0
            this._nextParameter = this.payload.readUInt16BE(this.payload.length - 2);
        }
        else {
            this._nextParameter = this.payload.readUInt16BE(nextParameterOffset);
            // Ensure the payload contains a byte for the 2nd option flags
            core_1.validatePayload(this.payload.length >= nextParameterOffset + 3);
            const options1 = this.payload[2];
            const options2 = this.payload[3 + 3 * this.valueSize + 2];
            this._altersCapabilities = !!(options1 & 128);
            this._isReadonly = !!(options1 & 64);
            this._isAdvanced = !!(options2 & 0b1);
            this._noBulkSupport = !!(options2 & 0b10);
        }
        // If we actually received parameter info, store it
        if (this._valueSize > 0) {
            const valueType = this._valueFormat === ValueFormat.SignedInteger ||
                this._valueFormat === ValueFormat.UnsignedInteger
                ? "number"
                : "number[]";
            const paramInfo = core_1.stripUndefined({
                type: valueType,
                valueFormat: this._valueFormat,
                valueSize: this._valueSize,
                minValue: this._minValue,
                maxValue: this._maxValue,
                defaultValue: this._defaultValue,
                requiresReInclusion: this._altersCapabilities,
                writeable: !this._isReadonly,
                isAdvanced: this._isAdvanced,
                noBulkSupport: this._noBulkSupport,
            });
            this.extendParamInformation(this._parameter, undefined, paramInfo);
        }
    }
    get parameter() {
        return this._parameter;
    }
    get valueSize() {
        return this._valueSize;
    }
    get valueFormat() {
        return this._valueFormat;
    }
    get minValue() {
        return this._minValue;
    }
    get maxValue() {
        return this._maxValue;
    }
    get defaultValue() {
        return this._defaultValue;
    }
    get nextParameter() {
        return this._nextParameter;
    }
    get altersCapabilities() {
        return this._altersCapabilities;
    }
    get isReadonly() {
        return this._isReadonly;
    }
    get isAdvanced() {
        return this._isAdvanced;
    }
    get noBulkSupport() {
        return this._noBulkSupport;
    }
    toLogEntry() {
        const message = {
            "parameter #": this._parameter,
            "next param #": this._nextParameter,
            "value size": this._valueSize,
            "value format": shared_1.getEnumMemberName(ValueFormat, this._valueFormat),
        };
        if (this._minValue != undefined) {
            message["min value"] = configValueToString(this._minValue);
        }
        if (this._maxValue != undefined) {
            message["max value"] = configValueToString(this._maxValue);
        }
        if (this._defaultValue != undefined) {
            message["default value"] = configValueToString(this._defaultValue);
        }
        if (this._altersCapabilities != undefined) {
            message["alters capabilities"] = this._altersCapabilities;
        }
        if (this._isReadonly != undefined) {
            message.readonly = this._isReadonly;
        }
        if (this._isAdvanced != undefined) {
            message.advanced = this._isAdvanced;
        }
        if (this._noBulkSupport != undefined) {
            message["bulk support"] = !this._noBulkSupport;
        }
        return {
            ...super.toLogEntry(),
            message,
        };
    }
};
ConfigurationCCPropertiesReport = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.PropertiesReport)
], ConfigurationCCPropertiesReport);
exports.ConfigurationCCPropertiesReport = ConfigurationCCPropertiesReport;
let ConfigurationCCPropertiesGet = class ConfigurationCCPropertiesGet extends ConfigurationCC {
    constructor(driver, options) {
        super(driver, options);
        if (CommandClass_1.gotDeserializationOptions(options)) {
            // TODO: Deserialize payload
            throw new core_1.ZWaveError(`${this.constructor.name}: deserialization not implemented`, core_1.ZWaveErrorCodes.Deserialization_NotImplemented);
        }
        else {
            this.parameter = options.parameter;
        }
    }
    serialize() {
        this.payload = Buffer.allocUnsafe(2);
        this.payload.writeUInt16BE(this.parameter, 0);
        return super.serialize();
    }
    toLogEntry() {
        return {
            ...super.toLogEntry(),
            message: { "parameter #": this.parameter },
        };
    }
};
ConfigurationCCPropertiesGet = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.PropertiesGet),
    CommandClass_1.expectedCCResponse(ConfigurationCCPropertiesReport)
], ConfigurationCCPropertiesGet);
exports.ConfigurationCCPropertiesGet = ConfigurationCCPropertiesGet;
let ConfigurationCCDefaultReset = class ConfigurationCCDefaultReset extends ConfigurationCC {
};
ConfigurationCCDefaultReset = __decorate([
    CommandClass_1.CCCommand(ConfigurationCommand.DefaultReset)
], ConfigurationCCDefaultReset);
exports.ConfigurationCCDefaultReset = ConfigurationCCDefaultReset;
function isSafeValue(value, size, format) {
    let minValue;
    let maxValue;
    switch (format) {
        case ValueFormat.SignedInteger:
            minValue = -Math.pow(2, 8 * size - 1);
            maxValue = Math.pow(2, 8 * size - 1) - 1;
            break;
        case ValueFormat.UnsignedInteger:
        case ValueFormat.Enumerated:
            minValue = 0;
            maxValue = Math.pow(2, 8 * size);
            break;
        case ValueFormat.BitField:
        default:
            throw new Error("not implemented");
    }
    return minValue <= value && value <= maxValue;
}
/** Interprets values from the payload depending on the value format */
function parseValue(raw, size, format) {
    switch (format) {
        case ValueFormat.SignedInteger:
            return raw.readIntBE(0, size);
        case ValueFormat.UnsignedInteger:
        case ValueFormat.Enumerated:
            return raw.readUIntBE(0, size);
        case ValueFormat.BitField:
            return new Set(core_1.parseBitMask(raw.slice(0, size)));
    }
}
function throwInvalidValueError(value, parameter, valueSize, valueFormat) {
    throw new core_1.ZWaveError(`The value ${value} is invalid for configuration parameter ${parameter} (size = ${valueSize}, format = ${shared_1.getEnumMemberName(ValueFormat, valueFormat)})!`, core_1.ZWaveErrorCodes.Argument_Invalid);
}
function tryCatchOutOfBoundsError(e, value, parameter, valueSize, valueFormat) {
    if (e.message.includes("out of bounds")) {
        throwInvalidValueError(value, parameter, valueSize, valueFormat);
    }
    else {
        throw e;
    }
}
/** Serializes values into the payload according to the value format */
function serializeValue(payload, offset, size, format, value) {
    switch (format) {
        case ValueFormat.SignedInteger:
            payload.writeIntBE(value, offset, size);
            return;
        case ValueFormat.UnsignedInteger:
        case ValueFormat.Enumerated:
            payload.writeUIntBE(value, offset, size);
            return;
        case ValueFormat.BitField: {
            const mask = core_1.encodeBitMask([...value.values()], size * 8);
            mask.copy(payload, offset);
            return;
        }
    }
}

//# sourceMappingURL=ConfigurationCC.js.map
