"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZWaveController = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const arrays_1 = require("alcalzone-shared/arrays");
const deferred_promise_1 = require("alcalzone-shared/deferred-promise");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const events_1 = require("events");
const ManufacturerSpecificCC_1 = require("../commandclass/ManufacturerSpecificCC");
const Constants_1 = require("../message/Constants");
const DeviceClass_1 = require("../node/DeviceClass");
const Node_1 = require("../node/Node");
const Types_1 = require("../node/Types");
const VirtualNode_1 = require("../node/VirtualNode");
const AddNodeToNetworkRequest_1 = require("./AddNodeToNetworkRequest");
const AssignReturnRouteMessages_1 = require("./AssignReturnRouteMessages");
const DeleteReturnRouteMessages_1 = require("./DeleteReturnRouteMessages");
const GetControllerCapabilitiesMessages_1 = require("./GetControllerCapabilitiesMessages");
const GetControllerIdMessages_1 = require("./GetControllerIdMessages");
const GetControllerVersionMessages_1 = require("./GetControllerVersionMessages");
const GetRoutingInfoMessages_1 = require("./GetRoutingInfoMessages");
const GetSerialApiCapabilitiesMessages_1 = require("./GetSerialApiCapabilitiesMessages");
const GetSerialApiInitDataMessages_1 = require("./GetSerialApiInitDataMessages");
const GetSUCNodeIdMessages_1 = require("./GetSUCNodeIdMessages");
const HardResetRequest_1 = require("./HardResetRequest");
const IsFailedNodeMessages_1 = require("./IsFailedNodeMessages");
const RemoveFailedNodeMessages_1 = require("./RemoveFailedNodeMessages");
const RemoveNodeFromNetworkRequest_1 = require("./RemoveNodeFromNetworkRequest");
const ReplaceFailedNodeRequest_1 = require("./ReplaceFailedNodeRequest");
const RequestNodeNeighborUpdateMessages_1 = require("./RequestNodeNeighborUpdateMessages");
const SetSerialApiTimeoutsMessages_1 = require("./SetSerialApiTimeoutsMessages");
const ZWaveLibraryTypes_1 = require("./ZWaveLibraryTypes");
class ZWaveController extends events_1.EventEmitter {
    /** @internal */
    constructor(driver) {
        super();
        this.driver = driver;
        this._exclusionActive = false;
        this._inclusionActive = false;
        this._includeNonSecure = false;
        this._includeController = false;
        this._healNetworkActive = false;
        this._healNetworkProgress = new Map();
        this._nodes = new Map();
        this._nodes.getOrThrow = function (nodeId) {
            const node = this.get(nodeId);
            if (!node) {
                throw new core_1.ZWaveError(`Node ${nodeId} was not found!`, core_1.ZWaveErrorCodes.Controller_NodeNotFound);
            }
            return node;
        }.bind(this._nodes);
        // register message handlers
        driver.registerRequestHandler(Constants_1.FunctionType.AddNodeToNetwork, this.handleAddNodeRequest.bind(this));
        driver.registerRequestHandler(Constants_1.FunctionType.RemoveNodeFromNetwork, this.handleRemoveNodeRequest.bind(this));
        driver.registerRequestHandler(Constants_1.FunctionType.ReplaceFailedNode, this.handleReplaceNodeRequest.bind(this));
    }
    get libraryVersion() {
        return this._libraryVersion;
    }
    get type() {
        return this._type;
    }
    /** A 32bit number identifying the current network */
    get homeId() {
        return this._homeId;
    }
    /** The ID of the controller in the current network */
    get ownNodeId() {
        return this._ownNodeId;
    }
    get isSecondary() {
        return this._isSecondary;
    }
    get isUsingHomeIdFromOtherNetwork() {
        return this._isUsingHomeIdFromOtherNetwork;
    }
    get isSISPresent() {
        return this._isSISPresent;
    }
    get wasRealPrimary() {
        return this._wasRealPrimary;
    }
    get isStaticUpdateController() {
        return this._isStaticUpdateController;
    }
    get isSlave() {
        return this._isSlave;
    }
    get serialApiVersion() {
        return this._serialApiVersion;
    }
    get manufacturerId() {
        return this._manufacturerId;
    }
    get productType() {
        return this._productType;
    }
    get productId() {
        return this._productId;
    }
    get supportedFunctionTypes() {
        return this._supportedFunctionTypes;
    }
    /** Checks if a given Z-Wave function type is supported by this controller */
    isFunctionSupported(functionType) {
        if (this._supportedFunctionTypes == null) {
            throw new core_1.ZWaveError("Cannot check yet if a function is supported by the controller. The interview process has not been completed.", core_1.ZWaveErrorCodes.Driver_NotReady);
        }
        return this._supportedFunctionTypes.indexOf(functionType) > -1;
    }
    get sucNodeId() {
        return this._sucNodeId;
    }
    get supportsTimers() {
        return this._supportsTimers;
    }
    /** A dictionary of the nodes connected to this controller */
    get nodes() {
        return this._nodes;
    }
    /** Returns a reference to the (virtual) broadcast node, which allows sending commands to all nodes */
    getBroadcastNode() {
        return new VirtualNode_1.VirtualNode(core_1.NODE_ID_BROADCAST, this.driver, this.nodes.values());
    }
    /** Creates a virtual node that can be used to send multicast commands to several nodes */
    getMulticastGroup(nodeIDs) {
        return new VirtualNode_1.VirtualNode(undefined, this.driver, nodeIDs.map((id) => this._nodes.get(id)));
    }
    /**
     * @internal
     * Interviews the controller for the necessary information.
     * @param initValueDBs Asynchronous callback for the driver to initialize the Value DBs before nodes are created
     * @param restoreFromCache Asynchronous callback for the driver to restore the network from cache after nodes are created
     */
    async interview(initValueDBs, restoreFromCache) {
        this.driver.controllerLog.print("beginning interview...");
        // get basic controller version info
        this.driver.controllerLog.print(`querying version info...`);
        const version = await this.driver.sendMessage(new GetControllerVersionMessages_1.GetControllerVersionRequest(this.driver), {
            supportCheck: false,
        });
        this._libraryVersion = version.libraryVersion;
        this._type = version.controllerType;
        this.driver.controllerLog.print(`received version info:
  controller type: ${ZWaveLibraryTypes_1.ZWaveLibraryTypes[this._type]}
  library version: ${this._libraryVersion}`);
        // get the home and node id of the controller
        this.driver.controllerLog.print(`querying controller IDs...`);
        const ids = await this.driver.sendMessage(new GetControllerIdMessages_1.GetControllerIdRequest(this.driver), { supportCheck: false });
        this._homeId = ids.homeId;
        this._ownNodeId = ids.ownNodeId;
        this.driver.controllerLog.print(`received controller IDs:
  home ID:     ${shared_1.num2hex(this._homeId)}
  own node ID: ${this._ownNodeId}`);
        // find out what the controller can do
        this.driver.controllerLog.print(`querying controller capabilities...`);
        const ctrlCaps = await this.driver.sendMessage(new GetControllerCapabilitiesMessages_1.GetControllerCapabilitiesRequest(this.driver), {
            supportCheck: false,
        });
        this._isSecondary = ctrlCaps.isSecondary;
        this._isUsingHomeIdFromOtherNetwork =
            ctrlCaps.isUsingHomeIdFromOtherNetwork;
        this._isSISPresent = ctrlCaps.isSISPresent;
        this._wasRealPrimary = ctrlCaps.wasRealPrimary;
        this._isStaticUpdateController = ctrlCaps.isStaticUpdateController;
        this.driver.controllerLog.print(`received controller capabilities:
  controller role:     ${this._isSecondary ? "secondary" : "primary"}
  is in other network: ${this._isUsingHomeIdFromOtherNetwork}
  is SIS present:      ${this._isSISPresent}
  was real primary:    ${this._wasRealPrimary}
  is a SUC:            ${this._isStaticUpdateController}`);
        // find out which part of the API is supported
        this.driver.controllerLog.print(`querying API capabilities...`);
        const apiCaps = await this.driver.sendMessage(new GetSerialApiCapabilitiesMessages_1.GetSerialApiCapabilitiesRequest(this.driver), {
            supportCheck: false,
        });
        this._serialApiVersion = apiCaps.serialApiVersion;
        this._manufacturerId = apiCaps.manufacturerId;
        this._productType = apiCaps.productType;
        this._productId = apiCaps.productId;
        this._supportedFunctionTypes = apiCaps.supportedFunctionTypes;
        this.driver.controllerLog.print(`received API capabilities:
  serial API version:  ${this._serialApiVersion}
  manufacturer ID:     ${shared_1.num2hex(this._manufacturerId)}
  product type:        ${shared_1.num2hex(this._productType)}
  product ID:          ${shared_1.num2hex(this._productId)}
  supported functions: ${this._supportedFunctionTypes
            .map((fn) => `\n  · ${Constants_1.FunctionType[fn]} (${shared_1.num2hex(fn)})`)
            .join("")}`);
        // now we can check if a function is supported
        // find the SUC
        this.driver.controllerLog.print(`finding SUC...`);
        const suc = await this.driver.sendMessage(new GetSUCNodeIdMessages_1.GetSUCNodeIdRequest(this.driver), { supportCheck: false });
        this._sucNodeId = suc.sucNodeId;
        if (this._sucNodeId === 0) {
            this.driver.controllerLog.print(`no SUC present`);
        }
        else {
            this.driver.controllerLog.print(`SUC has node ID ${this.sucNodeId}`);
        }
        // TODO: if configured, enable this controller as SIS if there's no SUC
        // https://github.com/OpenZWave/open-zwave/blob/a46f3f36271f88eed5aea58899a6cb118ad312a2/cpp/src/Driver.cpp#L2586
        // if it's a bridge controller, request the virtual nodes
        if (this.type === ZWaveLibraryTypes_1.ZWaveLibraryTypes["Bridge Controller"] &&
            this.isFunctionSupported(Constants_1.FunctionType.FUNC_ID_ZW_GET_VIRTUAL_NODES)) {
            // TODO: send FUNC_ID_ZW_GET_VIRTUAL_NODES message
        }
        // Give the Driver time to set up the value DBs
        await initValueDBs();
        // Request information about all nodes with the GetInitData message
        this.driver.controllerLog.print(`querying node information...`);
        const initData = await this.driver.sendMessage(new GetSerialApiInitDataMessages_1.GetSerialApiInitDataRequest(this.driver));
        // override the information we might already have
        this._isSecondary = initData.isSecondary;
        this._isStaticUpdateController = initData.isStaticUpdateController;
        // and remember the new info
        this._isSlave = initData.isSlave;
        this._supportsTimers = initData.supportsTimers;
        // ignore the initVersion, no clue what to do with it
        this.driver.controllerLog.print(`received node information:
  controller role:            ${this._isSecondary ? "secondary" : "primary"}
  controller is a SUC:        ${this._isStaticUpdateController}
  controller is a slave:      ${this._isSlave}
  controller supports timers: ${this._supportsTimers}
  nodes in the network:       ${initData.nodeIds.join(", ")}`);
        // Index the value DB for optimal performance
        const valueDBIndexes = core_1.indexDBsByNode([
            this.driver.valueDB,
            this.driver.metadataDB,
        ]);
        // create an empty entry in the nodes map so we can initialize them afterwards
        for (const nodeId of initData.nodeIds) {
            this._nodes.set(nodeId, new Node_1.ZWaveNode(nodeId, this.driver, undefined, undefined, undefined, 
            // Use the previously created index to avoid doing extra work when creating the value DB
            this.createValueDBForNode(nodeId, valueDBIndexes.get(nodeId))));
        }
        // Now try to deserialize all nodes from the cache
        await restoreFromCache();
        // Set manufacturer information for the controller node
        const controllerValueDB = this._nodes.get(this._ownNodeId).valueDB;
        controllerValueDB.setMetadata(ManufacturerSpecificCC_1.getManufacturerIdValueId(), ManufacturerSpecificCC_1.getManufacturerIdValueMetadata());
        controllerValueDB.setMetadata(ManufacturerSpecificCC_1.getProductTypeValueId(), ManufacturerSpecificCC_1.getProductTypeValueMetadata());
        controllerValueDB.setMetadata(ManufacturerSpecificCC_1.getProductIdValueId(), ManufacturerSpecificCC_1.getProductIdValueMetadata());
        controllerValueDB.setValue(ManufacturerSpecificCC_1.getManufacturerIdValueId(), this._manufacturerId);
        controllerValueDB.setValue(ManufacturerSpecificCC_1.getProductTypeValueId(), this._productType);
        controllerValueDB.setValue(ManufacturerSpecificCC_1.getProductIdValueId(), this._productId);
        if (this.type !== ZWaveLibraryTypes_1.ZWaveLibraryTypes["Bridge Controller"] &&
            this.isFunctionSupported(Constants_1.FunctionType.SetSerialApiTimeouts)) {
            const { ack, byte } = this.driver.options.timeouts;
            this.driver.controllerLog.print(`setting serial API timeouts: ack = ${ack} ms, byte = ${byte} ms`);
            const resp = await this.driver.sendMessage(new SetSerialApiTimeoutsMessages_1.SetSerialApiTimeoutsRequest(this.driver, {
                ackTimeout: ack,
                byteTimeout: byte,
            }));
            this.driver.controllerLog.print(`serial API timeouts overwritten. The old values were: ack = ${resp.oldAckTimeout} ms, byte = ${resp.oldByteTimeout} ms`);
        }
        // TODO: Tell the Z-Wave stick what kind of application this is
        //   The Z-Wave Application Layer MUST use the \ref ApplicationNodeInformation
        //   function to generate the Node Information frame and to save information about
        //   node capabilities. All Z Wave application related fields of the Node Information
        //   structure MUST be initialized by this function.
        // Afterwards, a hard reset is required, so we need to move this into another method
        // if (
        // 	this.isFunctionSupported(
        // 		FunctionType.FUNC_ID_SERIAL_API_APPL_NODE_INFORMATION,
        // 	)
        // ) {
        // 	this.driver.controllerLog.print(`sending application info...`);
        // 	// TODO: Generate this list dynamically
        // 	// A list of all CCs the controller will respond to
        // 	const supportedCCs = [CommandClasses.Time];
        // 	// Turn the CCs into buffers and concat them
        // 	const supportedCCBuffer = Buffer.concat(
        // 		supportedCCs.map(cc =>
        // 			cc >= 0xf1
        // 				? // extended CC
        // 				  Buffer.from([cc >>> 8, cc & 0xff])
        // 				: // normal CC
        // 				  Buffer.from([cc]),
        // 		),
        // 	);
        // 	const appInfoMsg = new Message(this.driver, {
        // 		type: MessageType.Request,
        // 		functionType:
        // 			FunctionType.FUNC_ID_SERIAL_API_APPL_NODE_INFORMATION,
        // 		payload: Buffer.concat([
        // 			Buffer.from([
        // 				0x01, // APPLICATION_NODEINFO_LISTENING
        // 				GenericDeviceClasses["Static Controller"],
        // 				0x01, // specific static PC controller
        // 				supportedCCBuffer.length, // length of supported CC list
        // 			]),
        // 			// List of supported CCs
        // 			supportedCCBuffer,
        // 		]),
        // 	});
        // 	await this.driver.sendMessage(appInfoMsg, {
        // 		priority: MessagePriority.Controller,
        // 		supportCheck: false,
        // 	});
        // }
        this.driver.controllerLog.print("Interview completed");
    }
    createValueDBForNode(nodeId, ownKeys) {
        return new core_1.ValueDB(nodeId, this.driver.valueDB, this.driver.metadataDB, ownKeys);
    }
    /**
     * Performs a hard reset on the controller. This wipes out all configuration!
     * Warning: The driver needs to re-interview the controller, so don't call this directly
     * @internal
     */
    hardReset() {
        this.driver.controllerLog.print("performing hard reset...");
        // wotan-disable-next-line async-function-assignability
        return new Promise(async (resolve, reject) => {
            // handle the incoming message
            const handler = (_msg) => {
                this.driver.controllerLog.print(`  hard reset succeeded`);
                // Clean up
                this._nodes.forEach((node) => node.removeAllListeners());
                this._nodes.clear();
                resolve();
                return true;
            };
            this.driver.registerRequestHandler(Constants_1.FunctionType.HardReset, handler, true);
            // begin the reset process
            try {
                await this.driver.sendMessage(new HardResetRequest_1.HardResetRequest(this.driver), { supportCheck: false });
            }
            catch (e) {
                // in any case unregister the handler
                this.driver.controllerLog.print(`  hard reset failed: ${e.message}`, "error");
                this.driver.unregisterRequestHandler(Constants_1.FunctionType.HardReset, handler);
                reject(e);
            }
        });
    }
    /**
     * Starts the inclusion process of new nodes.
     * Resolves to true when the process was started, and false if the inclusion was already active.
     *
     * @param includeNonSecure Whether the node should be included non-securely, even if it supports Security. By default, all nodes will be included securely if possible
     */
    async beginInclusion(includeNonSecure = false) {
        // don't start it twice
        if (this._inclusionActive || this._exclusionActive)
            return false;
        this._inclusionActive = true;
        this._includeNonSecure = includeNonSecure;
        this.driver.controllerLog.print(`starting inclusion process...`);
        // create the promise we're going to return
        this._beginInclusionPromise = deferred_promise_1.createDeferredPromise();
        // kick off the inclusion process
        await this.driver.sendMessage(new AddNodeToNetworkRequest_1.AddNodeToNetworkRequest(this.driver, {
            addNodeType: AddNodeToNetworkRequest_1.AddNodeType.Any,
            highPower: true,
            networkWide: true,
        }));
        return this._beginInclusionPromise;
    }
    /** Is used internally to stop an active inclusion process without creating deadlocks */
    async stopInclusionInternal() {
        // don't stop it twice
        if (!this._inclusionActive)
            return;
        this._inclusionActive = false;
        this.driver.controllerLog.print(`stopping inclusion process...`);
        // create the promise we're going to return
        this._stopInclusionPromise = deferred_promise_1.createDeferredPromise();
        // kick off the inclusion process
        await this.driver.sendMessage(new AddNodeToNetworkRequest_1.AddNodeToNetworkRequest(this.driver, {
            addNodeType: AddNodeToNetworkRequest_1.AddNodeType.Stop,
            highPower: true,
            networkWide: true,
        }));
        // Don't await the promise or we create a deadlock
        void this._stopInclusionPromise.then(() => {
            this.driver.controllerLog.print(`the inclusion process was stopped`);
            this.emit("inclusion stopped");
        });
    }
    /**
     * Stops an active inclusion process. Resolves to true when the controller leaves inclusion mode,
     * and false if the inclusion was not active.
     */
    async stopInclusion() {
        // don't stop it twice
        if (!this._inclusionActive)
            return false;
        await this.stopInclusionInternal();
        return this._stopInclusionPromise;
    }
    /**
     * Starts the exclusion process of new nodes.
     * Resolves to true when the process was started,
     * and false if an inclusion or exclusion process was already active
     */
    async beginExclusion() {
        // don't start it twice
        if (this._inclusionActive || this._exclusionActive)
            return false;
        this._exclusionActive = true;
        this.driver.controllerLog.print(`starting exclusion process...`);
        // create the promise we're going to return
        this._beginInclusionPromise = deferred_promise_1.createDeferredPromise();
        // kick off the inclusion process
        await this.driver.sendMessage(new RemoveNodeFromNetworkRequest_1.RemoveNodeFromNetworkRequest(this.driver, {
            removeNodeType: RemoveNodeFromNetworkRequest_1.RemoveNodeType.Any,
            highPower: true,
            networkWide: true,
        }));
        return this._beginInclusionPromise;
    }
    /** Is used internally to stop an active inclusion process without creating deadlocks */
    async stopExclusionInternal() {
        // don't stop it twice
        if (!this._exclusionActive)
            return;
        this._exclusionActive = false;
        this.driver.controllerLog.print(`stopping exclusion process...`);
        // create the promise we're going to return
        this._stopInclusionPromise = deferred_promise_1.createDeferredPromise();
        // kick off the inclusion process
        await this.driver.sendMessage(new RemoveNodeFromNetworkRequest_1.RemoveNodeFromNetworkRequest(this.driver, {
            removeNodeType: RemoveNodeFromNetworkRequest_1.RemoveNodeType.Stop,
            highPower: true,
            networkWide: true,
        }));
        void this._stopInclusionPromise.then(() => {
            this.driver.controllerLog.print(`the exclusion process was stopped`);
            this.emit("exclusion stopped");
        });
    }
    async secureBootstrapS0(node, assumeSecure = false) {
        // If security has been set up and we are allowed to include the node securely, try to do it
        if (this.driver.securityManager &&
            (assumeSecure || node.supportsCC(core_1.CommandClasses.Security))) {
            // Only try once, otherwise the node stays unsecure
            try {
                // When replacing a node, we receive no NIF, so we cannot know that the Security CC is supported.
                // Querying the node info however kicks some devices out of secure inclusion mode.
                // Therefore we must assume that the node supports Security in order to support replacing a node securely
                if (assumeSecure && !node.supportsCC(core_1.CommandClasses.Security)) {
                    node.addCC(core_1.CommandClasses.Security, {
                        secure: true,
                        isSupported: true,
                        version: 1,
                    });
                }
                // SDS13783 - impose a 10s timeout on each message
                const api = node.commandClasses.Security.withOptions({
                    expire: 10000,
                });
                // Request security scheme, because it is required by the specs
                await api.getSecurityScheme(); // ignore the result
                // Request nonce separately, so we can impose a timeout
                await api.getNonce({
                    standalone: true,
                    storeAsFreeNonce: true,
                });
                // send the network key
                await api.setNetworkKey(this.driver.securityManager.networkKey);
                if (this._includeController) {
                    // Tell the controller which security scheme to use
                    await api.inheritSecurityScheme();
                }
                // Remember that the node is secure
                node.isSecure = true;
            }
            catch (e) {
                let errorMessage = `Security bootstrapping failed, the node is included insecurely`;
                if (!(e instanceof core_1.ZWaveError)) {
                    errorMessage += `: ${e}`;
                }
                else if (e.code === core_1.ZWaveErrorCodes.Controller_MessageExpired) {
                    errorMessage += ": a secure inclusion timer has elapsed.";
                }
                else if (e.code !== core_1.ZWaveErrorCodes.Controller_MessageDropped &&
                    e.code !== core_1.ZWaveErrorCodes.Controller_NodeTimeout) {
                    errorMessage += `: ${e.message}`;
                }
                this.driver.controllerLog.logNode(node.id, errorMessage, "warn");
                // Remember that the node is non-secure
                node.isSecure = false;
                node.removeCC(core_1.CommandClasses.Security);
            }
        }
        else {
            // Remember that the node is non-secure
            node.isSecure = false;
        }
    }
    /** Ensures that the node knows where to reach the controller */
    async bootstrapLifelineAndWakeup(node) {
        if (node.supportsCC(core_1.CommandClasses["Z-Wave Plus Info"])) {
            // SDS11846: The Z-Wave+ lifeline must be assigned to a node as the very first thing
            if (node.supportsCC(core_1.CommandClasses.Association) ||
                node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
                this.driver.controllerLog.logNode(node.id, {
                    message: `Configuring Z-Wave+ Lifeline association...`,
                    direction: "none",
                });
                const ownNodeId = this.driver.controller.ownNodeId;
                try {
                    if (node.supportsCC(core_1.CommandClasses.Association)) {
                        await node.commandClasses.Association.addNodeIds(1, ownNodeId);
                    }
                    else {
                        await node.commandClasses["Multi Channel Association"].addDestinations({
                            groupId: 1,
                            endpoints: [{ nodeId: ownNodeId, endpoint: 0 }],
                        });
                    }
                }
                catch (e) {
                    if (core_1.isTransmissionError(e)) {
                        this.driver.controllerLog.logNode(node.id, {
                            message: `Failed to configure Z-Wave+ Lifeline association: ${e.message}`,
                            direction: "none",
                            level: "warn",
                        });
                    }
                    else {
                        throw e;
                    }
                }
            }
            else {
                this.driver.controllerLog.logNode(node.id, {
                    message: `Cannot configure Z-Wave+ Lifeline association: Node does not support associations...`,
                    direction: "none",
                    level: "warn",
                });
            }
        }
        if (node.supportsCC(core_1.CommandClasses["Wake Up"])) {
            try {
                // Query the version, so we can setup the wakeup destination correctly.
                let supportedVersion;
                if (node.supportsCC(core_1.CommandClasses.Version)) {
                    supportedVersion = await node.commandClasses.Version.getCCVersion(core_1.CommandClasses["Wake Up"]);
                }
                // If querying the version can't be done, we should at least assume that it supports V1
                supportedVersion !== null && supportedVersion !== void 0 ? supportedVersion : (supportedVersion = 1);
                if (supportedVersion > 0) {
                    node.addCC(core_1.CommandClasses["Wake Up"], {
                        version: supportedVersion,
                    });
                    const instance = node.createCCInstance(core_1.CommandClasses["Wake Up"]);
                    await instance.interview();
                }
            }
            catch (e) {
                if (core_1.isTransmissionError(e)) {
                    this.driver.controllerLog.logNode(node.id, {
                        message: `Cannot configure wakeup destination: ${e.message}`,
                        direction: "none",
                        level: "warn",
                    });
                }
                else {
                    // we want to pass all other errors through
                    throw e;
                }
            }
        }
    }
    /**
     * Stops an active exclusion process. Resolves to true when the controller leaves exclusion mode,
     * and false if the inclusion was not active.
     */
    async stopExclusion() {
        // don't stop it twice
        if (!this._exclusionActive)
            return false;
        await this.stopExclusionInternal();
        return this._stopInclusionPromise;
    }
    /**
     * Is called when an AddNode request is received from the controller.
     * Handles and controls the inclusion process.
     */
    async handleAddNodeRequest(msg) {
        var _a, _b, _c;
        this.driver.controllerLog.print(`handling add node request (status = ${AddNodeToNetworkRequest_1.AddNodeStatus[msg.status]})`);
        if (!this._inclusionActive && msg.status !== AddNodeToNetworkRequest_1.AddNodeStatus.Done) {
            this.driver.controllerLog.print(`  inclusion is NOT active, ignoring it...`);
            return true; // Don't invoke any more handlers
        }
        switch (msg.status) {
            case AddNodeToNetworkRequest_1.AddNodeStatus.Ready:
                // this is called when inclusion was started successfully
                this.driver.controllerLog.print(`  the controller is now ready to add nodes`);
                if (this._beginInclusionPromise != null) {
                    this._beginInclusionPromise.resolve(true);
                    this.emit("inclusion started", !this._includeNonSecure);
                }
                break;
            case AddNodeToNetworkRequest_1.AddNodeStatus.Failed:
                // this is called when inclusion could not be started...
                if (this._beginInclusionPromise != null) {
                    this.driver.controllerLog.print(`  starting the inclusion failed`, "error");
                    this._beginInclusionPromise.reject(new core_1.ZWaveError("The inclusion could not be started.", core_1.ZWaveErrorCodes.Controller_InclusionFailed));
                }
                else {
                    // ...or adding a node failed
                    this.driver.controllerLog.print(`  adding the node failed`, "error");
                    this.emit("inclusion failed");
                }
                // in any case, stop the inclusion process so we don't accidentally add another node
                try {
                    await this.stopInclusionInternal();
                }
                catch (_d) {
                    /* ok */
                }
                break;
            case AddNodeToNetworkRequest_1.AddNodeStatus.AddingController:
                this._includeController = true;
            // fall through!
            case AddNodeToNetworkRequest_1.AddNodeStatus.AddingSlave: {
                // this is called when a new node is added
                this._nodePendingInclusion = new Node_1.ZWaveNode(msg.statusContext.nodeId, this.driver, new DeviceClass_1.DeviceClass(this.driver.configManager, msg.statusContext.basic, msg.statusContext.generic, msg.statusContext.specific), msg.statusContext.supportedCCs, msg.statusContext.controlledCCs, 
                // Create an empty value DB and specify that it contains no values
                // to avoid indexing the existing values
                this.createValueDBForNode(msg.statusContext.nodeId, new Set()));
                // TODO: According to INS13954-7, there are several more steps and different timeouts when including a controller
                // For now do the absolute minimum - that is include the controller
                return true; // Don't invoke any more handlers
            }
            case AddNodeToNetworkRequest_1.AddNodeStatus.ProtocolDone: {
                // this is called after a new node is added
                // stop the inclusion process so we don't accidentally add another node
                try {
                    await this.stopInclusionInternal();
                }
                catch (_e) {
                    /* ok */
                }
                break;
            }
            case AddNodeToNetworkRequest_1.AddNodeStatus.Done: {
                // this is called when the inclusion was completed
                this.driver.controllerLog.print(`done called for ${msg.statusContext.nodeId}`);
                // stopping the inclusion was acknowledged by the controller
                if (this._stopInclusionPromise != null)
                    this._stopInclusionPromise.resolve(true);
                if (this._nodePendingInclusion != null) {
                    const newNode = this._nodePendingInclusion;
                    const supportedCommandClasses = [
                        ...newNode.implementedCommandClasses.entries(),
                    ]
                        .filter(([, info]) => info.isSupported)
                        .map(([cc]) => cc);
                    const controlledCommandClasses = [
                        ...newNode.implementedCommandClasses.entries(),
                    ]
                        .filter(([, info]) => info.isControlled)
                        .map(([cc]) => cc);
                    this.driver.controllerLog.print(`finished adding node ${newNode.id}:
  basic device class:    ${(_a = newNode.deviceClass) === null || _a === void 0 ? void 0 : _a.basic.label}
  generic device class:  ${(_b = newNode.deviceClass) === null || _b === void 0 ? void 0 : _b.generic.label}
  specific device class: ${(_c = newNode.deviceClass) === null || _c === void 0 ? void 0 : _c.specific.label}
  supported CCs: ${supportedCommandClasses
                        .map((cc) => `\n  · ${core_1.CommandClasses[cc]} (${shared_1.num2hex(cc)})`)
                        .join("")}
  controlled CCs: ${controlledCommandClasses
                        .map((cc) => `\n  · ${core_1.CommandClasses[cc]} (${shared_1.num2hex(cc)})`)
                        .join("")}`);
                    // remember the node
                    this._nodes.set(newNode.id, newNode);
                    this._nodePendingInclusion = undefined;
                    // We're communicating with the device, so assume it is alive
                    // If it is actually a sleeping device, it will be marked as such later
                    newNode.markAsAlive();
                    // Assign return route to make sure the node's responses reach us
                    try {
                        this.driver.controllerLog.logNode(newNode.id, {
                            message: `Assigning return route to controller...`,
                            direction: "outbound",
                        });
                        await this.assignReturnRoute(newNode.id, this._ownNodeId);
                    }
                    catch (e) {
                        this.driver.controllerLog.logNode(newNode.id, `assigning return route failed: ${e.message}`, "warn");
                    }
                    if (!this._includeNonSecure) {
                        await this.secureBootstrapS0(newNode);
                    }
                    this._includeController = false;
                    // Bootstrap the node's lifelines, so it knows where the controller is
                    await this.bootstrapLifelineAndWakeup(newNode);
                    // We're done adding this node, notify listeners
                    this.emit("node added", newNode);
                }
                break;
            }
            default:
                // not sure what to do with this message
                return false;
        }
        return true; // Don't invoke any more handlers
    }
    /**
     * Is called when an ReplaceFailed request is received from the controller.
     * Handles and controls the replace process.
     */
    async handleReplaceNodeRequest(msg) {
        var _a, _b, _c;
        this.driver.controllerLog.print(`handling replace node request (status = ${ReplaceFailedNodeRequest_1.ReplaceFailedNodeStatus[msg.replaceStatus]})`);
        switch (msg.replaceStatus) {
            case ReplaceFailedNodeRequest_1.ReplaceFailedNodeStatus.NodeOK:
                (_a = this._replaceFailedPromise) === null || _a === void 0 ? void 0 : _a.reject(new core_1.ZWaveError(`The node could not be replaced because it has responded`, core_1.ZWaveErrorCodes.ReplaceFailedNode_NodeOK));
                break;
            case ReplaceFailedNodeRequest_1.ReplaceFailedNodeStatus.FailedNodeReplaceFailed:
                (_b = this._replaceFailedPromise) === null || _b === void 0 ? void 0 : _b.reject(new core_1.ZWaveError(`The failed node has not been replaced`, core_1.ZWaveErrorCodes.ReplaceFailedNode_Failed));
                break;
            case ReplaceFailedNodeRequest_1.ReplaceFailedNodeStatus.FailedNodeReplace:
                // failed node is now ready to be replaced and controller is ready to add a new
                // node with the nodeID of the failed node
                this.driver.controllerLog.print(`The failed node is ready to be replaced, inclusion started...`);
                this.emit("inclusion started", !this._includeNonSecure);
                this._inclusionActive = true;
                (_c = this._replaceFailedPromise) === null || _c === void 0 ? void 0 : _c.resolve(true);
                // stop here, don't emit inclusion failed
                return true;
            case ReplaceFailedNodeRequest_1.ReplaceFailedNodeStatus.FailedNodeReplaceDone:
                this.driver.controllerLog.print(`The failed node was replaced`);
                this.emit("inclusion stopped");
                if (this._nodePendingReplace) {
                    this.emit("node removed", this._nodePendingReplace, true);
                    this._nodes.delete(this._nodePendingReplace.id);
                    // Create a fresh node instance and forget the old one
                    const newNode = new Node_1.ZWaveNode(this._nodePendingReplace.id, this.driver, undefined, undefined, undefined, 
                    // Create an empty value DB and specify that it contains no values
                    // to avoid indexing the existing values
                    this.createValueDBForNode(this._nodePendingReplace.id, new Set()));
                    this._nodePendingReplace = undefined;
                    this._nodes.set(newNode.id, newNode);
                    // We're communicating with the device, so assume it is alive
                    // If it is actually a sleeping device, it will be marked as such later
                    newNode.markAsAlive();
                    // Assign return route to make sure the node's responses reach us
                    try {
                        this.driver.controllerLog.logNode(newNode.id, {
                            message: `Assigning return route to controller...`,
                            direction: "outbound",
                        });
                        await this.assignReturnRoute(newNode.id, this._ownNodeId);
                    }
                    catch (e) {
                        this.driver.controllerLog.logNode(newNode.id, `assigning return route failed: ${e.message}`, "warn");
                    }
                    // Try perform the security bootstrap process
                    if (!this._includeNonSecure) {
                        await this.secureBootstrapS0(newNode, true);
                    }
                    // Bootstrap the node's lifelines, so it knows where the controller is
                    await this.bootstrapLifelineAndWakeup(newNode);
                    // We're done adding this node, notify listeners. This also kicks off the node interview
                    this.emit("node added", newNode);
                }
                // stop here, don't emit inclusion failed
                return true;
        }
        this.emit("inclusion failed");
        return false; // Don't invoke any more handlers
    }
    /**
     * Is called when a RemoveNode request is received from the controller.
     * Handles and controls the exclusion process.
     */
    async handleRemoveNodeRequest(msg) {
        this.driver.controllerLog.print(`handling remove node request (status = ${RemoveNodeFromNetworkRequest_1.RemoveNodeStatus[msg.status]})`);
        if (!this._exclusionActive && msg.status !== RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.Done) {
            this.driver.controllerLog.print(`  exclusion is NOT active, ignoring it...`);
            return true; // Don't invoke any more handlers
        }
        switch (msg.status) {
            case RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.Ready:
                // this is called when inclusion was started successfully
                this.driver.controllerLog.print(`  the controller is now ready to remove nodes`);
                if (this._beginInclusionPromise != null) {
                    this._beginInclusionPromise.resolve(true);
                    this.emit("exclusion started");
                }
                break;
            case RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.Failed:
                // this is called when inclusion could not be started...
                if (this._beginInclusionPromise != null) {
                    this.driver.controllerLog.print(`  starting the exclusion failed`, "error");
                    this._beginInclusionPromise.reject(new core_1.ZWaveError("The exclusion could not be started.", core_1.ZWaveErrorCodes.Controller_ExclusionFailed));
                }
                else {
                    // ...or removing a node failed
                    this.driver.controllerLog.print(`  removing the node failed`, "error");
                    this.emit("exclusion failed");
                }
                // in any case, stop the exclusion process so we don't accidentally remove another node
                try {
                    await this.stopExclusionInternal();
                }
                catch (_a) {
                    /* ok */
                }
                break;
            case RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.RemovingSlave:
            case RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.RemovingController: {
                // this is called when a node is removed
                this._nodePendingExclusion = this.nodes.get(msg.statusContext.nodeId);
                return true; // Don't invoke any more handlers
            }
            case RemoveNodeFromNetworkRequest_1.RemoveNodeStatus.Done: {
                // this is called when the exclusion was completed
                // stop the exclusion process so we don't accidentally remove another node
                try {
                    await this.stopExclusionInternal();
                }
                catch (_b) {
                    /* ok */
                }
                // stopping the inclusion was acknowledged by the controller
                if (this._stopInclusionPromise != null)
                    this._stopInclusionPromise.resolve(true);
                if (this._nodePendingExclusion != null) {
                    this.driver.controllerLog.print(`Node ${this._nodePendingExclusion.id} was removed`);
                    // notify listeners
                    this.emit("node removed", this._nodePendingExclusion, false);
                    // and forget the node
                    this._nodes.delete(this._nodePendingExclusion.id);
                    this._nodePendingExclusion = undefined;
                }
                break;
            }
            default:
                // not sure what to do with this message
                return false;
        }
        return true; // Don't invoke any more handlers
    }
    /**
     * Requests all alive slave nodes to update their neighbor lists
     */
    beginHealingNetwork() {
        // Don't start the process twice
        if (this._healNetworkActive)
            return false;
        this._healNetworkActive = true;
        this.driver.controllerLog.print(`starting network heal...`);
        // Reset all nodes to "not healed"
        this._healNetworkProgress.clear();
        for (const [id, node] of this._nodes) {
            if (id === this._ownNodeId)
                continue;
            if (
            // The node is known to be dead
            node.status === Types_1.NodeStatus.Dead ||
                // The node is assumed asleep but has never been interviewed.
                // It is most likely dead
                (node.status === Types_1.NodeStatus.Asleep &&
                    node.interviewStage === Types_1.InterviewStage.ProtocolInfo)) {
                // Don't interview dead nodes
                this.driver.controllerLog.logNode(id, `Skipping heal because the node is not responding.`);
                this._healNetworkProgress.set(id, "skipped");
            }
            else {
                this._healNetworkProgress.set(id, "pending");
            }
        }
        // Do the heal process in the background
        void (async () => {
            const tasks = [...this._healNetworkProgress]
                .filter(([, status]) => status === "pending")
                .map(async ([nodeId]) => {
                // await the heal process for each node and treat errors as a non-successful heal
                const result = await this.healNode(nodeId).catch(() => false);
                if (!this._healNetworkActive)
                    return;
                // Track the success in a map
                this._healNetworkProgress.set(nodeId, result ? "done" : "failed");
                // Notify listeners about the progress
                this.emit("heal network progress", new Map(this._healNetworkProgress));
            });
            await Promise.all(tasks);
            // Only emit the done event when the process wasn't stopped in the meantime
            if (this._healNetworkActive) {
                this.emit("heal network done", new Map(this._healNetworkProgress));
            }
            // We're done!
            this._healNetworkActive = false;
        })();
        // And update the progress once at the start
        this.emit("heal network progress", new Map(this._healNetworkProgress));
        return true;
    }
    /**
     * Stops an network healing process. Resolves false if the process was not active, true otherwise.
     */
    stopHealingNetwork() {
        // don't stop it twice
        if (!this._healNetworkActive)
            return false;
        this._healNetworkActive = false;
        this.driver.controllerLog.print(`stopping network heal...`);
        // Cancel all transactions that were created by the healing process
        this.driver.rejectTransactions((t) => t.message instanceof RequestNodeNeighborUpdateMessages_1.RequestNodeNeighborUpdateRequest ||
            t.message instanceof GetRoutingInfoMessages_1.GetRoutingInfoRequest ||
            t.message instanceof DeleteReturnRouteMessages_1.DeleteReturnRouteRequest ||
            t.message instanceof AssignReturnRouteMessages_1.AssignReturnRouteRequest);
        return true;
    }
    /**
     * Performs the healing process for a node
     */
    async healNode(nodeId) {
        // The healing process consists of four steps
        // Each step is tried up to 5 times before the healing process is considered failed
        const maxAttempts = 5;
        // 1. command the node to refresh its neighbor list
        for (let attempt = 1; attempt <= maxAttempts; attempt++) {
            // If the process was stopped in the meantime, cancel
            if (!this._healNetworkActive)
                return false;
            this.driver.controllerLog.logNode(nodeId, {
                message: `refreshing neighbor list (attempt ${attempt})...`,
                direction: "outbound",
            });
            try {
                const resp = await this.driver.sendMessage(new RequestNodeNeighborUpdateMessages_1.RequestNodeNeighborUpdateRequest(this.driver, {
                    nodeId,
                }));
                if (resp.updateStatus === RequestNodeNeighborUpdateMessages_1.NodeNeighborUpdateStatus.UpdateDone) {
                    this.driver.controllerLog.logNode(nodeId, {
                        message: "neighbor list refreshed...",
                        direction: "inbound",
                    });
                    // this step was successful, continue with the next
                    break;
                }
                else {
                    // UpdateFailed
                    this.driver.controllerLog.logNode(nodeId, {
                        message: "refreshing neighbor list failed...",
                        direction: "inbound",
                        level: "warn",
                    });
                }
            }
            catch (e) {
                this.driver.controllerLog.logNode(nodeId, `refreshing neighbor list failed: ${e.message}`, "warn");
            }
            if (attempt === maxAttempts) {
                this.driver.controllerLog.logNode(nodeId, {
                    message: `failed to update the neighbor list after ${maxAttempts} attempts, healing failed`,
                    level: "warn",
                    direction: "none",
                });
                return false;
            }
        }
        // 2. retrieve the updated list
        for (let attempt = 1; attempt <= maxAttempts; attempt++) {
            // If the process was stopped in the meantime, cancel
            if (!this._healNetworkActive)
                return false;
            this.driver.controllerLog.logNode(nodeId, {
                message: `retrieving updated neighbor list (attempt ${attempt})...`,
                direction: "outbound",
            });
            try {
                // Retrieve the updated list from the node
                await this.nodes.get(nodeId).queryNeighborsInternal();
                break;
            }
            catch (e) {
                this.driver.controllerLog.logNode(nodeId, `retrieving the updated neighbor list failed: ${e.message}`, "warn");
            }
            if (attempt === maxAttempts) {
                this.driver.controllerLog.logNode(nodeId, {
                    message: `failed to retrieve the updated neighbor list after ${maxAttempts} attempts, healing failed`,
                    level: "warn",
                    direction: "none",
                });
                return false;
            }
        }
        // 3. delete all return routes so we can assign new ones
        for (let attempt = 1; attempt <= maxAttempts; attempt++) {
            this.driver.controllerLog.logNode(nodeId, {
                message: `deleting return routes (attempt ${attempt})...`,
                direction: "outbound",
            });
            try {
                await this.driver.sendMessage(new DeleteReturnRouteMessages_1.DeleteReturnRouteRequest(this.driver, { nodeId }));
                // this step was successful, continue with the next
                break;
            }
            catch (e) {
                this.driver.controllerLog.logNode(nodeId, `deleting return routes failed: ${e.message}`, "warn");
            }
            if (attempt === maxAttempts) {
                this.driver.controllerLog.logNode(nodeId, {
                    message: `failed to delete return routes after ${maxAttempts} attempts, healing failed`,
                    level: "warn",
                    direction: "none",
                });
                return false;
            }
        }
        // 4. Assign up to 4 return routes for associations, one of which should be the controller
        let associatedNodes = [];
        const maxReturnRoutes = 4;
        try {
            associatedNodes = arrays_1.distinct(shared_1.flatMap([...this.getAssociations(nodeId).values()], (assocs) => assocs.map((a) => a.nodeId))).sort();
        }
        catch (_a) {
            /* ignore */
        }
        // Always include ourselves first
        if (!associatedNodes.includes(this._ownNodeId)) {
            associatedNodes.unshift(this._ownNodeId);
        }
        if (associatedNodes.length > maxReturnRoutes) {
            associatedNodes = associatedNodes.slice(0, maxReturnRoutes);
        }
        this.driver.controllerLog.logNode(nodeId, {
            message: `assigning return routes to the following nodes:
${associatedNodes.join(", ")}`,
            direction: "outbound",
        });
        for (const destinationNodeId of associatedNodes) {
            for (let attempt = 1; attempt <= maxAttempts; attempt++) {
                this.driver.controllerLog.logNode(nodeId, {
                    message: `assigning return route to node ${destinationNodeId} (attempt ${attempt})...`,
                    direction: "outbound",
                });
                try {
                    await this.driver.sendMessage(new AssignReturnRouteMessages_1.AssignReturnRouteRequest(this.driver, {
                        nodeId,
                        destinationNodeId,
                    }));
                    // this step was successful, continue with the next
                    break;
                }
                catch (e) {
                    this.driver.controllerLog.logNode(nodeId, `assigning return route failed: ${e.message}`, "warn");
                }
                if (attempt === maxAttempts) {
                    this.driver.controllerLog.logNode(nodeId, {
                        message: `failed to assign return route after ${maxAttempts} attempts, healing failed`,
                        level: "warn",
                        direction: "none",
                    });
                    return false;
                }
            }
        }
        return true;
    }
    async assignReturnRoute(nodeId, destinationNodeId) {
        await this.driver.sendMessage(new AssignReturnRouteMessages_1.AssignReturnRouteRequest(this.driver, {
            nodeId,
            destinationNodeId,
        }));
    }
    /**
     * Returns a dictionary of all association groups of this node and their information.
     * This only works AFTER the interview process
     */
    getAssociationGroups(nodeId) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const node = this.nodes.getOrThrow(nodeId);
        // Check whether we have multi channel support or not
        let assocInstance;
        let mcInstance;
        if (node.supportsCC(core_1.CommandClasses.Association)) {
            assocInstance = node.createCCInstanceUnsafe(core_1.CommandClasses.Association);
        }
        else {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        if (node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
            mcInstance = node.createCCInstanceUnsafe(core_1.CommandClasses["Multi Channel Association"]);
        }
        const assocGroupCount = (_a = assocInstance.getGroupCountCached()) !== null && _a !== void 0 ? _a : 0;
        const mcGroupCount = (_b = mcInstance === null || mcInstance === void 0 ? void 0 : mcInstance.getGroupCountCached()) !== null && _b !== void 0 ? _b : 0;
        const groupCount = Math.max(assocGroupCount, mcGroupCount);
        const ret = new Map();
        if (node.supportsCC(core_1.CommandClasses["Association Group Information"])) {
            // We can read all information we need from the AGI CC
            const agiInstance = node.createCCInstance(core_1.CommandClasses["Association Group Information"]);
            for (let group = 1; group <= groupCount; group++) {
                const assocConfig = (_d = (_c = node.deviceConfig) === null || _c === void 0 ? void 0 : _c.associations) === null || _d === void 0 ? void 0 : _d.get(group);
                const multiChannel = !!mcInstance && group <= mcGroupCount;
                ret.set(group, {
                    maxNodes: (multiChannel
                        ? mcInstance
                        : assocInstance).getMaxNodesCached(group) || 1,
                    // AGI implies Z-Wave+ where group 1 is the lifeline
                    isLifeline: group === 1,
                    label: 
                    // prefer the configured label if we have one
                    (_f = (_e = assocConfig === null || assocConfig === void 0 ? void 0 : assocConfig.label) !== null && _e !== void 0 ? _e : 
                    // the ones reported by AGI are sometimes pretty bad
                    agiInstance.getGroupNameCached(group)) !== null && _f !== void 0 ? _f : 
                    // but still better than "unnamed"
                    `Unnamed group ${group}`,
                    multiChannel,
                    profile: agiInstance.getGroupProfileCached(group),
                    issuedCommands: agiInstance.getIssuedCommandsCached(group),
                });
            }
        }
        else {
            // we need to consult the device config
            for (let group = 1; group <= groupCount; group++) {
                const assocConfig = (_h = (_g = node.deviceConfig) === null || _g === void 0 ? void 0 : _g.associations) === null || _h === void 0 ? void 0 : _h.get(group);
                const multiChannel = !!mcInstance && group <= mcGroupCount;
                ret.set(group, {
                    maxNodes: (multiChannel
                        ? mcInstance
                        : assocInstance).getMaxNodesCached(group) ||
                        (assocConfig === null || assocConfig === void 0 ? void 0 : assocConfig.maxNodes) ||
                        1,
                    isLifeline: (_j = assocConfig === null || assocConfig === void 0 ? void 0 : assocConfig.isLifeline) !== null && _j !== void 0 ? _j : group === 1,
                    label: (_k = assocConfig === null || assocConfig === void 0 ? void 0 : assocConfig.label) !== null && _k !== void 0 ? _k : `Unnamed group ${group}`,
                    multiChannel,
                });
            }
        }
        return ret;
    }
    /** Returns all Associations (Multi Channel or normal) that are configured on a node */
    getAssociations(nodeId) {
        const node = this.nodes.getOrThrow(nodeId);
        const ret = new Map();
        if (node.supportsCC(core_1.CommandClasses.Association)) {
            const cc = node.createCCInstanceUnsafe(core_1.CommandClasses.Association);
            const destinations = cc.getAllDestinationsCached();
            for (const [groupId, assocs] of destinations) {
                ret.set(groupId, assocs);
            }
        }
        else {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        // Merge the "normal" destinations with multi channel destinations
        if (node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
            const cc = node.createCCInstanceUnsafe(core_1.CommandClasses["Multi Channel Association"]);
            const destinations = cc.getAllDestinationsCached();
            for (const [groupId, assocs] of destinations) {
                if (ret.has(groupId)) {
                    const normalAssociations = ret.get(groupId);
                    ret.set(groupId, [
                        ...normalAssociations,
                        // Eliminate potential duplicates
                        ...assocs.filter((a1) => normalAssociations.findIndex((a2) => a1.nodeId === a2.nodeId &&
                            a1.endpoint === a2.endpoint) === -1),
                    ]);
                }
                else {
                    ret.set(groupId, assocs);
                }
            }
        }
        return ret;
    }
    /** Checks if a given association is allowed */
    isAssociationAllowed(nodeId, group, association) {
        var _a;
        const node = this.nodes.getOrThrow(nodeId);
        const targetNode = this.nodes.getOrThrow(association.nodeId);
        const targetEndpoint = targetNode.getEndpoint((_a = association.endpoint) !== null && _a !== void 0 ? _a : 0);
        if (!targetEndpoint) {
            throw new core_1.ZWaveError(`The endpoint ${association.endpoint} was not found on node ${association.nodeId}!`, core_1.ZWaveErrorCodes.Controller_EndpointNotFound);
        }
        // SDS14223:
        // A controlling node MUST NOT associate Node A to a Node B destination that does not support
        // the Command Class that the Node A will be controlling
        //
        // To determine this, the node must support the AGI CC or we have no way of knowing which
        // CCs the node will control
        if (!node.supportsCC(core_1.CommandClasses.Association) &&
            !node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        else if (!node.supportsCC(core_1.CommandClasses["Association Group Information"])) {
            return true;
        }
        // The following checks don't apply to Lifeline associations
        if (association.nodeId === this._ownNodeId)
            return true;
        const groupCommandList = node
            .createCCInstanceInternal(core_1.CommandClasses["Association Group Information"])
            .getIssuedCommandsCached(group);
        if (!groupCommandList || !groupCommandList.size) {
            // We don't know which CCs this group controls, just allow it
            return true;
        }
        const groupCCs = [...groupCommandList.keys()];
        // A controlling node MAY create an association to a destination supporting an
        // actuator Command Class if the actual association group sends Basic Control Command Class.
        if (groupCCs.includes(core_1.CommandClasses.Basic) &&
            core_1.actuatorCCs.some((cc) => targetEndpoint.supportsCC(cc))) {
            return true;
        }
        // Enforce that at least one issued CC is supported
        return groupCCs.some((cc) => targetEndpoint.supportsCC(cc));
    }
    /**
     * Adds associations to a node
     */
    async addAssociations(nodeId, group, associations) {
        var _a, _b, _c, _d, _e;
        const node = this.nodes.getOrThrow(nodeId);
        // Check whether we should add any associations the device does not have support for
        let assocInstance;
        let mcInstance;
        // Split associations into conventional and endpoint associations
        const nodeAssociations = arrays_1.distinct(associations
            .filter((a) => a.endpoint == undefined)
            .map((a) => a.nodeId));
        const endpointAssociations = associations.filter((a) => a.endpoint != undefined);
        if (node.supportsCC(core_1.CommandClasses.Association)) {
            assocInstance = node.createCCInstanceUnsafe(core_1.CommandClasses.Association);
        }
        else if (nodeAssociations.length > 0) {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        if (node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
            mcInstance = node.createCCInstanceUnsafe(core_1.CommandClasses["Multi Channel Association"]);
        }
        else if (endpointAssociations.length > 0) {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support multi channel associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        const assocGroupCount = (_a = assocInstance === null || assocInstance === void 0 ? void 0 : assocInstance.getGroupCountCached()) !== null && _a !== void 0 ? _a : 0;
        const mcGroupCount = (_b = mcInstance === null || mcInstance === void 0 ? void 0 : mcInstance.getGroupCountCached()) !== null && _b !== void 0 ? _b : 0;
        const groupCount = Math.max(assocGroupCount, mcGroupCount);
        if (group > groupCount) {
            throw new core_1.ZWaveError(`Group ${group} does not exist on node ${nodeId}`, core_1.ZWaveErrorCodes.AssociationCC_InvalidGroup);
        }
        const groupIsMultiChannel = !!mcInstance &&
            group <= mcGroupCount &&
            !((_e = (_d = (_c = node.deviceConfig) === null || _c === void 0 ? void 0 : _c.associations) === null || _d === void 0 ? void 0 : _d.get(group)) === null || _e === void 0 ? void 0 : _e.noEndpoint);
        if (groupIsMultiChannel) {
            // Check that all associations are allowed
            const disallowedAssociations = associations.filter((a) => !this.isAssociationAllowed(nodeId, group, a));
            if (disallowedAssociations.length) {
                let message = `The following associations are not allowed:`;
                message += disallowedAssociations
                    .map((a) => `\n· Node ${a.nodeId}${a.endpoint ? `, endpoint ${a.endpoint}` : ""}`)
                    .join("");
                throw new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.AssociationCC_NotAllowed);
            }
            // And add them
            await node.commandClasses["Multi Channel Association"].addDestinations({
                groupId: group,
                nodeIds: nodeAssociations,
                endpoints: endpointAssociations,
            });
            // Refresh the association list
            await node.commandClasses["Multi Channel Association"].getGroup(group);
        }
        else {
            // Although the node supports multi channel associations, this group only supports "normal" associations
            if (associations.some((a) => a.endpoint != undefined)) {
                throw new core_1.ZWaveError(`Node ${nodeId}, group ${group} does not support multi channel associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
            }
            // Check that all associations are allowed
            const disallowedAssociations = associations.filter((a) => !this.isAssociationAllowed(nodeId, group, a));
            if (disallowedAssociations.length) {
                throw new core_1.ZWaveError(`The associations to the following nodes are not allowed: ${disallowedAssociations
                    .map((a) => a.nodeId)
                    .join(", ")}`, core_1.ZWaveErrorCodes.AssociationCC_NotAllowed);
            }
            await node.commandClasses.Association.addNodeIds(group, ...associations.map((a) => a.nodeId));
            // Refresh the association list
            await node.commandClasses.Association.getGroup(group);
        }
    }
    /**
     * Removes the specific associations from a node
     */
    async removeAssociations(nodeId, group, associations) {
        const node = this.nodes.getOrThrow(nodeId);
        let groupExistsAsMultiChannel = false;
        // Split associations into conventional and endpoint associations
        const nodeAssociations = arrays_1.distinct(associations
            .filter((a) => a.endpoint == undefined)
            .map((a) => a.nodeId));
        const endpointAssociations = associations.filter((a) => a.endpoint != undefined);
        // Removing associations is not either/or - we could have a device with duplicated associations between
        // Association CC and Multi Channel Association CC
        if (node.supportsCC(core_1.CommandClasses["Multi Channel Association"])) {
            // Prefer multi channel associations
            const cc = node.createCCInstanceUnsafe(core_1.CommandClasses["Multi Channel Association"]);
            if (group > cc.getGroupCountCached()) {
                throw new core_1.ZWaveError(`Group ${group} does not exist on node ${nodeId}`, core_1.ZWaveErrorCodes.AssociationCC_InvalidGroup);
            }
            else {
                // Remember that the group exists as a multi channel group, otherwise the "normal" association code
                // will throw if we try to remove the association from a non-existing "normal" group
                groupExistsAsMultiChannel = true;
            }
            await node.commandClasses["Multi Channel Association"].removeDestinations({
                groupId: group,
                nodeIds: nodeAssociations,
                endpoints: endpointAssociations,
            });
            // Refresh the multi channel association list
            await node.commandClasses["Multi Channel Association"].getGroup(group);
        }
        else if (endpointAssociations.length > 0) {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support multi channel associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
        if (node.supportsCC(core_1.CommandClasses.Association)) {
            // Use normal associations as a fallback
            const cc = node.createCCInstanceUnsafe(core_1.CommandClasses.Association);
            if (group > cc.getGroupCountCached()) {
                // Don't throw if the group existed as multi channel - this branch is only a fallback
                if (groupExistsAsMultiChannel)
                    return;
                throw new core_1.ZWaveError(`Group ${group} does not exist on node ${nodeId}`, core_1.ZWaveErrorCodes.AssociationCC_InvalidGroup);
            }
            // Remove the remaining node associations
            await node.commandClasses.Association.removeNodeIds({
                groupId: group,
                nodeIds: nodeAssociations,
            });
            // Refresh the association list
            await node.commandClasses.Association.getGroup(group);
        }
        else if (nodeAssociations.length > 0) {
            throw new core_1.ZWaveError(`Node ${nodeId} does not support associations!`, core_1.ZWaveErrorCodes.CC_NotSupported);
        }
    }
    /**
     * Removes a node from all other nodes' associations
     * WARNING: It is not recommended to await this method
     */
    async removeNodeFromAllAssocations(nodeId) {
        // Create all async tasks
        const tasks = [...this.nodes.values()]
            .filter((node) => node.id !== this._ownNodeId && node.id !== nodeId)
            .map((node) => {
            // Prefer multi channel associations if that is available
            if (node.commandClasses["Multi Channel Association"].isSupported()) {
                return node.commandClasses["Multi Channel Association"].removeDestinations({
                    nodeIds: [nodeId],
                });
            }
            else if (node.commandClasses.Association.isSupported()) {
                return node.commandClasses.Association.removeNodeIdsFromAllGroups([nodeId]);
            }
        })
            .filter((task) => !!task);
        await Promise.all(tasks);
    }
    /**
     * Tests if a node is marked as failed in the controller's memory
     * @param nodeId The id of the node in question
     */
    async isFailedNode(nodeId) {
        const result = await this.driver.sendMessage(new IsFailedNodeMessages_1.IsFailedNodeRequest(this.driver, { failedNodeId: nodeId }));
        return result.result;
    }
    /**
     * Removes a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to remove
     */
    async removeFailedNode(nodeId) {
        const node = this.nodes.getOrThrow(nodeId);
        if (await node.ping()) {
            throw new core_1.ZWaveError(`The node removal process could not be started because the node responded to a ping.`, core_1.ZWaveErrorCodes.ReplaceFailedNode_Failed);
        }
        const result = await this.driver.sendMessage(new RemoveFailedNodeMessages_1.RemoveFailedNodeRequest(this.driver, { failedNodeId: nodeId }));
        if (result instanceof RemoveFailedNodeMessages_1.RemoveFailedNodeResponse) {
            // This implicates that the process was unsuccessful.
            let message = `The node removal process could not be started due to the following reasons:`;
            if (!!(result.removeStatus &
                RemoveFailedNodeMessages_1.RemoveFailedNodeStartFlags.NotPrimaryController)) {
                message += "\n· This controller is not the primary controller";
            }
            if (!!(result.removeStatus &
                RemoveFailedNodeMessages_1.RemoveFailedNodeStartFlags.NodeNotFound)) {
                message += `\n· Node ${nodeId} is not in the list of failed nodes`;
            }
            if (!!(result.removeStatus &
                RemoveFailedNodeMessages_1.RemoveFailedNodeStartFlags.RemoveProcessBusy)) {
                message += `\n· The node removal process is currently busy`;
            }
            if (!!(result.removeStatus &
                RemoveFailedNodeMessages_1.RemoveFailedNodeStartFlags.RemoveFailed)) {
                message += `\n· The controller is busy or the node has responded`;
            }
            throw new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.RemoveFailedNode_Failed);
        }
        else {
            switch (result.removeStatus) {
                case RemoveFailedNodeMessages_1.RemoveFailedNodeStatus.NodeOK:
                    throw new core_1.ZWaveError(`The node could not be removed because it has responded`, core_1.ZWaveErrorCodes.RemoveFailedNode_NodeOK);
                case RemoveFailedNodeMessages_1.RemoveFailedNodeStatus.NodeNotRemoved:
                    throw new core_1.ZWaveError(`The removal process could not be completed`, core_1.ZWaveErrorCodes.RemoveFailedNode_Failed);
                default:
                    // If everything went well, the status is RemoveFailedNodeStatus.NodeRemoved
                    // Emit the removed event so the driver and applications can react
                    this.emit("node removed", this.nodes.get(nodeId), false);
                    // and forget the node
                    this._nodes.delete(nodeId);
                    return;
            }
        }
    }
    /**
     * Replace a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to replace
     * @param includeNonSecure Whether the new node should be included non-securely, even if it supports Security. By default, all nodes will be included securely if possible
     *
     */
    async replaceFailedNode(nodeId, includeNonSecure = false) {
        // don't start it twice
        if (this._inclusionActive || this._exclusionActive)
            return false;
        this.driver.controllerLog.print(`starting replace failed node process...`);
        const node = this.nodes.getOrThrow(nodeId);
        if (await node.ping()) {
            throw new core_1.ZWaveError(`The node replace process could not be started because the node responded to a ping.`, core_1.ZWaveErrorCodes.ReplaceFailedNode_Failed);
        }
        this._includeNonSecure = includeNonSecure;
        const result = await this.driver.sendMessage(new ReplaceFailedNodeRequest_1.ReplaceFailedNodeRequest(this.driver, {
            failedNodeId: nodeId,
        }));
        if (!result.isOK()) {
            // This implicates that the process was unsuccessful.
            let message = `The node replace process could not be started due to the following reasons:`;
            if (!!(result.replaceStatus &
                ReplaceFailedNodeRequest_1.ReplaceFailedNodeStartFlags.NotPrimaryController)) {
                message += "\n· This controller is not the primary controller";
            }
            if (!!(result.replaceStatus &
                ReplaceFailedNodeRequest_1.ReplaceFailedNodeStartFlags.NodeNotFound)) {
                message += `\n· Node ${nodeId} is not in the list of failed nodes`;
            }
            if (!!(result.replaceStatus &
                ReplaceFailedNodeRequest_1.ReplaceFailedNodeStartFlags.ReplaceProcessBusy)) {
                message += `\n· The node replace process is currently busy`;
            }
            if (!!(result.replaceStatus &
                ReplaceFailedNodeRequest_1.ReplaceFailedNodeStartFlags.ReplaceFailed)) {
                message += `\n· The controller is busy or the node has responded`;
            }
            throw new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.ReplaceFailedNode_Failed);
        }
        else {
            // Remember which node we're trying to replace
            this._nodePendingReplace = this.nodes.get(nodeId);
            this._replaceFailedPromise = deferred_promise_1.createDeferredPromise();
            return this._replaceFailedPromise;
        }
    }
    /**
     * @internal
     * Serializes the controller information and all nodes to store them in a cache.
     */
    serialize() {
        return {
            nodes: objects_1.composeObject([...this.nodes.entries()].map(([id, node]) => [id.toString(), node.serialize()])),
        };
    }
    /**
     * @internal
     * Deserializes the controller information and all nodes from the cache.
     */
    async deserialize(serialized) {
        if (typeguards_1.isObject(serialized.nodes)) {
            for (const nodeId of Object.keys(serialized.nodes)) {
                const serializedNode = serialized.nodes[nodeId];
                if (!serializedNode ||
                    typeof serializedNode.id !== "number" ||
                    serializedNode.id.toString() !== nodeId) {
                    throw new core_1.ZWaveError("The cache file is invalid", core_1.ZWaveErrorCodes.Driver_InvalidCache);
                }
                if (this.nodes.has(serializedNode.id)) {
                    await this.nodes
                        .get(serializedNode.id)
                        .deserialize(serializedNode);
                }
            }
        }
    }
}
exports.ZWaveController = ZWaveController;

//# sourceMappingURL=Controller.js.map
