"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ControllerLogger = exports.CONTROLLER_LABEL = void 0;
const core_1 = require("@zwave-js/core");
const typeguards_1 = require("alcalzone-shared/typeguards");
const Types_1 = require("../node/Types");
exports.CONTROLLER_LABEL = "CNTRLR";
const CONTROLLER_LOGLEVEL = "info";
const VALUE_LOGLEVEL = "debug";
class ControllerLogger extends core_1.ZWaveLoggerBase {
    constructor(loggers) {
        super(loggers, exports.CONTROLLER_LABEL);
        this.valueEventPrefixes = Object.freeze({
            added: "+",
            updated: "~",
            removed: "-",
            notification: "!",
        });
    }
    isValueLogVisible() {
        if (this._isValueLogVisible === undefined) {
            this._isValueLogVisible = this.container.isLoglevelVisible(VALUE_LOGLEVEL);
        }
        return this._isValueLogVisible;
    }
    isControllerLogVisible() {
        if (this._isControllerLogVisible === undefined) {
            this._isControllerLogVisible = this.container.isLoglevelVisible(CONTROLLER_LOGLEVEL);
        }
        return this._isControllerLogVisible;
    }
    /**
     * Logs a message
     * @param msg The message to output
     */
    print(message, level) {
        const actualLevel = level || CONTROLLER_LOGLEVEL;
        if (!this.container.isLoglevelVisible(actualLevel))
            return;
        this.logger.log({
            level: actualLevel,
            message,
            direction: core_1.getDirectionPrefix("none"),
        });
    }
    logNode(nodeId, messageOrOptions, logLevel) {
        if (typeof messageOrOptions === "string") {
            return this.logNode(nodeId, {
                message: messageOrOptions,
                level: logLevel,
            });
        }
        const { level, message, direction, endpoint } = messageOrOptions;
        const actualLevel = level || CONTROLLER_LOGLEVEL;
        if (!this.container.isLoglevelVisible(actualLevel))
            return;
        if (!this.container.shouldLogNode(nodeId))
            return;
        this.logger.log({
            level: actualLevel,
            primaryTags: core_1.tagify([core_1.getNodeTag(nodeId)]),
            message,
            secondaryTags: endpoint
                ? core_1.tagify([`Endpoint ${endpoint}`])
                : undefined,
            direction: core_1.getDirectionPrefix(direction || "none"),
        });
    }
    formatValue(value) {
        if (typeguards_1.isObject(value))
            return JSON.stringify(value);
        if (typeof value !== "string")
            return String(value);
        return `"${value}"`;
    }
    value(change, args) {
        if (!this.isValueLogVisible())
            return;
        const primaryTags = [
            core_1.getNodeTag(args.nodeId),
            this.valueEventPrefixes[change],
            core_1.CommandClasses[args.commandClass],
        ];
        const secondaryTags = [];
        if (args.endpoint != undefined) {
            secondaryTags.push(`Endpoint ${args.endpoint}`);
        }
        if (args.internal === true) {
            secondaryTags.push("internal");
        }
        let message = args.property.toString();
        if (args.propertyKey != undefined) {
            message += `[${args.propertyKey}]`;
        }
        switch (change) {
            case "added":
                message += `: ${this.formatValue(args.newValue)}`;
                break;
            case "updated": {
                const _args = args;
                message += `: ${this.formatValue(_args.prevValue)} => ${this.formatValue(_args.newValue)}`;
                break;
            }
            case "removed":
                message += ` (was ${this.formatValue(args.prevValue)})`;
                break;
            case "notification":
                message += `: ${this.formatValue(args.value)}`;
                break;
        }
        this.logger.log({
            level: VALUE_LOGLEVEL,
            primaryTags: core_1.tagify(primaryTags),
            secondaryTags: core_1.tagify(secondaryTags),
            message,
            direction: core_1.getDirectionPrefix("none"),
        });
    }
    /** Prints a log message for updated metadata of a value id */
    metadataUpdated(args) {
        if (!this.isValueLogVisible())
            return;
        const primaryTags = [
            core_1.getNodeTag(args.nodeId),
            core_1.CommandClasses[args.commandClass],
        ];
        const secondaryTags = [];
        if (args.endpoint != undefined) {
            secondaryTags.push(`Endpoint ${args.endpoint}`);
        }
        if (args.internal === true) {
            secondaryTags.push("internal");
        }
        let message = args.property.toString();
        if (args.propertyKey != undefined) {
            message += `[${args.propertyKey}]`;
        }
        message += ": metadata updated";
        this.logger.log({
            level: VALUE_LOGLEVEL,
            primaryTags: core_1.tagify(primaryTags),
            secondaryTags: core_1.tagify(secondaryTags),
            message,
            direction: core_1.getDirectionPrefix("none"),
        });
    }
    /** Logs the interview progress of a node */
    interviewStage(node) {
        if (!this.isControllerLogVisible())
            return;
        if (!this.container.shouldLogNode(node.id))
            return;
        this.logger.log({
            level: CONTROLLER_LOGLEVEL,
            primaryTags: core_1.tagify([core_1.getNodeTag(node.id)]),
            message: node.interviewStage === Types_1.InterviewStage.Complete
                ? "Interview completed"
                : `Interview stage completed: ${Types_1.InterviewStage[node.interviewStage]}`,
            direction: core_1.getDirectionPrefix("none"),
        });
    }
    /** Logs the interview progress of a node */
    interviewStart(node) {
        if (!this.isControllerLogVisible())
            return;
        if (!this.container.shouldLogNode(node.id))
            return;
        const message = `Beginning interview - last completed stage: ${Types_1.InterviewStage[node.interviewStage]}`;
        this.logger.log({
            level: CONTROLLER_LOGLEVEL,
            primaryTags: core_1.tagify([core_1.getNodeTag(node.id)]),
            message,
            direction: core_1.getDirectionPrefix("none"),
        });
    }
}
exports.ControllerLogger = ControllerLogger;

//# sourceMappingURL=Controller.js.map
