export interface JsonlDBOptions<V> {
    /**
     * Whether errors reading the db file (e.g. invalid JSON) should silently be ignored.
     * **Warning:** This may result in inconsistent data!
     */
    ignoreReadErrors?: boolean;
    /**
     * An optional reviver function (similar to JSON.parse) to transform parsed values before they are accessible in the database.
     * If this function is defined, it must always return a value.
     */
    reviver?: (key: string, value: any) => V;
    /**
     * An optional serializer function (similar to JSON.serialize) to transform values before they are written to the database file.
     * If this function is defined, it must always return a value.
     */
    serializer?: (key: string, value: V) => any;
    /**
     * Configure when the DB should be automatically compressed.
     * If multiple conditions are configured, the DB is compressed when any of them are fulfilled
     */
    autoCompress?: Partial<{
        /**
         * Compress when uncompressedSize >= size * sizeFactor. Default: +Infinity
         */
        sizeFactor: number;
        /**
         * Configure the minimum size necessary for auto-compression based on size. Default: 0
         */
        sizeFactorMinimumSize: number;
        /**
         * Compress after a certain time has passed. Default: never
         */
        intervalMs: number;
        /**
         * Configure the minimum count of changes for auto-compression based on time. Default: 1
         */
        intervalMinChanges: number;
        /** Compress when closing the DB. Default: false */
        onClose: boolean;
        /** Compress after opening the DB. Default: false */
        onOpen: boolean;
    }>;
    /**
     * Can be used to throttle write accesses to the filesystem. By default,
     * every change is immediately written to the FS
     */
    throttleFS?: {
        /**
         * Minimum wait time between two consecutive write accesses. Default: 0
         */
        intervalMs: number;
        /**
         * Maximum commands to be buffered before forcing a write access. Default: +Infinity
         */
        maxBufferedCommands?: number;
    };
}
/** This is the same as `fs-extra`'s WriteOptions */
export interface FsWriteOptions {
    encoding?: string | null;
    flag?: string;
    mode?: number;
    fs?: object;
    replacer?: any;
    spaces?: number | string;
    EOL?: string;
}
export declare class JsonlDB<V extends unknown = unknown> {
    constructor(filename: string, options?: JsonlDBOptions<V>);
    private validateOptions;
    readonly filename: string;
    readonly dumpFilename: string;
    readonly backupFilename: string;
    private options;
    private _db;
    forEach: Map<string, V>["forEach"];
    get: Map<string, V>["get"];
    has: Map<string, V>["has"];
    [Symbol.iterator]: () => IterableIterator<[string, V]>;
    entries: Map<string, V>["entries"];
    keys: Map<string, V>["keys"];
    values: Map<string, V>["values"];
    get size(): number;
    private _uncompressedSize;
    /** Returns the line count of the appendonly file, excluding empty lines */
    get uncompressedSize(): number;
    private _changesSinceLastCompress;
    private _isOpen;
    get isOpen(): boolean;
    private _fd;
    private _dumpFd;
    private _compressBacklog;
    private _writeBacklog;
    private _writeCorkCount;
    private _writeCorkTimeout;
    private _dumpBacklog;
    private _compressInterval;
    private _openPromise;
    open(): Promise<void>;
    /**
     * Makes sure that there are no remains of a previous broken compress attempt and restores
     * a DB backup if it exists.
     */
    private tryRecoverDBFiles;
    /** Reads a line and extracts the key without doing a full-blown JSON.parse() */
    private parseKey;
    /** Parses a line and updates the internal DB correspondingly */
    private parseLine;
    clear(): void;
    delete(key: string): boolean;
    set(key: string, value: V): this;
    private importJsonFile;
    importJson(filename: string): Promise<void>;
    importJson(json: Record<string, any>): void;
    exportJson(filename: string, options?: FsWriteOptions): Promise<void>;
    private updateStatistics;
    private needToCompress;
    private cork;
    private uncork;
    private autoCork;
    /**
     * Writes a line into the correct backlog
     * @param noAutoCompress Whether auto-compression should be disabled
     */
    private write;
    private entryToLine;
    /** Saves a compressed copy of the DB into `<filename>.dump` */
    dump(): Promise<void>;
    /** Asynchronously performs all write actions */
    private writeThread;
    private compressPromise;
    private compressInternal;
    /** Compresses the db by dumping it and overwriting the aof file. */
    compress(): Promise<void>;
    /** Resolves when the `writeThread()` is finished */
    private _writePromise;
    /** Resolves when the `dump()` method is finished */
    private _dumpPromise;
    /** Closes the DB and waits for all data to be written */
    close(): Promise<void>;
}
//# sourceMappingURL=db.d.ts.map