"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonlDB = void 0;
const deferred_promise_1 = require("alcalzone-shared/deferred-promise");
const objects_1 = require("alcalzone-shared/objects");
const fs = __importStar(require("fs-extra"));
const path = __importStar(require("path"));
const lockfile = __importStar(require("proper-lockfile"));
const readline = __importStar(require("readline"));
const stream = __importStar(require("stream"));
/**
 * fsync on a directory ensures there are no rename operations etc. which haven't been persisted to disk.
 */
async function fsyncDir(dirname) {
    // Windows will cause `EPERM: operation not permitted, fsync`
    // for directories, so don't do this
    /* istanbul ignore else */
    if (process.platform === "win32")
        return;
    else {
        const fd = await fs.open(dirname, "r");
        await fs.fsync(fd);
        await fs.close(fd);
    }
}
class JsonlDB {
    constructor(filename, options = {}) {
        this._db = new Map();
        this._uncompressedSize = Number.NaN;
        this._changesSinceLastCompress = 0;
        this._isOpen = false;
        this._writeCorkCount = 0;
        this.validateOptions(options);
        this.filename = filename;
        this.dumpFilename = this.filename + ".dump";
        this.backupFilename = this.filename + ".bak";
        this.options = options;
        // Bind all map properties we can use directly
        this.forEach = this._db.forEach.bind(this._db);
        this.get = this._db.get.bind(this._db);
        this.has = this._db.has.bind(this._db);
        this.entries = this._db.entries.bind(this._db);
        this.keys = this._db.keys.bind(this._db);
        this.values = this._db.values.bind(this._db);
        this[Symbol.iterator] = this._db[Symbol.iterator].bind(this._db);
    }
    validateOptions(options) {
        if (options.autoCompress) {
            const { sizeFactor, sizeFactorMinimumSize, intervalMs, intervalMinChanges, } = options.autoCompress;
            if (sizeFactor != undefined && sizeFactor <= 1) {
                throw new Error("sizeFactor must be > 1");
            }
            if (sizeFactorMinimumSize != undefined &&
                sizeFactorMinimumSize < 0) {
                throw new Error("sizeFactorMinimumSize must be >= 0");
            }
            if (intervalMs != undefined && intervalMs < 10) {
                throw new Error("intervalMs must be >= 10");
            }
            if (intervalMinChanges != undefined && intervalMinChanges < 1) {
                throw new Error("intervalMinChanges must be >= 1");
            }
        }
        if (options.throttleFS) {
            const { intervalMs, maxBufferedCommands } = options.throttleFS;
            if (intervalMs < 0) {
                throw new Error("intervalMs must be >= 0");
            }
            if (maxBufferedCommands != undefined && maxBufferedCommands < 0) {
                throw new Error("maxBufferedCommands must be >= 0");
            }
        }
    }
    get size() {
        return this._db.size;
    }
    /** Returns the line count of the appendonly file, excluding empty lines */
    get uncompressedSize() {
        if (!this._isOpen) {
            throw new Error("The database is not open!");
        }
        return this._uncompressedSize;
    }
    get isOpen() {
        return this._isOpen;
    }
    // /** Opens the database file or creates it if it doesn't exist */
    async open() {
        var _a;
        // Open the file for appending and reading
        await fs.ensureDir(path.dirname(this.filename));
        try {
            await lockfile.lock(this.filename, {
                // We cannot be sure that the file exists before acquiring the lock
                realpath: false,
                stale: 
                // Avoid timeouts during testing
                process.env.NODE_ENV === "test"
                    ? 100000
                    : /* istanbul ignore next - this is impossible to test */ undefined,
                onCompromised: /* istanbul ignore next */ () => {
                    // do nothing
                },
            });
        }
        catch (e) {
            throw new Error(`Failed to lock DB file "${this.filename}"!`);
        }
        // If the application crashed previously, try to recover from it
        await this.tryRecoverDBFiles();
        this._fd = await fs.open(this.filename, "a+");
        const readStream = fs.createReadStream(this.filename, {
            encoding: "utf8",
            fd: this._fd,
            autoClose: false,
        });
        const rl = readline.createInterface(readStream);
        let lineNo = 0;
        this._uncompressedSize = 0;
        try {
            await new Promise((resolve, reject) => {
                const actualLines = new Map();
                rl.on("line", (line) => {
                    // Count source lines for the error message
                    lineNo++;
                    // Skip empty lines
                    if (!line)
                        return;
                    try {
                        this._uncompressedSize++;
                        // Extract the key and only remember the last line for each one
                        const key = this.parseKey(line);
                        actualLines.set(key, [lineNo, line]);
                    }
                    catch (e) {
                        if (this.options.ignoreReadErrors === true) {
                            return;
                        }
                        else {
                            reject(new Error(`Cannot open file: Invalid data in line ${lineNo}`));
                        }
                    }
                });
                rl.on("close", () => {
                    // We've read all lines, now only parse those that contain useful data
                    for (const [lineNo, line] of actualLines.values()) {
                        try {
                            this.parseLine(line);
                        }
                        catch (e) {
                            if (this.options.ignoreReadErrors === true) {
                                continue;
                            }
                            else {
                                reject(new Error(`Cannot open file: Invalid data in line ${lineNo}`));
                            }
                        }
                    }
                    resolve();
                });
            });
        }
        finally {
            // Close the file again to avoid EBADF
            rl.close();
            await fs.close(this._fd);
            this._fd = undefined;
        }
        const { onOpen, intervalMs, intervalMinChanges = 1, } = (_a = this.options.autoCompress) !== null && _a !== void 0 ? _a : {};
        // If the DB should be compressed while opening, do it before starting the write thread
        if (onOpen) {
            await this.compressInternal();
        }
        // Start the write thread
        this._openPromise = deferred_promise_1.createDeferredPromise();
        void this.writeThread();
        await this._openPromise;
        this._isOpen = true;
        // Start regular auto-compression
        if (intervalMs) {
            this._compressInterval = setInterval(() => {
                if (this._changesSinceLastCompress >= intervalMinChanges) {
                    void this.compress();
                }
            }, intervalMs);
        }
    }
    /**
     * Makes sure that there are no remains of a previous broken compress attempt and restores
     * a DB backup if it exists.
     */
    async tryRecoverDBFiles() {
        // During the compression, the following sequence of events happens:
        // 1. A .jsonl.dump file gets written with a compressed copy of the data
        // 2. Files get renamed: .jsonl -> .jsonl.bak, .jsonl.dump -> .jsonl
        // 3. .bak file gets removed
        // 4. Buffered data gets written to the .jsonl file
        // This means if the .jsonl file is absent or truncated, we should be able to pick either the .dump or the .bak file
        // and restore the .jsonl file from it
        let dbFileIsOK = false;
        try {
            const dbFileStats = await fs.stat(this.filename);
            dbFileIsOK = dbFileStats.isFile() && dbFileStats.size > 0;
        }
        catch {
            // ignore
        }
        // Prefer the DB file if it exists, remove the others in case they exist
        if (dbFileIsOK) {
            try {
                await fs.remove(this.backupFilename);
            }
            catch {
                // ignore
            }
            try {
                await fs.remove(this.dumpFilename);
            }
            catch {
                // ignore
            }
            return;
        }
        // The backup file should have complete data - the dump file could be subject to an incomplete write
        let bakFileIsOK = false;
        try {
            const bakFileStats = await fs.stat(this.backupFilename);
            bakFileIsOK = bakFileStats.isFile() && bakFileStats.size > 0;
        }
        catch {
            // ignore
        }
        if (bakFileIsOK) {
            // Overwrite the broken db file with it and delete the dump file
            await fs.move(this.backupFilename, this.filename, {
                overwrite: true,
            });
            try {
                await fs.remove(this.dumpFilename);
            }
            catch {
                // ignore
            }
            return;
        }
        // Try the dump file as a last attempt
        let dumpFileIsOK = false;
        try {
            const dumpFileStats = await fs.stat(this.dumpFilename);
            dumpFileIsOK = dumpFileStats.isFile() && dumpFileStats.size > 0;
        }
        catch {
            // ignore
        }
        if (dumpFileIsOK) {
            // Overwrite the broken db file with the dump file and delete the backup file
            await fs.move(this.dumpFilename, this.filename, {
                overwrite: true,
            });
            try {
                await fs.remove(this.backupFilename);
            }
            catch {
                // ignore
            }
            return;
        }
    }
    /** Reads a line and extracts the key without doing a full-blown JSON.parse() */
    parseKey(line) {
        if (0 !== line.indexOf(`{"k":"`)) {
            throw new Error("invalid data");
        }
        const keyStart = 6;
        let keyEnd = line.indexOf(`","v":`, keyStart);
        if (-1 === keyEnd) {
            // try again with a delete command
            if (line.endsWith(`"}`)) {
                keyEnd = line.length - 2;
            }
            else {
                throw new Error("invalid data");
            }
        }
        return line.slice(keyStart, keyEnd);
    }
    /** Parses a line and updates the internal DB correspondingly */
    parseLine(line) {
        const record = JSON.parse(line);
        const { k, v } = record;
        if (v !== undefined) {
            this._db.set(k, typeof this.options.reviver === "function"
                ? this.options.reviver(k, v)
                : v);
        }
        else {
            this._db.delete(k);
        }
    }
    clear() {
        if (!this._isOpen) {
            throw new Error("The database is not open!");
        }
        this._db.clear();
        this.write("");
    }
    delete(key) {
        if (!this._isOpen) {
            throw new Error("The database is not open!");
        }
        const ret = this._db.delete(key);
        if (ret) {
            // Something was deleted
            this.write(this.entryToLine(key));
        }
        return ret;
    }
    set(key, value) {
        if (!this._isOpen) {
            throw new Error("The database is not open!");
        }
        this._db.set(key, value);
        this.write(this.entryToLine(key, value));
        return this;
    }
    async importJsonFile(filename) {
        const json = await fs.readJSON(filename);
        return this.importJson(json);
    }
    importJson(jsonOrFile) {
        if (typeof jsonOrFile === "string") {
            if (!this._isOpen) {
                return Promise.reject(new Error("The database is not open!"));
            }
            return this.importJsonFile(jsonOrFile);
        }
        else {
            if (!this._isOpen) {
                throw new Error("The database is not open!");
            }
        }
        for (const [key, value] of Object.entries(jsonOrFile)) {
            this._db.set(key, value);
            this.write(this.entryToLine(key, value), true);
        }
    }
    async exportJson(filename, options) {
        if (!this._isOpen) {
            return Promise.reject(new Error("The database is not open!"));
        }
        return fs.writeJSON(filename, objects_1.composeObject([...this._db]), options);
    }
    updateStatistics(command) {
        if (command === "") {
            this._uncompressedSize = 0;
        }
        else {
            this._uncompressedSize++;
        }
        this._changesSinceLastCompress++;
    }
    needToCompress() {
        var _a;
        // compression is busy?
        if (this.compressPromise)
            return false;
        const { sizeFactor = Number.POSITIVE_INFINITY, sizeFactorMinimumSize = 0, } = (_a = this.options.autoCompress) !== null && _a !== void 0 ? _a : {};
        if (this.uncompressedSize >= sizeFactorMinimumSize &&
            this.uncompressedSize >= sizeFactor * this.size) {
            return true;
        }
        return false;
    }
    cork() {
        /* istanbul ignore else - this is impossible to test */
        if (this._writeBacklog && this._writeCorkCount === 0) {
            this._writeBacklog.cork();
            this._writeCorkCount++;
        }
    }
    uncork() {
        if (this._writeCorkCount > 0 && this._writeCorkTimeout) {
            clearTimeout(this._writeCorkTimeout);
            this._writeCorkTimeout = undefined;
        }
        while (this._writeBacklog && this._writeCorkCount > 0) {
            this._writeBacklog.uncork();
            this._writeCorkCount--;
        }
    }
    autoCork() {
        var _a, _b, _c;
        if (!((_a = this.options.throttleFS) === null || _a === void 0 ? void 0 : _a.intervalMs))
            return;
        const maybeUncork = () => {
            var _a;
            if (this._writeBacklog && this._writeBacklog.writableLength > 0) {
                // This gets the stream flowing again. The write thread will call
                // autoCork when it is done
                this.uncork();
            }
            else {
                // Nothing to uncork, schedule the next timeout
                (_a = this._writeCorkTimeout) === null || _a === void 0 ? void 0 : _a.refresh();
            }
        };
        // Cork once and schedule the uncork
        this.cork();
        this._writeCorkTimeout =
            (_c = (_b = this._writeCorkTimeout) === null || _b === void 0 ? void 0 : _b.refresh()) !== null && _c !== void 0 ? _c : setTimeout(maybeUncork, this.options.throttleFS.intervalMs);
    }
    /**
     * Writes a line into the correct backlog
     * @param noAutoCompress Whether auto-compression should be disabled
     */
    write(line, noAutoCompress = false) {
        var _a;
        /* istanbul ignore else */
        if (this._compressBacklog && !this._compressBacklog.destroyed) {
            // The compress backlog handling also handles the file statistics
            this._compressBacklog.write(line);
        }
        else if (this._writeBacklog && !this._writeBacklog.destroyed) {
            // Update line statistics
            this.updateStatistics(line);
            // Either compress or write to the main file, never both
            if (!noAutoCompress && this.needToCompress()) {
                this.compress();
            }
            else {
                this._writeBacklog.write(line);
                // If this is a throttled stream, uncork it as soon as the write
                // buffer is larger than configured
                if (((_a = this.options.throttleFS) === null || _a === void 0 ? void 0 : _a.maxBufferedCommands) != undefined &&
                    this._writeBacklog.writableLength >
                        this.options.throttleFS.maxBufferedCommands) {
                    this.uncork();
                }
            }
        }
        else {
            throw new Error("Cannot write into the database while no streams are open!");
        }
        // If necessary, write to the dump backlog, so the dump doesn't miss any data
        if (this._dumpBacklog && !this._dumpBacklog.destroyed) {
            this._dumpBacklog.write(line);
        }
    }
    entryToLine(key, value) {
        var _a, _b, _c;
        if (value !== undefined) {
            return JSON.stringify({
                k: key,
                v: (_c = (_b = (_a = this.options).serializer) === null || _b === void 0 ? void 0 : _b.call(_a, key, value)) !== null && _c !== void 0 ? _c : value,
            });
        }
        else {
            return JSON.stringify({ k: key });
        }
    }
    /** Saves a compressed copy of the DB into `<filename>.dump` */
    async dump() {
        this._dumpPromise = deferred_promise_1.createDeferredPromise();
        // Open the file for writing (or truncate if it exists)
        this._dumpFd = await fs.open(this.dumpFilename, "w+");
        // And start dumping the DB
        // Start by creating a dump backlog, so parallel writes will be remembered
        this._dumpBacklog = new stream.PassThrough({ objectMode: true });
        // Create a copy of the other entries in the DB
        const entries = [...this._db];
        // And persist them
        for (const [key, value] of entries) {
            await fs.appendFile(this._dumpFd, this.entryToLine(key, value) + "\n");
        }
        // In case there is any data in the backlog stream, persist that too
        let line;
        while (null !== (line = this._dumpBacklog.read())) {
            await fs.appendFile(this._dumpFd, line + "\n");
        }
        this._dumpBacklog.destroy();
        this._dumpBacklog = undefined;
        // The dump backlog was closed, this means that the dump is finished.
        // Close the file and resolve the close promise
        await fs.fsync(this._dumpFd); // The dump should be on disk ASAP, so we fsync
        await fs.close(this._dumpFd);
        this._dumpFd = undefined;
        this._dumpPromise.resolve();
    }
    /** Asynchronously performs all write actions */
    async writeThread() {
        var _a;
        // This must be called before any awaits
        this._writeBacklog = new stream.PassThrough({ objectMode: true });
        this.autoCork();
        this._writePromise = deferred_promise_1.createDeferredPromise();
        // Open the file for appending and reading
        this._fd = await fs.open(this.filename, "a+");
        (_a = this._openPromise) === null || _a === void 0 ? void 0 : _a.resolve();
        for await (const action of this
            ._writeBacklog) {
            if (action === "") {
                // Since we opened the file in append mode, we cannot truncate
                // therefore close and open in write mode again
                await fs.close(this._fd);
                this._fd = await fs.open(this.filename, "w+");
            }
            else {
                await fs.appendFile(this._fd, action + "\n");
            }
            // When this is a throttled stream, auto-cork it when it was drained
            if (this._writeBacklog.readableLength === 0 && this._isOpen) {
                this.autoCork();
            }
        }
        this._writeBacklog.destroy();
        // The write backlog was closed, this means that the DB is being closed
        // Flush the file contents to disk, close the file and resolve the close promise
        await fs.fsync(this._fd);
        await fs.close(this._fd);
        this._writePromise.resolve();
    }
    async compressInternal() {
        if (this.compressPromise)
            return;
        this.compressPromise = deferred_promise_1.createDeferredPromise();
        // Immediately remember the database size or writes while compressing
        // will be incorrectly reflected
        this._uncompressedSize = this.size;
        this._changesSinceLastCompress = 0;
        await this.dump();
        // After dumping, restart the write thread so no duplicate entries get written
        // Disable writing into the backlog stream and buffer all writes
        // in the compress backlog in the meantime
        this._compressBacklog = new stream.PassThrough({ objectMode: true });
        this.uncork();
        // Replace the aof file. To make sure that the data fully reaches the storage, we employ the following strategy:
        // 1. Ensure there are no pending rename operations or file creations
        await fsyncDir(path.dirname(this.filename));
        // 2. Ensure the db file is fully written to disk. The write thread will fsync before closing
        if (this._writeBacklog) {
            this._writeBacklog.end();
            await this._writePromise;
            this._writeBacklog = undefined;
        }
        // 3. Create backup, rename the dump file, then ensure the directory entries are written to disk
        await fs.move(this.filename, this.backupFilename, {
            overwrite: true,
        });
        await fs.move(this.dumpFilename, this.filename, { overwrite: true });
        await fsyncDir(path.dirname(this.filename));
        // 4. Delete backup
        await fs.unlink(this.backupFilename);
        if (this._isOpen) {
            // Start the write thread again
            this._openPromise = deferred_promise_1.createDeferredPromise();
            void this.writeThread();
            await this._openPromise;
        }
        // In case there is any data in the backlog stream, persist that too
        let line;
        while (null !== (line = this._compressBacklog.read())) {
            this.updateStatistics(line);
            this._writeBacklog.write(line);
        }
        this._compressBacklog.destroy();
        this._compressBacklog = undefined;
        // If any method is waiting for the compress process, signal it that we're done
        this.compressPromise.resolve();
        this.compressPromise = undefined;
    }
    /** Compresses the db by dumping it and overwriting the aof file. */
    async compress() {
        if (!this._isOpen)
            return;
        return this.compressInternal();
    }
    /** Closes the DB and waits for all data to be written */
    async close() {
        var _a;
        this._isOpen = false;
        if (this._compressInterval)
            clearInterval(this._compressInterval);
        if (this._writeCorkTimeout)
            clearTimeout(this._writeCorkTimeout);
        if (this.compressPromise) {
            // Wait until any pending compress processes are complete
            await this.compressPromise;
        }
        else if ((_a = this.options.autoCompress) === null || _a === void 0 ? void 0 : _a.onClose) {
            // Compress if required
            await this.compressInternal();
        }
        // Disable writing into the backlog stream and wait for the write process to finish
        if (this._writeBacklog) {
            this.uncork();
            this._writeBacklog.end();
            await this._writePromise;
        }
        // Also wait for a potential dump process to finish
        /* istanbul ignore next - this is impossible to test since it requires exact timing */
        if (this._dumpBacklog) {
            // Disable writing into the dump backlog stream
            this._dumpBacklog.end();
            await this._dumpPromise;
        }
        // Reset all variables
        this._writePromise = undefined;
        this._dumpPromise = undefined;
        this._db.clear();
        this._fd = undefined;
        this._dumpFd = undefined;
        this._changesSinceLastCompress = 0;
        this._uncompressedSize = Number.NaN;
        this._writeCorkCount = 0;
        // Free the lock
        try {
            if (await lockfile.check(this.filename, { realpath: false }))
                await lockfile.unlock(this.filename, { realpath: false });
        }
        catch {
            // whatever - just don't crash
        }
    }
}
exports.JsonlDB = JsonlDB;
//# sourceMappingURL=db.js.map