/*! @sentry/integrations 6.19.7 (5b3a175) | https://github.com/getsentry/sentry-javascript */
(function (__window) {
var exports = {};

/**
 * NOTE: In order to avoid circular dependencies, if you add a function to this module and it needs to print something,
 * you must either a) use `console.log` rather than the logger, or b) put your function elsewhere.
 */
const fallbackGlobalObject = {};
/**
 * Safely get global scope object
 *
 * @returns Global scope object
 */
function getGlobalObject() {
    return (typeof window !== 'undefined' // eslint-disable-line no-restricted-globals
            ? window // eslint-disable-line no-restricted-globals
            : typeof self !== 'undefined'
                ? self
                : fallbackGlobalObject);
}
/**
 * Returns a global singleton contained in the global `__SENTRY__` object.
 *
 * If the singleton doesn't already exist in `__SENTRY__`, it will be created using the given factory
 * function and added to the `__SENTRY__` object.
 *
 * @param name name of the global singleton on __SENTRY__
 * @param creator creator Factory function to create the singleton if it doesn't already exist on `__SENTRY__`
 * @param obj (Optional) The global object on which to look for `__SENTRY__`, if not `getGlobalObject`'s return value
 * @returns the singleton
 */
function getGlobalSingleton(name, creator, obj) {
    const global = (obj || getGlobalObject());
    const __SENTRY__ = (global.__SENTRY__ = global.__SENTRY__ || {});
    const singleton = __SENTRY__[name] || (__SENTRY__[name] = creator());
    return singleton;
}

/*
 * This file defines flags and constants that can be modified during compile time in order to facilitate tree shaking
 * for users.
 *
 * Debug flags need to be declared in each package individually and must not be imported across package boundaries,
 * because some build tools have trouble tree-shaking imported guards.
 *
 * As a convention, we define debug flags in a `flags.ts` file in the root of a package's `src` folder.
 *
 * Debug flag files will contain "magic strings" like `true` that may get replaced with actual values during
 * our, or the user's build process. Take care when introducing new flags - they must not throw if they are not
 * replaced.
 */
/** Flag that is true for debug builds, false otherwise. */
const IS_DEBUG_BUILD = true;

// TODO: Implement different loggers for different environments
const global = getGlobalObject();
/** Prefix for logging strings */
const PREFIX = 'Sentry Logger ';
const CONSOLE_LEVELS = ['debug', 'info', 'warn', 'error', 'log', 'assert'];
/**
 * Temporarily disable sentry console instrumentations.
 *
 * @param callback The function to run against the original `console` messages
 * @returns The results of the callback
 */
function consoleSandbox(callback) {
    const global = getGlobalObject();
    if (!('console' in global)) {
        return callback();
    }
    const originalConsole = global.console;
    const wrappedLevels = {};
    // Restore all wrapped console methods
    CONSOLE_LEVELS.forEach(level => {
        // TODO(v7): Remove this check as it's only needed for Node 6
        const originalWrappedFunc = originalConsole[level] && originalConsole[level].__sentry_original__;
        if (level in global.console && originalWrappedFunc) {
            wrappedLevels[level] = originalConsole[level];
            originalConsole[level] = originalWrappedFunc;
        }
    });
    try {
        return callback();
    }
    finally {
        // Revert restoration to wrapped state
        Object.keys(wrappedLevels).forEach(level => {
            originalConsole[level] = wrappedLevels[level];
        });
    }
}
function makeLogger() {
    let enabled = false;
    const logger = {
        enable: () => {
            enabled = true;
        },
        disable: () => {
            enabled = false;
        },
    };
    if (IS_DEBUG_BUILD) {
        CONSOLE_LEVELS.forEach(name => {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            logger[name] = (...args) => {
                if (enabled) {
                    consoleSandbox(() => {
                        global.console[name](`${PREFIX}[${name}]:`, ...args);
                    });
                }
            };
        });
    }
    else {
        CONSOLE_LEVELS.forEach(name => {
            logger[name] = () => undefined;
        });
    }
    return logger;
}
if (IS_DEBUG_BUILD) {
    getGlobalSingleton('logger', makeLogger);
}
else {
    makeLogger();
}

/**
 * Integration to debug sent Sentry events.
 * This integration should not be used in production
 */
class Debug {
    constructor(options) {
        /**
         * @inheritDoc
         */
        this.name = Debug.id;
        this._options = Object.assign({ debugger: false, stringify: false }, options);
    }
    /**
     * @inheritDoc
     */
    setupOnce(addGlobalEventProcessor, getCurrentHub) {
        addGlobalEventProcessor((event, hint) => {
            const self = getCurrentHub().getIntegration(Debug);
            if (self) {
                if (self._options.debugger) {
                    // eslint-disable-next-line no-debugger
                    debugger;
                }
                /* eslint-disable no-console */
                consoleSandbox(() => {
                    if (self._options.stringify) {
                        console.log(JSON.stringify(event, null, 2));
                        if (hint) {
                            console.log(JSON.stringify(hint, null, 2));
                        }
                    }
                    else {
                        console.log(event);
                        if (hint) {
                            console.log(hint);
                        }
                    }
                });
                /* eslint-enable no-console */
            }
            return event;
        });
    }
}
/**
 * @inheritDoc
 */
Debug.id = 'Debug';

exports.Debug = Debug;


  // Add this module's exports to the global `Sentry.Integrations`
  __window.Sentry = __window.Sentry || {};
  __window.Sentry.Integrations = __window.Sentry.Integrations || {};
  for (var key in exports) {
    if (Object.prototype.hasOwnProperty.call(exports, key)) {
      __window.Sentry.Integrations[key] = exports[key];
    }
  }
}(window));
//# sourceMappingURL=debug.es6.js.map
