"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeviceMetadata = exports.ParamInformation = exports.AssociationConfig = exports.DeviceConfig = exports.loadFulltextDeviceIndexInternal = exports.loadDeviceIndexInternal = exports.fulltextIndexPath = exports.indexPath = exports.devicesDir = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs = __importStar(require("fs-extra"));
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const CompatConfig_1 = require("./CompatConfig");
const JsonTemplate_1 = require("./JsonTemplate");
const Logic_1 = require("./Logic");
const utils_1 = require("./utils");
exports.devicesDir = path_1.default.join(utils_1.configDir, "devices");
exports.indexPath = path_1.default.join(exports.devicesDir, "index.json");
exports.fulltextIndexPath = path_1.default.join(exports.devicesDir, "fulltext_index.json");
async function hasChangedDeviceFiles(dir, lastChange) {
    // Check if there are any files BUT index.json that were changed
    // or directories that were modified
    const filesAndDirs = await fs.readdir(dir);
    for (const f of filesAndDirs) {
        const fullPath = path_1.default.join(dir, f);
        const stat = await fs.stat(fullPath);
        if ((dir !== exports.devicesDir || f !== "index.json") &&
            (stat.isFile() || stat.isDirectory()) &&
            stat.mtime > lastChange) {
            return true;
        }
        else if (stat.isDirectory()) {
            // we need to go deeper!
            if (await hasChangedDeviceFiles(fullPath, lastChange))
                return true;
        }
    }
    return false;
}
async function loadDeviceIndexShared(indexPath, extractIndexEntries, logger) {
    // The index file needs to be regenerated if it does not exist
    let needsUpdate = !(await fs_extra_1.pathExists(indexPath));
    let index;
    let mtimeIndex;
    // ...or if cannot be parsed
    if (!needsUpdate) {
        try {
            const fileContents = await fs_extra_1.readFile(indexPath, "utf8");
            index = json5_1.default.parse(fileContents);
            mtimeIndex = (await fs.stat(indexPath)).mtime;
        }
        catch (_a) {
            logger === null || logger === void 0 ? void 0 : logger.print("Error while parsing index file - regenerating...", "warn");
            needsUpdate = true;
        }
        finally {
            if (!index) {
                logger === null || logger === void 0 ? void 0 : logger.print("Index file was malformed - regenerating...", "warn");
                needsUpdate = true;
            }
        }
    }
    // ...or if there were any changes in the file system
    if (!needsUpdate) {
        needsUpdate = await hasChangedDeviceFiles(exports.devicesDir, mtimeIndex);
        if (needsUpdate) {
            logger === null || logger === void 0 ? void 0 : logger.print("Device configuration files on disk changed - regenerating index...", "verbose");
        }
    }
    if (needsUpdate) {
        index = [];
        const configFiles = await shared_1.enumFilesRecursive(exports.devicesDir, (file) => file.endsWith(".json") &&
            !file.endsWith("index.json") &&
            !file.includes("/templates/") &&
            !file.includes("\\templates\\"));
        for (const file of configFiles) {
            const relativePath = path_1.default
                .relative(exports.devicesDir, file)
                .replace(/\\/g, "/");
            // Try parsing the file
            try {
                const config = await DeviceConfig.from(file, {
                    relativeTo: exports.devicesDir,
                });
                // Add the file to the index
                index.push(...extractIndexEntries(config).map((entry) => ({
                    ...entry,
                    filename: relativePath,
                })));
            }
            catch (e) {
                const message = `Error parsing config file ${relativePath}: ${e.message}`;
                // Crash hard during tests, just print an error when in production systems.
                // A user could have changed a config file
                if (process.env.NODE_ENV === "test" || !!process.env.CI) {
                    throw new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Config_Invalid);
                }
                else {
                    logger === null || logger === void 0 ? void 0 : logger.print(message, "error");
                }
            }
        }
        // Save the index to disk
        try {
            await fs_extra_1.writeFile(path_1.default.join(indexPath), `// This file is auto-generated. DO NOT edit it by hand if you don't know what you're doing!"
${shared_1.stringify(index, "\t")}
`, "utf8");
            logger === null || logger === void 0 ? void 0 : logger.print("Device index regenerated", "verbose");
        }
        catch (e) {
            logger === null || logger === void 0 ? void 0 : logger.print(`Writing the device index to disk failed: ${e.message}`, "error");
        }
    }
    return index;
}
/**
 * @internal
 * Loads the index file to quickly access the device configs.
 * Transparently handles updating the index if necessary
 */
async function loadDeviceIndexInternal(logger) {
    return loadDeviceIndexShared(exports.indexPath, (config) => config.devices.map((dev) => ({
        manufacturerId: shared_1.formatId(config.manufacturerId.toString(16)),
        manufacturer: config.manufacturer,
        label: config.label,
        ...dev,
        firmwareVersion: config.firmwareVersion,
    })), logger);
}
exports.loadDeviceIndexInternal = loadDeviceIndexInternal;
/**
 * @internal
 * Loads the full text index file to quickly search the device configs.
 * Transparently handles updating the index if necessary
 */
async function loadFulltextDeviceIndexInternal(logger) {
    return loadDeviceIndexShared(exports.indexPath, (config) => config.devices.map((dev) => ({
        manufacturerId: shared_1.formatId(config.manufacturerId.toString(16)),
        manufacturer: config.manufacturer,
        label: config.label,
        description: config.description,
        ...dev,
        firmwareVersion: config.firmwareVersion,
    })), logger);
}
exports.loadFulltextDeviceIndexInternal = loadFulltextDeviceIndexInternal;
function isHexKeyWith4Digits(val) {
    return typeof val === "string" && utils_1.hexKeyRegex4Digits.test(val);
}
const firmwareVersionRegex = /^\d{1,3}\.\d{1,3}$/;
function isFirmwareVersion(val) {
    return (typeof val === "string" &&
        firmwareVersionRegex.test(val) &&
        val
            .split(".")
            .map((str) => parseInt(str, 10))
            .every((num) => num >= 0 && num <= 255));
}
function conditionApplies(condition, context) {
    try {
        return !!Logic_1.evaluate(condition, context);
    }
    catch (e) {
        throw new core_1.ZWaveError(`Invalid condition "condition"!`, core_1.ZWaveErrorCodes.Config_Invalid);
    }
}
class DeviceConfig {
    constructor(filename, definition, deviceId) {
        this.filename = filename;
        if (!isHexKeyWith4Digits(definition.manufacturerId)) {
            utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
manufacturer id must be a hexadecimal number with 4 digits`);
        }
        this.manufacturerId = parseInt(definition.manufacturerId, 16);
        for (const prop of ["manufacturer", "label", "description"]) {
            if (typeof definition[prop] !== "string") {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
${prop} is not a string`);
            }
            this[prop] = definition[prop];
        }
        if (!typeguards_1.isArray(definition.devices) ||
            !definition.devices.every((dev) => typeguards_1.isObject(dev) &&
                isHexKeyWith4Digits(dev.productType) &&
                isHexKeyWith4Digits(dev.productId))) {
            utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
devices is malformed (not an object or type/id that is not a 4-digit hex key)`);
        }
        this.devices = definition.devices.map(({ productType, productId }) => ({ productType, productId }));
        if (!typeguards_1.isObject(definition.firmwareVersion) ||
            !isFirmwareVersion(definition.firmwareVersion.min) ||
            !isFirmwareVersion(definition.firmwareVersion.max)) {
            utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
firmwareVersion is malformed or invalid`);
        }
        else {
            const { min, max } = definition.firmwareVersion;
            this.firmwareVersion = { min, max };
        }
        if (definition.associations != undefined) {
            const associations = new Map();
            if (!typeguards_1.isObject(definition.associations)) {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
associations is not an object`);
            }
            for (const [key, assocDefinition] of objects_1.entries(definition.associations)) {
                if (!/^[1-9][0-9]*$/.test(key)) {
                    utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
found non-numeric group id "${key}" in associations`);
                }
                // Check if this entry applies for the actual config
                if (deviceId &&
                    "$if" in assocDefinition &&
                    !conditionApplies(assocDefinition.$if, deviceId)) {
                    continue;
                }
                const keyNum = parseInt(key, 10);
                associations.set(keyNum, new AssociationConfig(filename, keyNum, assocDefinition));
            }
            this.associations = associations;
        }
        if (definition.paramInformation != undefined) {
            const paramInformation = new shared_1.ObjectKeyMap();
            if (!typeguards_1.isObject(definition.paramInformation)) {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
paramInformation is not an object`);
            }
            for (const [key, paramDefinition] of objects_1.entries(definition.paramInformation)) {
                const match = /^(\d+)(?:\[0x([0-9a-fA-F]+)\])?$/.exec(key);
                if (!match) {
                    utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}: 
found invalid param number "${key}" in paramInformation`);
                }
                if (!typeguards_1.isObject(paramDefinition) &&
                    !(typeguards_1.isArray(paramDefinition) &&
                        paramDefinition.every((p) => typeguards_1.isObject(p)))) {
                    utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}: 
paramInformation "${key}" is invalid: Every entry must either be an object or an array of objects!`);
                }
                // Normalize to an array
                const defns = typeguards_1.isArray(paramDefinition)
                    ? paramDefinition
                    : [paramDefinition];
                if (!defns.every((d, index) => index === defns.length - 1 || "$if" in d)) {
                    utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}: 
paramInformation "${key}" is invalid: When there are multiple definitions, every definition except the last one MUST have an "$if" condition!`);
                }
                for (const def of defns) {
                    // Check if this entry applies for the actual config
                    if (deviceId &&
                        "$if" in def &&
                        !conditionApplies(def.$if, deviceId)) {
                        continue;
                    }
                    const keyNum = parseInt(match[1], 10);
                    const bitMask = match[2] != undefined
                        ? parseInt(match[2], 16)
                        : undefined;
                    paramInformation.set({ parameter: keyNum, valueBitMask: bitMask }, new ParamInformation(this, keyNum, bitMask, def, deviceId));
                    // Only apply the first matching one
                    break;
                }
            }
            this.paramInformation = paramInformation;
        }
        if (definition.proprietary != undefined) {
            if (!typeguards_1.isObject(definition.proprietary)) {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
proprietary is not an object`);
            }
            this.proprietary = definition.proprietary;
        }
        if (definition.compat != undefined) {
            if (!typeguards_1.isObject(definition.compat)) {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
compat is not an object`);
            }
            this.compat = new CompatConfig_1.CompatConfig(filename, definition.compat);
        }
        if (definition.metadata != undefined) {
            if (!typeguards_1.isObject(definition.metadata)) {
                utils_1.throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
metadata is not an object`);
            }
            this.metadata = new DeviceMetadata(filename, definition.metadata);
        }
    }
    static async from(filename, options = {}) {
        const { relativeTo, deviceId } = options;
        const relativePath = relativeTo
            ? path_1.default.relative(relativeTo, filename).replace(/\\/g, "/")
            : filename;
        const json = await JsonTemplate_1.readJsonWithTemplate(filename);
        return new DeviceConfig(relativePath, json, deviceId);
    }
}
exports.DeviceConfig = DeviceConfig;
class AssociationConfig {
    constructor(filename, groupId, definition) {
        this.groupId = groupId;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
Association ${groupId} has a non-string label`);
        }
        this.label = definition.label;
        if (definition.description != undefined &&
            typeof definition.description !== "string") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
Association ${groupId} has a non-string description`);
        }
        this.description = definition.description;
        if (typeof definition.maxNodes !== "number") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
maxNodes for association ${groupId} is not a number`);
        }
        this.maxNodes = definition.maxNodes;
        if (definition.isLifeline != undefined &&
            definition.isLifeline !== true) {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
isLifeline in association ${groupId} must be either true or left out`);
        }
        this.isLifeline = !!definition.isLifeline;
        if (definition.noEndpoint != undefined &&
            definition.noEndpoint !== true) {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
noEndpoint in association ${groupId} must be either true or left out`);
        }
        this.noEndpoint = !!definition.noEndpoint;
    }
}
exports.AssociationConfig = AssociationConfig;
class ParamInformation {
    constructor(parent, parameterNumber, valueBitMask, definition, deviceId) {
        this.parameterNumber = parameterNumber;
        this.valueBitMask = valueBitMask;
        if (typeof definition.label !== "string") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string label`);
        }
        this.label = definition.label;
        if (definition.description != undefined &&
            typeof definition.description !== "string") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string description`);
        }
        this.description = definition.description;
        if (typeof definition.valueSize !== "number" ||
            definition.valueSize <= 0) {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has an invalid value size`);
        }
        this.valueSize = definition.valueSize;
        if (typeof definition.minValue !== "number") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property minValue`);
        }
        this.minValue = definition.minValue;
        if (typeof definition.maxValue !== "number") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property maxValue`);
        }
        this.maxValue = definition.maxValue;
        if (definition.unsigned != undefined &&
            typeof definition.unsigned !== "boolean") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-boolean property unsigned`);
        }
        this.unsigned = definition.unsigned === true;
        if (definition.unit != undefined &&
            typeof definition.unit !== "string") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string unit`);
        }
        this.unit = definition.unit;
        if (typeof definition.readOnly !== "boolean") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: readOnly must be a boolean!`);
        }
        this.readOnly = definition.readOnly;
        if (typeof definition.writeOnly !== "boolean") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: writeOnly must be a boolean!`);
        }
        this.writeOnly = definition.writeOnly;
        if (definition.defaultValue == undefined) {
            if (!this.readOnly) {
                utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} is missing defaultValue, which is required unless the parameter is readOnly`);
            }
        }
        else if (typeof definition.defaultValue !== "number") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property defaultValue`);
        }
        this.defaultValue = definition.defaultValue;
        if (typeof definition.allowManualEntry !== "boolean") {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: allowManualEntry must be a boolean!`);
        }
        this.allowManualEntry = definition.allowManualEntry;
        if (typeguards_1.isArray(definition.options) &&
            !definition.options.every((opt) => typeguards_1.isObject(opt) &&
                typeof opt.label === "string" &&
                typeof opt.value === "number")) {
            utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: options is malformed!`);
        }
        const options = [];
        if (definition.options) {
            for (const opt of definition.options) {
                // Check if this entry applies for the actual config
                if (deviceId &&
                    "$if" in opt &&
                    !conditionApplies(opt.$if, deviceId)) {
                    continue;
                }
                options.push(shared_1.pick(opt, ["label", "value"]));
            }
        }
        this.options = options;
    }
}
exports.ParamInformation = ParamInformation;
class DeviceMetadata {
    constructor(filename, definition) {
        for (const prop of [
            "inclusion",
            "exclusion",
            "reset",
            "manual",
        ]) {
            if (prop in definition) {
                const value = definition[prop];
                if (typeof value !== "string") {
                    utils_1.throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
The metadata entry ${prop} must be a string!`);
                }
                this[prop] = value;
            }
        }
    }
}
exports.DeviceMetadata = DeviceMetadata;
//# sourceMappingURL=Devices.js.map