"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsonWithTemplate = void 0;
const core_1 = require("@zwave-js/core");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs = __importStar(require("fs-extra"));
const json5_1 = __importDefault(require("json5"));
const path = __importStar(require("path"));
const IMPORT_KEY = "$import";
const importSpecifierRegex = /^(?<filename>[\w\d\/\\\._-]+\.json)?(?:#(?<selector>[\w\d\/\._-]+))?$/i;
/** Parses a JSON file with $import keys and replaces them with the selected objects */
async function readJsonWithTemplate(filename) {
    if (!(await fs.pathExists(filename))) {
        throw new core_1.ZWaveError(`Could not open config file ${filename}: not found!`, core_1.ZWaveErrorCodes.Config_NotFound);
    }
    return readJsonWithTemplateInternal(filename, undefined, [], new Map());
}
exports.readJsonWithTemplate = readJsonWithTemplate;
function assertImportSpecifier(val, source) {
    if (typeof val !== "string") {
        throw new core_1.ZWaveError(`Invalid import specifier ${String(val)}!${source != undefined ? ` Source: ${source}` : ""}`, core_1.ZWaveErrorCodes.Config_Invalid);
    }
    if (!importSpecifierRegex.test(val)) {
        throw new core_1.ZWaveError(`Import specifier "${val}" is invalid!${source != undefined ? ` Source: ${source}` : ""}`, core_1.ZWaveErrorCodes.Config_Invalid);
    }
}
function getImportSpecifier(filename, selector) {
    let ret = filename;
    if (selector)
        ret += `#${selector}`;
    return ret;
}
function select(obj, selector) {
    let ret = obj;
    const selectorParts = selector.split("/").filter((s) => !!s);
    for (const part of selectorParts) {
        ret = ret[part];
    }
    if (!typeguards_1.isObject(ret)) {
        throw new core_1.ZWaveError(`The import target "${selector}" is not an object!`, core_1.ZWaveErrorCodes.Config_Invalid);
    }
    return ret;
}
async function readJsonWithTemplateInternal(filename, selector, visited, fileCache) {
    filename = path.normalize(filename);
    const specifier = getImportSpecifier(filename, selector);
    if (visited.includes(specifier)) {
        const msg = `Circular $import in config files: ${[
            ...visited,
            specifier,
        ].join(" -> ")}\n`;
        // process.stderr.write(msg + "\n");
        throw new core_1.ZWaveError(msg, core_1.ZWaveErrorCodes.Config_CircularImport);
    }
    let json;
    if (fileCache.has(filename)) {
        json = fileCache.get(filename);
    }
    else {
        try {
            const fileContent = await fs.readFile(filename, "utf8");
            json = json5_1.default.parse(fileContent);
            fileCache.set(filename, json);
        }
        catch (e) {
            let message = `Could not parse config file ${filename}: ${e.message}`;
            const source = [...visited, selector ? `#${selector}` : undefined]
                .reverse()
                .filter((s) => !!s);
            if (source.length > 0) {
                message += `\nImport stack: ${source
                    .map((s) => `\n  in ${s}`)
                    .join("")}`;
            }
            throw new core_1.ZWaveError(message, core_1.ZWaveErrorCodes.Config_Invalid);
        }
    }
    // Resolve the JSON imports for (a subset) of the file and return the compound file
    return resolveJsonImports(selector ? select(json, selector) : json, filename, [...visited, specifier], fileCache);
}
/** Replaces all `$import` properties in a JSON object with object spreads of the referenced file/property */
async function resolveJsonImports(json, filename, visited, fileCache) {
    const ret = {};
    // Loop through all properties and copy them to the resulting object
    for (const [prop, val] of Object.entries(json)) {
        if (prop === IMPORT_KEY) {
            // This is an import statement. Make sure we're working with a string
            assertImportSpecifier(val, visited.join(" -> "));
            const { filename: importFilename, selector, } = importSpecifierRegex.exec(val).groups;
            // const importFilename = path.join(path.dirname(filename), val);
            const imported = await readJsonWithTemplateInternal(
            // If a filename is specified, resolve it - else use the current file
            importFilename
                ? path.join(path.dirname(filename), importFilename)
                : filename, selector, visited, fileCache);
            Object.assign(ret, imported);
        }
        else if (typeguards_1.isObject(val)) {
            // We're looking at an object, recurse into it
            ret[prop] = await resolveJsonImports(val, filename, visited, fileCache);
        }
        else if (typeguards_1.isArray(val)) {
            // We're looking at an array, check if there are objects we need to recurse into
            const tasks = val.map((v) => typeguards_1.isObject(v)
                ? resolveJsonImports(v, filename, visited, fileCache)
                : v);
            ret[prop] = await Promise.all(tasks);
        }
        else {
            ret[prop] = val;
        }
    }
    return ret;
}
//# sourceMappingURL=JsonTemplate.js.map