"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.saveManufacturersInternal = exports.loadManufacturersInternal = void 0;
const core_1 = require("@zwave-js/core");
const shared_1 = require("@zwave-js/shared");
const objects_1 = require("alcalzone-shared/objects");
const typeguards_1 = require("alcalzone-shared/typeguards");
const fs_extra_1 = require("fs-extra");
const json5_1 = __importDefault(require("json5"));
const path_1 = __importDefault(require("path"));
const utils_1 = require("./utils");
const configPath = path_1.default.join(utils_1.configDir, "manufacturers.json");
/** @internal */
async function loadManufacturersInternal() {
    if (!(await fs_extra_1.pathExists(configPath))) {
        throw new core_1.ZWaveError("The manufacturer config file does not exist!", core_1.ZWaveErrorCodes.Config_Invalid);
    }
    try {
        const fileContents = await fs_extra_1.readFile(configPath, "utf8");
        const definition = json5_1.default.parse(fileContents);
        if (!typeguards_1.isObject(definition)) {
            utils_1.throwInvalidConfig("manufacturers", `the database is not an object!`);
        }
        const manufacturers = new Map();
        for (const [id, name] of objects_1.entries(definition)) {
            if (!utils_1.hexKeyRegex4Digits.test(id)) {
                utils_1.throwInvalidConfig("manufacturers", `found non-hex key ${id} at the root level`);
            }
            if (typeof name !== "string") {
                utils_1.throwInvalidConfig("manufacturers", `Key ${id} has a non-string manufacturer name`);
            }
            const idNum = parseInt(id.slice(2), 16);
            manufacturers.set(idNum, name);
        }
        return manufacturers;
    }
    catch (e) {
        if (e instanceof core_1.ZWaveError) {
            throw e;
        }
        else {
            utils_1.throwInvalidConfig("manufacturers");
        }
    }
}
exports.loadManufacturersInternal = loadManufacturersInternal;
/**
 * Write current manufacturers map to json
 */
async function saveManufacturersInternal(manufacturers) {
    const data = {};
    const orderedMap = new Map([...manufacturers].sort((a, b) => (a[0] > b[0] ? 1 : -1)));
    for (const [id, name] of orderedMap) {
        data[shared_1.formatId(id)] = name;
    }
    await fs_extra_1.writeFile(configPath, shared_1.stringify(data, "\t") + "\n");
}
exports.saveManufacturersInternal = saveManufacturersInternal;
//# sourceMappingURL=Manufacturers.js.map