"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.encodeBitMask = exports.parseBitMask = exports.encodeFloatWithScale = exports.getIntegerLimits = exports.getMinIntegerSize = exports.IntegerLimits = exports.parseFloatWithScale = exports.parseNumber = exports.parseMaybeNumber = exports.parseBoolean = exports.parseMaybeBoolean = exports.unknownBoolean = exports.unknownNumber = void 0;
const ZWaveError_1 = require("../error/ZWaveError");
const misc_1 = require("../util/misc");
exports.unknownNumber = "unknown";
exports.unknownBoolean = "unknown";
/** Parses a boolean that is encoded as a single byte and might also be "unknown" */
function parseMaybeBoolean(val, preserveUnknown = true) {
    return val === 0xfe
        ? preserveUnknown
            ? exports.unknownBoolean
            : undefined
        : parseBoolean(val);
}
exports.parseMaybeBoolean = parseMaybeBoolean;
/** Parses a boolean that is encoded as a single byte */
function parseBoolean(val) {
    return val === 0 ? false : val === 0xff ? true : undefined;
}
exports.parseBoolean = parseBoolean;
/** Parses a single-byte number from 0 to 100, which might also be "unknown" */
function parseMaybeNumber(val, preserveUnknown = true) {
    return val === 0xfe
        ? preserveUnknown
            ? exports.unknownNumber
            : undefined
        : parseNumber(val);
}
exports.parseMaybeNumber = parseMaybeNumber;
/** Parses a single-byte number from 0 to 100 */
function parseNumber(val) {
    return val <= 99 ? val : val === 0xff ? 99 : undefined;
}
exports.parseNumber = parseNumber;
/** Parses a floating point value with a scale from a buffer */
function parseFloatWithScale(payload) {
    misc_1.validatePayload(payload.length >= 1);
    const precision = (payload[0] & 224) >>> 5;
    const scale = (payload[0] & 24) >>> 3;
    const size = payload[0] & 0b111;
    misc_1.validatePayload(size >= 1, size <= 4, payload.length >= 1 + size);
    const value = payload.readIntBE(1, size) / Math.pow(10, precision);
    return { value, scale, bytesRead: 1 + size };
}
exports.parseFloatWithScale = parseFloatWithScale;
function getPrecision(num) {
    if (!Number.isFinite(num))
        return 0;
    let e = 1;
    let p = 0;
    while (Math.round(num * e) / e !== num) {
        e *= 10;
        p++;
    }
    return p;
}
/** The minimum and maximum values that can be stored in each numeric value type */
exports.IntegerLimits = Object.freeze({
    UInt8: Object.freeze({ min: 0, max: 0xff }),
    UInt16: Object.freeze({ min: 0, max: 0xffff }),
    UInt24: Object.freeze({ min: 0, max: 0xffffff }),
    UInt32: Object.freeze({ min: 0, max: 0xffffffff }),
    Int8: Object.freeze({ min: -0x80, max: 0x7f }),
    Int16: Object.freeze({ min: -0x8000, max: 0x7fff }),
    Int24: Object.freeze({ min: -0x800000, max: 0x7fffff }),
    Int32: Object.freeze({ min: -0x80000000, max: 0x7fffffff }),
});
function getMinIntegerSize(value, signed) {
    if (signed) {
        if (value >= exports.IntegerLimits.Int8.min && value <= exports.IntegerLimits.Int8.max)
            return 1;
        else if (value >= exports.IntegerLimits.Int16.min &&
            value <= exports.IntegerLimits.Int16.max)
            return 2;
        else if (value >= exports.IntegerLimits.Int32.min &&
            value <= exports.IntegerLimits.Int32.max)
            return 4;
    }
    else if (value >= 0) {
        if (value <= exports.IntegerLimits.UInt8.max)
            return 1;
        if (value <= exports.IntegerLimits.UInt16.max)
            return 2;
        if (value <= exports.IntegerLimits.UInt32.max)
            return 4;
    }
    // Not a valid size
}
exports.getMinIntegerSize = getMinIntegerSize;
function getIntegerLimits(size, signed) {
    return exports.IntegerLimits[`${signed ? "" : "U"}Int${size * 8}`];
}
exports.getIntegerLimits = getIntegerLimits;
/**
 * Encodes a floating point value with a scale into a buffer
 * @param override can be used to overwrite the automatic computation of precision and size with fixed values
 */
function encodeFloatWithScale(value, scale, override = {}) {
    var _a;
    const precision = (_a = override.precision) !== null && _a !== void 0 ? _a : Math.min(getPrecision(value), 7);
    value = Math.round(value * Math.pow(10, precision));
    let size = getMinIntegerSize(value, true);
    if (size == undefined) {
        throw new ZWaveError_1.ZWaveError(`Cannot encode the value ${value} because its too large or too small to fit into 4 bytes`, ZWaveError_1.ZWaveErrorCodes.Arithmetic);
    }
    else if (override.size != undefined && override.size > size) {
        size = override.size;
    }
    const ret = Buffer.allocUnsafe(1 + size);
    ret[0] =
        ((precision & 0b111) << 5) | ((scale & 0b11) << 3) | (size & 0b111);
    ret.writeIntBE(value, 1, size);
    return ret;
}
exports.encodeFloatWithScale = encodeFloatWithScale;
/** Parses a bit mask into a numeric array */
function parseBitMask(mask, startValue = 1) {
    const numBits = mask.length * 8;
    const ret = [];
    for (let index = 1; index <= numBits; index++) {
        const byteNum = (index - 1) >>> 3; // id / 8
        const bitNum = (index - 1) % 8;
        if ((mask[byteNum] & (2 ** bitNum)) !== 0)
            ret.push(index + startValue - 1);
    }
    return ret;
}
exports.parseBitMask = parseBitMask;
/** Serializes a numeric array with a given maximum into a bit mask */
function encodeBitMask(values, maxValue) {
    const numBytes = Math.ceil(maxValue / 8);
    const ret = Buffer.alloc(numBytes, 0);
    for (let val = 1; val <= maxValue; val++) {
        if (values.indexOf(val) === -1)
            continue;
        const byteNum = (val - 1) >>> 3; // id / 8
        const bitNum = (val - 1) % 8;
        ret[byteNum] |= 2 ** bitNum;
    }
    return ret;
}
exports.encodeBitMask = encodeBitMask;
//# sourceMappingURL=Primitive.js.map