/// <reference types="node" />
import type { ParamInfoMap } from "@zwave-js/config";
import type { MessageOrCCLogEntry, ValueID } from "@zwave-js/core";
import { CacheMetadata, CacheValue, Maybe, ValueMetadataAny, ZWaveError, ZWaveErrorCodes } from "@zwave-js/core";
import type { Driver } from "../driver/Driver";
import { PhysicalCCAPI, PollValueImplementation, POLL_VALUE, SetValueImplementation, SET_VALUE } from "./API";
import { CCCommandOptions, CommandClass, CommandClassDeserializationOptions, CommandClassOptions } from "./CommandClass";
export declare enum ConfigurationCommand {
    Set = 4,
    Get = 5,
    Report = 6,
    BulkSet = 7,
    BulkGet = 8,
    BulkReport = 9,
    NameGet = 10,
    NameReport = 11,
    InfoGet = 12,
    InfoReport = 13,
    PropertiesGet = 14,
    PropertiesReport = 15,
    DefaultReset = 1
}
export declare enum ValueFormat {
    SignedInteger = 0,
    UnsignedInteger = 1,
    Enumerated = 2,
    BitField = 3
}
export interface ConfigurationMetadata extends ValueMetadataAny {
    min?: ConfigValue;
    max?: ConfigValue;
    default?: ConfigValue;
    unit?: string;
    valueSize?: number;
    format?: ValueFormat;
    name?: string;
    info?: string;
    noBulkSupport?: boolean;
    isAdvanced?: boolean;
    requiresReInclusion?: boolean;
    states?: Record<number, string>;
    allowManualEntry?: boolean;
    isFromConfig?: boolean;
}
/**
 * @publicAPI
 */
export declare type ConfigValue = number | Set<number>;
export declare class ConfigurationCCError extends ZWaveError {
    readonly message: string;
    readonly code: ZWaveErrorCodes;
    readonly argument: number;
    constructor(message: string, code: ZWaveErrorCodes, argument: number);
}
export declare function getParamInformationValueID(parameter: number, bitMask?: number): ValueID;
export declare class ConfigurationCCAPI extends PhysicalCCAPI {
    supportsCommand(cmd: ConfigurationCommand): Maybe<boolean>;
    protected [SET_VALUE]: SetValueImplementation;
    protected [POLL_VALUE]: PollValueImplementation;
    /**
     * Requests the current value of a given config parameter from the device.
     * This may timeout and return `undefined` if the node does not respond.
     * If the node replied with a different parameter number, a `ConfigurationCCError`
     * is thrown with the `argument` property set to the reported parameter number.
     */
    get(parameter: number, options?: {
        valueBitMask?: number;
        allowUnexpectedResponse?: boolean;
    }): Promise<ConfigValue | undefined>;
    /**
     * Sets a new value for a given config parameter of the device.
     */
    set(parameter: number, value: ConfigValue, valueSize: 1 | 2 | 4): Promise<void>;
    /**
     * Resets a configuration parameter to its default value.
     *
     * WARNING: This will throw on legacy devices (ConfigurationCC v3 and below)
     */
    reset(parameter: number): Promise<void>;
    /** Resets all configuration parameters to their default value */
    resetAll(): Promise<void>;
    getProperties(parameter: number): Promise<Pick<ConfigurationCCPropertiesReport, "valueSize" | "valueFormat" | "minValue" | "maxValue" | "defaultValue" | "nextParameter" | "altersCapabilities" | "isReadonly" | "isAdvanced" | "noBulkSupport"> | undefined>;
    /** Requests the name of a configuration parameter from the node */
    getName(parameter: number): Promise<string | undefined>;
    /** Requests usage info for a configuration parameter from the node */
    getInfo(parameter: number): Promise<string | undefined>;
    /**
     * This scans the node for the existing parameters. Found parameters will be reported
     * through the `value added` and `value updated` events.
     *
     * WARNING: This method throws for newer devices.
     *
     * WARNING: On nodes implementing V1 and V2, this process may take
     * **up to an hour**, depending on the configured timeout.
     *
     * WARNING: On nodes implementing V2, all parameters after 255 will be ignored.
     */
    scanParametersLegacy(): Promise<void>;
}
export declare class ConfigurationCC extends CommandClass {
    ccCommand: ConfigurationCommand;
    constructor(driver: Driver, options: CommandClassOptions);
    interview(complete?: boolean): Promise<void>;
    /**
     * Whether this node's param information was loaded from a config file.
     * If this is true, we don't trust what the node reports
     */
    protected get isParamInformationFromConfig(): boolean;
    /**
     * Returns stored config parameter metadata for all partial config params addressed with the given parameter number
     */
    getPartialParamInfos(parameter: number): (ValueID & {
        metadata: ConfigurationMetadata;
    })[];
    /**
     * Returns stored config parameter metadata for all partial config params addressed with the given parameter number
     */
    composePartialParamValue(parameter: number, valueBitMask: number, maskedValue: number): number;
    serializeValuesForCache(): CacheValue[];
    serializeMetadataForCache(): CacheMetadata[];
    /** Deserializes the config parameter info from a config file */
    deserializeParamInformationFromConfig(config: ParamInfoMap): void;
    translatePropertyKey(property: string | number, propertyKey?: string | number): string | undefined;
    translateProperty(property: string | number, propertyKey?: string | number): string;
}
export declare class ConfigurationCCReport extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _parameter;
    get parameter(): number;
    private _value;
    get value(): ConfigValue;
    private _valueSize;
    get valueSize(): number;
    toLogEntry(): MessageOrCCLogEntry;
}
interface ConfigurationCCGetOptions extends CCCommandOptions {
    parameter: number;
    /**
     * If this is `true`, responses with different parameters than expected are accepted
     * and treated as hints for the first parameter number.
     */
    allowUnexpectedResponse?: boolean;
}
export declare class ConfigurationCCGet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCGetOptions);
    parameter: number;
    allowUnexpectedResponse: boolean;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
declare type ConfigurationCCSetOptions = CCCommandOptions & ({
    parameter: number;
    resetToDefault: true;
} | {
    parameter: number;
    resetToDefault?: false;
    valueSize: number;
    value: ConfigValue;
});
export declare class ConfigurationCCSet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCSetOptions);
    resetToDefault: boolean;
    parameter: number;
    valueSize: number | undefined;
    value: ConfigValue | undefined;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
declare type ConfigurationCCBulkSetOptions = CCCommandOptions & {
    parameters: number[];
    handshake?: boolean;
} & ({
    resetToDefault: true;
} | {
    resetToDefault?: false;
    valueSize: number;
    values: number[];
});
export declare class ConfigurationCCBulkSet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCBulkSetOptions);
    private _parameters;
    get parameters(): number[];
    private _resetToDefault;
    get resetToDefault(): boolean;
    private _valueSize;
    get valueSize(): number;
    private _values;
    get values(): number[];
    private _handshake;
    get handshake(): boolean;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCBulkReport extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _reportsToFollow;
    get reportsToFollow(): number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    private _defaultValues;
    get defaultValues(): boolean;
    private _isHandshakeResponse;
    get isHandshakeResponse(): boolean;
    private _valueSize;
    get valueSize(): number;
    private _values;
    get values(): ReadonlyMap<number, ConfigValue>;
    toLogEntry(): MessageOrCCLogEntry;
}
interface ConfigurationCCBulkGetOptions extends CCCommandOptions {
    parameters: number[];
}
export declare class ConfigurationCCBulkGet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCBulkGetOptions);
    private _parameters;
    get parameters(): number[];
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCNameReport extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _parameter;
    get parameter(): number;
    private _name;
    get name(): string;
    private _reportsToFollow;
    get reportsToFollow(): number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    mergePartialCCs(partials: ConfigurationCCNameReport[]): void;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCNameGet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCGetOptions);
    parameter: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCInfoReport extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _parameter;
    get parameter(): number;
    private _info;
    get info(): string;
    private _reportsToFollow;
    get reportsToFollow(): number;
    getPartialCCSessionId(): Record<string, any> | undefined;
    expectMoreMessages(): boolean;
    mergePartialCCs(partials: ConfigurationCCInfoReport[]): void;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCInfoGet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCGetOptions);
    parameter: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCPropertiesReport extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _parameter;
    get parameter(): number;
    private _valueSize;
    get valueSize(): number;
    private _valueFormat;
    get valueFormat(): ValueFormat;
    private _minValue;
    get minValue(): ConfigValue | undefined;
    private _maxValue;
    get maxValue(): ConfigValue | undefined;
    private _defaultValue;
    get defaultValue(): ConfigValue | undefined;
    private _nextParameter;
    get nextParameter(): number;
    private _altersCapabilities;
    get altersCapabilities(): boolean | undefined;
    private _isReadonly;
    get isReadonly(): boolean | undefined;
    private _isAdvanced;
    get isAdvanced(): boolean | undefined;
    private _noBulkSupport;
    get noBulkSupport(): boolean | undefined;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCPropertiesGet extends ConfigurationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | ConfigurationCCGetOptions);
    parameter: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ConfigurationCCDefaultReset extends ConfigurationCC {
}
export {};
