/// <reference types="node" />
import { CommandClasses, Duration, Maybe, MessageOrCCLogEntry, ValueID, ValueMetadata } from "@zwave-js/core";
import { JSONObject } from "@zwave-js/shared";
import type { Driver } from "../driver/Driver";
import { PhysicalCCAPI } from "./API";
import { CCCommandOptions, CommandClass, CommandClassDeserializationOptions, CommandClassOptions } from "./CommandClass";
export declare enum NotificationCommand {
    EventSupportedGet = 1,
    EventSupportedReport = 2,
    Get = 4,
    Report = 5,
    Set = 6,
    SupportedGet = 7,
    SupportedReport = 8
}
/**
 * @publicAPI
 */
export declare type NotificationMetadata = ValueMetadata & {
    ccSpecific: {
        notificationType: number;
    };
};
/** Returns the ValueID used to store the supported notification types of a node */
export declare function getSupportedNotificationTypesValueId(): ValueID;
/** Returns the ValueID used to store the supported notification events of a node */
export declare function getSupportedNotificationEventsValueId(type: number): ValueID;
/** Returns the ValueID used to store whether a node supports push or pull */
export declare function getNotificationModeValueId(): ValueID;
export declare function getAlarmTypeValueId(endpoint?: number): ValueID;
export declare function getAlarmLevelValueId(endpoint?: number): ValueID;
export declare class NotificationCCAPI extends PhysicalCCAPI {
    supportsCommand(cmd: NotificationCommand): Maybe<boolean>;
    sendReport(options: NotificationCCReportOptions): Promise<void>;
    get(options: NotificationCCGetSpecificOptions): Promise<Pick<NotificationCCReport, "sequenceNumber" | "alarmLevel" | "notificationStatus" | "notificationEvent" | "zensorNetSourceNodeId" | "eventParameters"> | undefined>;
    set(notificationType: number, notificationStatus: boolean): Promise<void>;
    getSupported(): Promise<Pick<NotificationCCSupportedReport, "supportedNotificationTypes" | "supportsV1Alarm"> | undefined>;
    getSupportedEvents(notificationType: number): Promise<readonly number[] | undefined>;
}
export declare class NotificationCC extends CommandClass {
    ccCommand: NotificationCommand;
    constructor(driver: Driver, options: CommandClassOptions);
    determineRequiredCCInterviews(): readonly CommandClasses[];
    private determineNotificationMode;
    interview(complete?: boolean): Promise<void>;
    /** Whether the node implements push or pull notifications */
    get notificationMode(): "push" | "pull" | undefined;
}
interface NotificationCCSetOptions extends CCCommandOptions {
    notificationType: number;
    notificationStatus: boolean;
}
export declare class NotificationCCSet extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | NotificationCCSetOptions);
    notificationType: number;
    notificationStatus: boolean;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare type NotificationCCReportOptions = {
    alarmType: number;
    alarmLevel: number;
} | {
    notificationType: number;
    notificationEvent: number;
    eventParameters?: Buffer;
    sequenceNumber?: number;
};
export declare class NotificationCCReport extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | (NotificationCCReportOptions & CCCommandOptions));
    persistValues(): boolean;
    alarmType: number | undefined;
    alarmLevel: number | undefined;
    notificationType: number | undefined;
    notificationStatus: boolean | number | undefined;
    notificationEvent: number | undefined;
    readonly zensorNetSourceNodeId: number | undefined;
    eventParameters: Buffer | Duration | Record<string, number> | undefined;
    sequenceNumber: number | undefined;
    toLogEntry(): MessageOrCCLogEntry;
    toJSON(): JSONObject;
    private parseEventParameters;
    serialize(): Buffer;
}
declare type NotificationCCGetSpecificOptions = {
    alarmType: number;
} | {
    notificationType: number;
    notificationEvent?: number;
};
declare type NotificationCCGetOptions = CCCommandOptions & NotificationCCGetSpecificOptions;
export declare class NotificationCCGet extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | NotificationCCGetOptions);
    /** Proprietary V1/V2 alarm type */
    alarmType: number | undefined;
    /** Regulated V3+ notification type */
    notificationType: number | undefined;
    notificationEvent: number | undefined;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class NotificationCCSupportedReport extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    private _supportsV1Alarm;
    get supportsV1Alarm(): boolean;
    private _supportedNotificationTypes;
    get supportedNotificationTypes(): readonly number[];
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class NotificationCCSupportedGet extends NotificationCC {
}
export declare class NotificationCCEventSupportedReport extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions);
    persistValues(): boolean;
    private _notificationType;
    get notificationType(): number;
    private _supportedEvents;
    get supportedEvents(): readonly number[];
    toLogEntry(): MessageOrCCLogEntry;
}
interface NotificationCCEventSupportedGetOptions extends CCCommandOptions {
    notificationType: number;
}
export declare class NotificationCCEventSupportedGet extends NotificationCC {
    constructor(driver: Driver, options: CommandClassDeserializationOptions | NotificationCCEventSupportedGetOptions);
    notificationType: number;
    serialize(): Buffer;
    toLogEntry(): MessageOrCCLogEntry;
}
export {};
